/**
 *  '$RCSfile: DatasetPanel.java,v $'
 *  '$Author: ruland $'
 *  '$Date: 2006/01/30 15:27:22 $'
 *  '$Revision: 1.42 $'
 *
 *  For Details:
 *  http://kepler.ecoinformatics.org Copyright (c) 2004 The Regents of the
 *  University of California. All rights reserved. Permission is hereby granted,
 *  without written agreement and without license or royalty fees, to use, copy,
 *  modify, and distribute this software and its documentation for any purpose,
 *  provided that the above copyright notice and the following two paragraphs
 *  appear in all copies of this software. IN NO EVENT SHALL THE UNIVERSITY OF
 *  CALIFORNIA BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL,
 *  OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 *  DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE
 *  POSSIBILITY OF SUCH DAMAGE. THE UNIVERSITY OF CALIFORNIA SPECIFICALLY
 *  DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 *  SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.seek.ecogrid.quicksearch;


import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;

import org.ecoinformatics.seek.ecogrid.EcoGridQueryServicesController;
import org.ecoinformatics.seek.ecogrid.EcoGridServicesController;
import org.kepler.gui.SearchUIJPanel;

import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.vergil.tree.EntityTreeModel;
import ptolemy.vergil.tree.VisibleTreeModel;


/**
 *  This class will display a panel for search and search result from ecogrid
 *  service.
 *
 *@author     berkley
 *@created    February 17, 2005
 */
public class DatasetPanel extends JPanel {

  //declare controls
  //search part panel (above)
  private SearchUIJPanel searchPartPanel;
  private QuickSearchAction quickSearchAction;
  private QuickSearchCancelAction cancelQuickSearchAction;
  private JProgressBar _progressBar;
  private JLabel _progressLabel;

  //result part panel
  private CompositeEntity resultsRoot;
  private ResultPanel resultPanel;
  private EcoGridQueryServicesController searchController;

  private final String QUICKSEARCHLABEL = "Search";
  private final Dimension DIM = new Dimension(200, 500);

  /**
   *  Set up the pane with preference size and configuration for search
   *
   *@param  controller     Description of the Parameter
   *@exception  Exception  Description of the Exception
   */
  public DatasetPanel(EcoGridQueryServicesController controller)
                                                              throws Exception {
    searchController = controller;
    init();
  }


  /**
   *  Get the search text field value
   *
   *@return    The searchTextFieldValue value
   */
  public String getSearchTextFieldValue() {
    return searchPartPanel.getSearchTerm();
  }


  /**
   *  Gets the resultRoot attribute of the DatasetPanel object
   *
   *@return    The resultRoot value
   */
  public CompositeEntity getResultRoot() {
    return resultsRoot;
  }


  /**
   *  Initialize the panel
   *
   *@throws  Exception
   */
  public void init() throws Exception {
    this.setLayout(new BorderLayout());
    this.setMinimumSize(DIM);
    this.setPreferredSize(DIM);
    //init searchPartPanel
    initSearchPartPanel();

    //add searchPartPanel into this panel top(North)
    this.add(searchPartPanel, BorderLayout.NORTH);
    //add empty result scroll panel into this panel bottotm(South)
    resultPanel = new ResultPanel(createResultModel());
    this.add(resultPanel, BorderLayout.CENTER);

    _progressBar = new JProgressBar();
    _progressBar.setVisible(false);

    _progressLabel = new JLabel();
    _progressLabel.setVisible(true);

    JPanel progressPanel = new JPanel(new BorderLayout());
    progressPanel.add(_progressLabel, BorderLayout.NORTH);
    progressPanel.add(_progressBar, BorderLayout.SOUTH);
    this.add(progressPanel, BorderLayout.SOUTH);
  }


  private void initSearchPartPanel() throws Exception {

    quickSearchAction
      = new QuickSearchAction(searchController,
                              SearchUIJPanel.SEARCH_BUTTON_CAPTION, this);

    Action resetAction = new AbstractAction() {
        public void actionPerformed(ActionEvent e) {
            resetResultsPanel();
            searchPartPanel.setSearchTerm("");
        }};

    cancelQuickSearchAction
      = new QuickSearchCancelAction(quickSearchAction,
                                    SearchUIJPanel.CANCEL_BUTTON_CAPTION);

    DisplayCurrentServicesListAction sourceDialogAction
      = new DisplayCurrentServicesListAction(
                  SearchUIJPanel.SOURCE_BUTTON_CAPTION, searchController, this);

    searchPartPanel
      = new SearchUIJPanel(QUICKSEARCHLABEL, quickSearchAction, resetAction,
                           cancelQuickSearchAction, sourceDialogAction, null);
  }


  /**
   *  Create the tree model used to contain the result set. This is displayed in
   *  the JTree on the left of the window.
   *
   *@return    the EntityTreeModel that can contain the results
   */
  private EntityTreeModel createResultModel() {
    try {
      resultsRoot = new ResultTreeRoot("resultset");
      Attribute libraryMarker = new Attribute(resultsRoot, "_libraryMarker");
    } catch (IllegalActionException iae) {
      System.err.println("Could not create entity.");
    } catch (NameDuplicationException nde) {
      System.err.println("An entity with that name already exists.");
    }
    EntityTreeModel resultTreeModel = new VisibleTreeModel(resultsRoot);
    return resultTreeModel;
  }


  /**
   *  Method to get EcoGridServicesController
   *
   *@return    EcoGridServicesController
   */
  public EcoGridServicesController getSearchScope() {
    return this.searchController;
  }


  /**
   *  Set searchController for this panel
   *
   *@param  newTreeData   Description of the Parameter
   */
  /*
   *  public void setEcoGridServiceController(EcoGridServicesController controller)
   *  {
   *  this.searchController = controller;
   *  if (quickSearchAction != null)
   *  {
   *  Vector servicesVector = controller.getServicesList();
   *  quickSearchAction.setSearchSerivcesVector(servicesVector);
   *  }
   *  }
   */


  /**
   *  Update the result panel after searching
   *
   *@param  newTreeData  CompositeEntity, the search result
   */
  public void update(EntityTreeModel newTreeData) {
    resultPanel.setTreeModel(newTreeData);
  }


  /**
   *  The main program for the DatasetPanel class
   *
   *@param  args           The command line arguments
   *@exception  Exception  Description of the Exception
   */
  public static void main(String[] args) throws Exception {
    int width = 200;
    int height = 500;
    DatasetPanel datasetPanel = new DatasetPanel(EcoGridQueryServicesController.getInstance());
    //datasetPanel.setSearchScope(searchScopeVector);
    //set up a frame
    JFrame frame = new JFrame("SwingApplication");
    frame.setSize(width, height);
    frame.getContentPane().add(datasetPanel, BorderLayout.CENTER);
    //Finish setting up the frame, and show it.
    frame.addWindowListener(
      new WindowAdapter() {
      public void windowClosing(WindowEvent e) {
        System.exit(0);
      }});
    frame.setVisible(true);
  }


  /**
   *  This method will go though dataset panel's parent and ends a Window.
   *
   *@return    Window the parent window
   */
  public Container getParentFrame() {

    Container container = null;
    container = this.getParent();
    while (! (container instanceof Window)) {
      container = container.getParent();
    }
    return container;
  }


  /**
   *  Sets the progressLabel attribute of the DatasetPanel object
   *
   *@param  aMsg  The new progressLabel value
   */
  public void setProgressLabel(String aMsg) {
    _progressLabel.setText(aMsg);
  }


  /**
   * Method to starting search progress bar
   */
  public void startSearchProgressBar() {

    searchPartPanel.setSearchEnabled(false);
    _progressBar.setIndeterminate(true);
    _progressBar.setVisible(true);
    _progressLabel.setVisible(false);
    _progressLabel.setText("");
  }


  /**
   * Reset the search panel - enabling/disabling buttons as
   * appropriate and hiding progress bar
   */
  public void resetSearchPanel() {

    searchPartPanel.setSearchEnabled(true);
    _progressBar.setIndeterminate(false);
    _progressBar.setVisible(false);
    _progressLabel.setVisible(true);
  }


  /**
   * Reset the results panel to ist blank state
   */
  public void resetResultsPanel() {

    this.getResultRoot().removeAllEntities();
    this.update(new VisibleTreeModel(this.getResultRoot()));
  }
}
