/* An actor that creates an HTML representation from ADN schemas.

Copyright (c) 1998-2003 The Regents of the University of California.
All rights reserved.
Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.

                                       PT_COPYRIGHT_VERSION_2
                                       COPYRIGHTENDKEY

*/

package org.geon;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Iterator;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;

import org.dlese.adn.ADNmetadataType;
import org.dlese.adn.BoundBoxType;
import org.dlese.adn.CatalogType;
import org.dlese.adn.ContributorLifecycleType;
import org.dlese.adn.GeospatialCoverageType;
import org.dlese.adn.PeriodType;
import org.dlese.adn.PeriodsType;
import org.dlese.adn.PersonType;
import org.dlese.adn.TemporalCoveragesType;
import org.dlese.adn.TimeADType;
import org.dlese.adn.TimeAndPeriodType;
import org.dlese.adn.TimeBCType;
import org.dlese.adn.TimeRelativeType;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.StringToken;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

//////////////////////////////////////////////////////////////////////////
//// ADNToHTML
/**
This actor converts ADN schema to its HTML representation

@author Efrat Jaeger
@version $Id: ADNToHTML.java,v 1.4 2006/03/05 01:45:21 jaeger Exp $
@since Ptolemy II 3.0.2
*/
public class ADNToHTML extends TypedAtomicActor {

    /** Construct an actor with the given container and name.
     *  @param container The container.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the actor cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
    public ADNToHTML(CompositeEntity container, String name)
            throws NameDuplicationException, IllegalActionException  {
        super(container, name);

        id = new TypedIOPort(this, "id", true, false);
        id.setTypeEquals(BaseType.STRING);

        output = new TypedIOPort(this, "output", false, true);
        output.setTypeEquals(BaseType.STRING);

        _attachText("_iconDescription", "<svg>\n" +
                "<polygon points=\"-15,-15 15,15 15,-15 -15,15\" "
                + "style=\"fill:white\"/>\n" +
                "</svg>\n");
    }

    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////

    /** XML name-value input string. */
    public TypedIOPort id;

    /** ADN schema string. */
    public TypedIOPort output;

    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    /** Consume an XML string representing name value pairs and
     *  converts it to an ADN schema.
     *
     *  @exception IllegalActionException If there is no director.
     *  FIXME: Either verify that it does check for the director,
     *  or remove this statement.  This statement occurs in other
     *  conversion actor(s) as well.
     */
    public void fire() throws IllegalActionException {
        String inputId = ((StringToken) id.get(0)).stringValue();

        // Note:  Following line may assume 1 byte per character, not sure.
        String outputValue = process(inputId);
        output.send(0, new StringToken(outputValue));
    }


    private String process(String inputId) {

    	String html = "";
    	
    	html += "<html>\n  <head>\n";
		html += "<link href='stylesheet_small.css' rel='stylesheet' title='master' type='text/css'>\n";
		html += "<body bgcolor=white>\n";
    	
        try {
		    JAXBContext jc = JAXBContext.newInstance( "org.dlese.adn" ); 
		    Unmarshaller u = jc.createUnmarshaller();
	
		    // load the metadata
		    ADNmetadataType item = (ADNmetadataType)u.unmarshal(
					  new FileInputStream("C:/adn/" + inputId.substring(5)+".adn") ); 
	
		    html += "<blockquote>\n";
		    html += "<br><p>\n";
		    html += "<b><font size=+1>"+item.getGeneral().getTitle()+"</font></b>\n";
		    html += "<br><img src='images/spacer.gif' width=1 height=15 border=0><font color=gray>id: "+inputId+"</font>\n";
		    html += "<br><br><p>\n";
	
		    // review the general section
		    html += label("General") + "\n";
		    html += "<blockquote>\n";
		    html += "<table border=0 cellpadding=5>\n";
	            
		    html += message("Title", item.getGeneral().getTitle()) + "\n";
	
		    CatalogType catalog = (CatalogType)(item.getMetaMetadata().getCatalogEntries().getCatalog().iterator().next());
		    html += message("Format", catalog.getValue()) + "\n";
		    html += message("Status", "on") + "\n";
	
		    String subjects = "";
		    int c = 0;
		    for (Iterator i=item.getGeneral().getSubjects().getSubject().iterator(); i.hasNext(); ) {
				if (c > 0) subjects += ", "; 
			try{
				String tmp = (String)i.next();
				if (tmp.startsWith("DLESE:")) {
				    tmp = tmp.substring(6);
				}
				subjects += tmp;
			} catch (Exception ex) {
				ex.printStackTrace();
			}
		                c++;
			}

		    html += message("Subjects", subjects) +"\n";
	
		    String keywords = "No keywords were provided";
		    c = 0;

		    for (Iterator i=item.getGeneral().getKeywords().getKeyword().iterator(); i.hasNext(); ) {
				if (c > 0) {
				    keywords += ", ";
				} else {
				    keywords = "";
				}
				String tmp = (String)i.next();
				if (tmp.startsWith("DLESE:")) {
				    tmp = tmp.substring(6);
				}
				keywords += tmp;
		                c++;
			}

		    html += message("Keywords", keywords) + "\n";
	
		    html += message("Permission", item.getRights().getDescription()) + "\n";
		    html += message("Language", item.getGeneral().getLanguage()) + "\n";
		    html += message("Description", item.getGeneral().getDescription()) + "\n";
	
		    html += "</table>\n";
		    html += "</blockquote>\n";
	
		    // review the contributors
		    html += label("Contributors") + "\n";
		    html += "<blockquote>\n";
		    html += "<table border=0 cellpadding=5>\n";
			
		    for (Iterator i=item.getGeneral().getSubjects().getSubject().iterator(); i.hasNext(); ) {

		    	ContributorLifecycleType contributor = (ContributorLifecycleType)i.next();
				html += message("Role", contributor.getRole()) + "\n";
				PersonType person = contributor.getPerson();
				
				if (person != null) {
			
				    String name = person.getNameTitle() + ". " + person.getNameFirst();
				    String tmp = person.getNameMiddle();
				    if (tmp != null) {
				    	name += " "+tmp;
				    }
				    name += " "+person.getNameLast();
		
				    html += message("Name", name) +"\n";
				    html += message("Organization", person.getInstName()) + "\n";
				    html += message("Email", person.getEmailPrimary()) + "\n";
				}
				
		    }
	
		    html += "</table>\n";
		    html += "</blockquote>\n";
	
		    // review the spatial coverage
		    html += label("Spatial Coverage") + "\n";
		    html += "<blockquote>\n";
		    html += "<table border=0 cellpadding=5>\n";
	
		    java.text.DecimalFormat form = new java.text.DecimalFormat("###.##");
	
		    for (Iterator it=item.getGeneral().getSubjects().getSubject().iterator(); it.hasNext(); ) {
		    	GeospatialCoverageType spatial = (GeospatialCoverageType)it.next();
		
				String tmp = spatial.getGeodeticDatumGlobalOrHorz();
				if (tmp.startsWith("DLESE:")) {
				    tmp = tmp.substring(6);
				}
				html += message("Hrizontal Geodatic Datum", tmp) + "\n";
				html += message("Projection", spatial.getProjection().getValue()) + "\n";
		
				BoundBoxType box = spatial.getBoundBox();
		
				double north = box.getNorthCoord().doubleValue();
				double south = box.getSouthCoord().doubleValue();	
				double east = box.getEastCoord().doubleValue();
				double west = box.getWestCoord().doubleValue();
		
				html += message("Bounding Box", "") + "\n";
				html += messagePadding("West",  ""+west) + "\n";
				html += messagePadding("North", ""+north) + "\n";
				html += messagePadding("East",  ""+east) + "\n";
				html += messagePadding("South", ""+south) + "\n";
		    }

		    html += "</table>\n";
		    html += "</blockquote>\n";
	
		    // review the temporal coverage
		    html += label("Temporal Coverage") + "\n";
		    html += "<blockquote>\n";
		    html += "<table border=0 cellpadding=5>\n";
	
		    TemporalCoveragesType temporalCoverages = item.getTemporalCoverages();
		    if (temporalCoverages != null) {
	
				for (Iterator it=temporalCoverages.getTimeAndPeriod().iterator(); it.hasNext(); ) {
				    
				    // handle timeInfo 
				    TimeAndPeriodType timeAndPeriod = (TimeAndPeriodType)it.next();
		
				    TimeADType ad = timeAndPeriod.getTimeInfo().getTimeAD();
				    if (ad != null) {
						Calendar begin = (Calendar)ad.getBegin().getDate();
						Calendar end = (Calendar)ad.getEnd().getDate();
			
						html += message("Period in AD", "") + "\n";
						html += messagePadding("From", 
									   begin.get(Calendar.MONTH) + "/" + 
									   begin.get(Calendar.DAY_OF_MONTH)  + "/" + 
									   begin.get(Calendar.YEAR)) + "\n";
						html += messagePadding("To", 
									   end.get(Calendar.MONTH) + "/" + 
									   end.get(Calendar.DAY_OF_MONTH)  + "/" + 
									   end.get(Calendar.YEAR)) + "\n";
			
				    }

				    TimeBCType bc = timeAndPeriod.getTimeInfo().getTimeBC();
				    if (bc != null) {
						int begin = Integer.parseInt(bc.getBegin());
						int end = Integer.parseInt(bc.getEnd());
						html += message("Period in BC", "") + "\n";
						html += message("Period from BC to AD",  "from " + begin + " to " + end) + "\n";
				    }
		
				    TimeRelativeType relative = timeAndPeriod.getTimeInfo().getTimeRelative();
				    if (relative != null) {
				    	BigDecimal begin = relative.getBegin().getValue();
			                BigDecimal end = relative.getEnd().getValue();
			            if (begin.intValue() != 0 && end.intValue() != 0) {
			            	html += message("Period",  "from "+ begin + " to "+ end) + "\n";
			            }
				    }

				    // handle period
				    PeriodsType periods = timeAndPeriod.getPeriods();
				    if (periods != null) {
						for (Iterator j=periods.getPeriod().iterator(); j.hasNext(); ) {
						    PeriodType period = (PeriodType)j.next();
						    html += message("Geologic Age", period.getName()) + "\n";
						    html += message("Source", period.getSource()) + "\n";
			
						}
				    }

				}

		    } else {
		    	html += message("Temporal Coverage", "unknown") + "\n";
		    }
	    
		    html += "</table>\n";
		    html += "</blockquote>\n";
	
	
		    html += "</blockquote>\n";
	
	
	
			html += "</body>\n";
			html += "</html>\n";
			
	        return html;

        } catch (JAXBException jex) {
        	System.out.println(jex.toString());
        } catch (FileNotFoundException fnex) {
        }
        return html;

    }
   
 
    private String label(String string) {
	return 
	    "<table width=90% cellpadding=1 cellspacing=0 border=0>\n"+
	    "<tr><td bgcolor=Gainsboro>\n"+
	    "<font face=\"arial,sans-serif\" size=-1 color=#777777>\n"+
	    "&nbsp; <b>"+string+"</b>\n"+
	    "</font>\n"+
	    "</td></tr>\n"+
	    "</table>\n";
    }


    private String message(String key, String val) {
	return
	    "    <tr>\n" +
	    "        <td align=right width=150><div class=label><b>"+key+":</b></div></td>\n" +
	    "        <td align=left>"+val+"</td>\n" +
	    "    </tr>\n";
    }


    private String messagePadding(String key, String val) {
	return
	    "    <tr>\n" +
	    "        <td align=right width=150><div class=label>&nbsp;&nbsp;&nbsp;"+key+":</div></td>\n" +
	    "        <td align=left>"+val+"</td>\n" +
	    "    </tr>\n";
    }



}
