/* A database query actor.

 Copyright (c) 1998-2003 The Regents of the University of California.
 All rights reserved.
 Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.

 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.

                                        PT_COPYRIGHT_VERSION_2
                                        COPYRIGHTENDKEY
*/

package org.geon;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.Statement;
import java.sql.Types;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import org.ecoinformatics.seek.querybuilder.DBQueryDef;
import org.ecoinformatics.seek.querybuilder.DBQueryDefParserEmitter;
import org.ecoinformatics.seek.querybuilder.DBSchemaParserEmitter;
import org.kepler.objectmanager.data.db.DSSchemaIFace;
import org.kepler.objectmanager.data.db.QBTableauFactory;

import ptolemy.actor.IOPort;
import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.actor.gui.style.TextStyle;
import ptolemy.actor.parameters.PortParameter;
import ptolemy.data.ArrayToken;
import ptolemy.data.BooleanToken;
import ptolemy.data.DBConnectionToken;
import ptolemy.data.ObjectToken;
import ptolemy.data.RecordToken;
import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.data.expr.StringParameter;
import ptolemy.data.type.ArrayType;
import ptolemy.data.type.BaseType;
import ptolemy.data.type.RecordType;
import ptolemy.data.type.Type;
import ptolemy.gui.GraphicalMessageHandler;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.Port;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.InternalErrorException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.NamedObj;
import ptolemy.kernel.util.Settable;
import ptolemy.kernel.util.StringAttribute;

//////////////////////////////////////////////////////////////////////////
//// DatabaseQuery
/**
This actor performs database queries against a specific database.
It accepts a string quey and a database connection reference as inputs.
The actor produces the output of the query in the user's selected output format, 
specified by the outputType parameter, either as an XML, Record, string or
a in a relational form with no metadata. The user can also specify whether to 
broadcast each row at a time or the whole result at once.

@author Efrat Jaeger
@version $Id: DatabaseQuery.java,v 1.25 2006/03/05 01:47:39 jaeger Exp $
@since Ptolemy II 3.0.2
*/
public class DatabaseQuery extends TypedAtomicActor {

    /** Construct an actor with the given container and name.
     *  @param container The container.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the actor cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
    public DatabaseQuery(CompositeEntity container, String name)
            throws NameDuplicationException, IllegalActionException  {
        super(container, name);

        // Parameters
        outputType = new StringParameter(this, "outputType");
        outputType.setExpression("XML");
        _outputType = _XML;
        outputType.addChoice("XML");
        outputType.addChoice("record");
        outputType.addChoice("array");
        outputType.addChoice("string");
        outputType.addChoice("no metadata");
        outputType.addChoice("result set"); // send result set as is.

        // Ports
        dbcon = new TypedIOPort(this, "dbcon", true, false);
        dbcon.setTypeEquals(BaseType.DBCONNECTION);
        query = new PortParameter(this, "query");
        query.setStringMode(true);
        result = new TypedIOPort(this, "result", false, true);
//        result.setTypeEquals(BaseType.GENERAL);

        _schemaAttr = new StringAttribute(this, "schemaDef");
        TextStyle schemaDefTS = new TextStyle(_schemaAttr, "schemaDef");

        _sqlAttr = new StringAttribute(this, "sqlDef");
        _sqlAttr.setVisibility(Settable.NONE);
        TextStyle sqlDefTS = new TextStyle(_sqlAttr, "sqlDef");

        outputEachRowSeparately = new Parameter(this, "outputEachRowSeparately", new BooleanToken(false));
        outputEachRowSeparately.setTypeEquals(BaseType.BOOLEAN);
        attributeChanged(outputEachRowSeparately);

        // create tableau for editting the SQL String
        _qbTableauFactory = new QBTableauFactory(this, "_tableauFactory");

        _attachText("_iconDescription", "<svg>\n"
                    + "<ellipse cx=\"0\" cy=\"-30\" "
                    + "rx=\"20\" ry=\"10\"/>\n"
                    + "<line x1=\"20\" y1=\"0\" "
                    + "x2=\"20\" y2=\"-30\"/>\n"
                    + "<line x1=\"-20\" y1=\"0\" "
                    + "x2=\"-20\" y2=\"-30\"/>\n"
                    + "<line x1=\"-20\" y1=\"0\" "
                    + "x2=\"20\" y2=\"0\"/>\n"
                    + "</svg>\n");
    }

    ///////////////////////////////////////////////////////////////////
    ////                     ports and parameters                  ////

    /** The output format: XML, Record or String
     * or a relational string with no metadata information.
     */
    public StringParameter outputType;

    /** Specify whether to display the complete result at once or each row
     *  separately.
     */
    public Parameter outputEachRowSeparately;

    /** A reference to the database connection.
     */
    public TypedIOPort dbcon;

    /** An input query string.
     */
    public PortParameter query;

    /** The query result.
     */
    public TypedIOPort result;
    
    /** Hidden variable containing the xml representation of the query as 
     * returned by the query builder.
     */
    public StringAttribute _sqlAttr = null;

    /** The schema of the database.
     */
    public StringAttribute _schemaAttr = null;


    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    /** Determine the output format
     *  @param attribute The attribute that changed.
     *  @exception IllegalActionException If the output type is not recognized.
     */
    public void attributeChanged(Attribute attribute)
            throws  IllegalActionException {
        try {
            if (attribute == outputType) {
                String strOutputType = outputType.getExpression();
                if (strOutputType.equals("XML")) {
                    _outputType = _XML;
                } else if (strOutputType.equals("record")) {
                    _outputType = _RECORD;
                } else if (strOutputType.equals("array")) {
                    _outputType = _ARR;
                } else if (strOutputType.equals("string")) {
                    _outputType = _STR;
                } else if (strOutputType.startsWith("no")) {
                    _outputType = _NOMD;
                } else if (strOutputType.startsWith("result")) {
                    _outputType = _RS;
                } else {
                    throw new IllegalActionException(this,
                            "Unrecognized math function: " + strOutputType);
                }
            } else if (attribute == outputEachRowSeparately) {
                _separate = ((BooleanToken)outputEachRowSeparately.getToken()).booleanValue();
            } else if (attribute == _sqlAttr) {
            	if (_sqlAttr != null && !_sqlAttr.equals("")) {
            		String sqlXMLStr = ((Settable) _sqlAttr).getExpression();
            		DBQueryDef queryDef = DBQueryDefParserEmitter.parseQueryDef(
                        _schemaDef, sqlXMLStr);
                    String sqlStr = DBQueryDefParserEmitter.createSQL(_schemaDef, queryDef);
                    if (sqlStr != null) {
                    	query.setToken(new StringToken(sqlStr));
                    }
            	}
            } else if (attribute == _schemaAttr) {
            	String schemaDef = ((Settable) _schemaAttr).getExpression();
            	if (schemaDef.length() > 0) {
            		_schemaDef = DBSchemaParserEmitter.parseSchemaDef(schemaDef);
            	}
            } else {
                super.attributeChanged(attribute);
            }
        } catch (Exception nameDuplication) {
 /*           throw new InternalErrorException(this, nameDuplication,
                    "Unexpected name duplication");*/
        }
    }

    /**
     * Try to set the database schema once the database connection 
     * port has been connected.
     */
    public void connectionsChanged(Port port) {
        super.connectionsChanged(port);
        if (port == dbcon) {
        	List conPortsList = dbcon.connectedPortList();
        	Iterator conPorts = conPortsList.iterator();
        	while (conPorts.hasNext()) {
        		IOPort p = (IOPort) conPorts.next();
        		if (p.isOutput() && p.getName().equals("dbcon")) {
        			NamedObj container = p.getContainer();
        			if (container instanceof OpenDBConnection) {
        				String schema = "";
        				try {
        					schema = ((OpenDBConnection)container).sendSchemaToConnected();
        				} catch (Exception ex) {
        					schema = "";
        				}
        				if (!schema.equals("")) {
        					try {
        						_schemaAttr.setExpression(schema);
        					} catch (IllegalActionException iaex) {
        						// unable to set schema attribute..
        					}
        				}
        			}
        		}
        	}
        }
    }

    /** Get the database connection.
     * 
     * @throws IllegalActionException
     */
    public Connection getConnection() throws IllegalActionException {
        DBConnectionToken _dbcon = (DBConnectionToken) dbcon.get(0);
        Connection _con = null;
        try {
          _con = _dbcon.getValue();
          return _con;
        }
        catch (Exception e) {
          throw new IllegalActionException(this, e,
                                           "CONNECTION FAILURE");
        }
    }
    
    /** Consume a query and a database connection reference. Compute
     *  the query result according to the specified output format.
     *  @exception IllegalActionException If there is no director.
     */
    public void fire() throws IllegalActionException {
    	
    	if (_con == null) {
    		_con = getConnection();
    	}
    	
    	if (_con != null) { 
	        query.update();
	        _query = ((StringToken)query.getToken()).stringValue();
	        if (!_query.equals(_prevQuery) || query.getPort().getWidth() > 0) { // if this is a different query.
	        	_prevQuery = _query;
		        try{
		          Statement st = _con.createStatement();
		          ResultSet rs;
		          try{
		            rs = st.executeQuery(_query);
		          } catch (Exception e1) {
		            throw new IllegalActionException(this, e1,
		                               "SQL executeQuery exception");
		          }
		
		          switch(_outputType) {
		          case _XML:
		              _createXML(rs);
		              break;
		          case _RECORD:
		          	_createRecord(rs);
		              break;
		          case _ARR:
		            _createArr(rs);
		            break;
		          case _STR:
		            _createString(rs);
		            break;
		          case _NOMD:
		            _createNoMetadata(rs);
		            break;
		        case _RS:
		          _sendResultSet(rs);
		          break;
		          default:
		              throw new InternalErrorException(
		                      "Invalid value for _outputType private variable. "
		                      + "DatabaseQuery actor (" + getFullName()
		                      + ")"
		                      + " on output type " + _outputType);
		          }
		
		        } catch (Exception ex) {
		            throw new IllegalActionException(this, ex,
		                                           "exception in SQL");
		        }
	        } else {
	        	// if the query comes only from the parameter and hasn't changed don't refire.
	        	if (query.getPort().getWidth() == 0) {
	        		_refire = false;
	        	}
	        }
        } else throw new IllegalActionException(this, "Database connection is not available.");
    }

    /** Takes care of halting the execution in case the query is not 
     *  updated from a port and hasn't changed.
     */
    public boolean postfire() throws IllegalActionException {
    	if (!_refire)
    		return false;
    	
    	return super.postfire(); 	
    }
    
    /** Read the outputType parameter and set output type accordingly.
     *  @exception IllegalActionException If the file or URL cannot be
     *   opened, or if the first line cannot be read.
     */
    public void preinitialize() throws IllegalActionException {
        super.preinitialize();

        _prevQuery = "";
        _con = null;
        _refire = true;
        
        // Set the output type.
        switch(_outputType) {
        case _XML:
            result.setTypeEquals(BaseType.STRING);
            break;
        case _RECORD:
            result.setTypeEquals(BaseType.GENERAL);
            break;
        case _ARR:
            result.setTypeEquals(new ArrayType(BaseType.STRING));
            break;
        case _STR:
            result.setTypeEquals(BaseType.STRING);
            break;
        case _NOMD:
            result.setTypeEquals(BaseType.STRING);
            break;
        case _RS:
            result.setTypeEquals(BaseType.GENERAL);
            break;
        default:
            throw new InternalErrorException(
                    "Invalid value for _outputType private variable. "
                    + "DatabaseQuery actor (" + getFullName()
                    + ")"
                    + " on output type " + _outputType);
        }
    }
    
    public void wrapup() {
    	_prevQuery = "";
    	_con = null;
    	_refire = true;
    }

    ///////////////////////////////////////////////////////////////////
    ////                         private methods                   ////

    /** Send result set as is (separate is not applicable in this case).
     */

    private void _sendResultSet(ResultSet rs) throws IllegalActionException {
        result.broadcast(new ObjectToken(rs));
    }

    /** Create a string result.
     */
    private void _createString(ResultSet rs) throws IllegalActionException {
        try {
          ResultSetMetaData md = rs.getMetaData();
          String res = "";
          while (rs.next()) {
            for (int i = 1; i <= md.getColumnCount(); i++) {
              res += md.getColumnName(i) + ": ";
              String val = rs.getString(i);
              if (val == null)
                res += "";
              else
                res += val;
              res += " ;  ";
            }
            if (_separate) {
              result.broadcast(new StringToken(res));
              res = "";
            }
            else {
              res += "\n";
            }
          }
          if (!_separate) {
            result.broadcast(new StringToken(res));
          }
          rs.close();
        } catch (Exception ex) {
            throw new IllegalActionException(this, ex, "exception in create String result");
        }
      }

      /** Create result as an array of string. (due to problems with record array)
       */
      private void _createArr(ResultSet rs) throws IllegalActionException {
          try {
              Vector results = new Vector();
              Token resultTokens[] = null;
              ResultSetMetaData md = rs.getMetaData();
              while (rs.next()) {
                  String res = "";
                  for (int i = 1; i <= md.getColumnCount(); i++) {
                     String val = rs.getString(i);
                     if (val == null)
                         res += ",";
                     else
                         res += val + ",";
                 }
                 // remove last comma.
                 int lstCmaInd = res.lastIndexOf(",");
                 if (lstCmaInd > -1) {
                     res = res.substring(0,lstCmaInd);
                 }
                 if (_separate) {
                     resultTokens = new Token[1];
                     resultTokens[0] = new StringToken(res);
                     result.broadcast(new ArrayToken(resultTokens));
                 }
                 else {
                     results.add(new StringToken(res));
                 }
             }
             if (!_separate) {
                 if (results.size() == 0) {
                   GraphicalMessageHandler.message(
                       "No matching result for query:\n" +
                       "\"" + _query + "\".");
                 } else {
                   resultTokens = new Token[results.size()];
                   results.toArray(resultTokens);
                   result.broadcast(new ArrayToken(resultTokens));
                 }
             }
             rs.close();
         }
         catch (Exception ex) {
             throw new IllegalActionException(this, ex,
                 "exception in create String result");
         }
     }

    /** Create an XML stream result.
     */
    private void _createXML(ResultSet rs) throws IllegalActionException {
        try {
            String tab = "    ";
            String finalResult = "<?xml version=\"1.0\"?> \n";
            finalResult += "<result> \n";
            ResultSetMetaData md = rs.getMetaData();

            int colNum = md.getColumnCount();
            String tag[] = new String[colNum]; // holds all the result tags.
            for (int i = 0; i < colNum; i++) {
                tag[i] = md.getColumnName(i+1);
                tag[i] = tag[i].replace(' ','_');
                if (tag[i].startsWith("#")) {
                    tag[i] = tag[i].substring(1);
                }

                //when joining two or more tables that have the same columns we'd like to distinguish between them.
                int count = 1;
                int j;
                while (true) { //if the same tag appears more then once add an incremental index to it.
                    for (j=0; j<i; j++) {
                        if (tag[i].equals(tag[j])) { //the new tag already exist
                            if (count == 1) { // first duplicate
                                tag[i] = tag[i] + count;
                            }
                            else {
                                int tmp = count-1;
                                String strCnt = "" + tmp;
                                int index = tag[i].lastIndexOf(strCnt);
                                tag[i] = tag[i].substring(0,index);  //remove the prev index.
                                tag[i] = tag[i] + count;
                            }
                            count++;
                            break;
                        }
                    }
                    if (j==i) {//the tag was not found in existing tags.
                        count = 1;
                        break;
                    }
                }
            }

            while (rs.next()) {
                String res = tab + "<row> \n";

                for (int i = 0; i < colNum; i++) {
                    String val = rs.getString(i+1);
                    res += tab + tab;
                    if (val == null) {
                      res += "<" + tag[i] + "/>\n";
                    }
                    else {
                      res += "<" + tag[i] + ">" + val + "</" + tag[i] + ">\n";
                    }
                }
                res += tab + "</row> \n";

                if (_separate) {
                    finalResult += res + "</result>";
                    result.broadcast(new StringToken(finalResult));
                    finalResult = "<?xml version=\"1.0\"?> \n";
                    finalResult += "<result>\n";
                }
                else {
                  finalResult += res;
                }
          }
          if (!_separate) {
            finalResult += "</result>";
            result.broadcast(new StringToken(finalResult));
          }
          rs.close();
        } catch (Exception ex) {
            throw new IllegalActionException(this, ex, "exception in create XML stream");
        }
    }

    /** Create a record result.
     */
    private void _createRecord(ResultSet rs) throws IllegalActionException {
        try {
          ResultSetMetaData md = rs.getMetaData();
          String res = "";
          int colNum = md.getColumnCount();
          String labels[] = new String[colNum];
          for (int i = 1; i <= colNum; i++) {
          	labels[i-1] = md.getColumnName(i);
          }
          Token values[] = new Token[colNum];
          result.setTypeEquals(new RecordType(new String[colNum],new Type[colNum]));
          while (rs.next()) {
            for (int i = 1; i <= colNum; i++) {
              String val = rs.getString(i);
              if (val == null)
                res += "";
              else
                res += val;
              values[i-1] = new StringToken(val);
            }
            result.broadcast(new RecordToken(labels,values));
          }
          rs.close();
        } catch (Exception ex) {
            throw new IllegalActionException(this, ex, "exception in create String result");
        }
      }

    /** Create a tabular form result string with no metadata information.
     */
    private void _createNoMetadata(ResultSet rs)  throws IllegalActionException {
    	try {
	        ResultSetMetaData md = rs.getMetaData();
	        int colNum = md.getColumnCount();
	        String res = "";
	    	while (rs.next()) {
	    		String currRow = "";
	    		for (int i = 1; i <= colNum; i++) {
	        		String currVal = rs.getString(i);
	        		if (currVal == null || currVal.equals("")) {
	        			int type = md.getColumnType(i);
	        			if (type == Types.CHAR || type == Types.VARCHAR) {
	        				currVal = "-";
	        			} else currVal = "-1";
	        		}
	        		currVal = currVal.replace(' ','_');
	        		currRow += currVal;

	        		// for display purposes.
	        		int colWidth = md.getColumnDisplaySize(i);
	        		int numSpace = colWidth - currVal.length();
	        		for (int j=0; j<numSpace; j++) {
	        			currRow += " ";
	        		}
	        	}
	        	if (_separate) {
	        		result.broadcast(new StringToken(currRow));
	        	}
	        	else {
	        		res += currRow + "\n";
	        	}
	        }
	    	if (!_separate) {
	    		//remove the last carriage return.
	    		int lastCRInd = res.lastIndexOf("\n");
	    		if (lastCRInd > -1) {
	    			res = res.substring(0,lastCRInd);
	    		}
	    		result.broadcast(new StringToken(res));
	    	}
	        rs.close();
    	} catch (Exception ex) {
    		throw new IllegalActionException(this, ex, "exception in create custom result");
    	}
    }

    ///////////////////////////////////////////////////////////////////
    ////                         private variables                 ////


    /** Database connection object.
     */
    private Connection _con;

    /** Output indicator parameter.
     */
    private int _outputType;

    /** Output indicator parameter.
     */
    private boolean _separate;

    /** Query string.
     */
    private String _query;
    
    /** Previously queried query..
     */
    private String _prevQuery = "";
    
    /** Refire flag.
     */
    private boolean _refire = true;

    /** Query builder tableau factory.
     */
    protected QBTableauFactory _qbTableauFactory = null;

    /** Schema definition interface, used by the query builder
     */
    protected DSSchemaIFace _schemaDef = null;

    // Constants used for more efficient execution.
    private static final int _XML = 0;
    private static final int _RECORD = 1;
    private static final int _STR = 2;
    private static final int _NOMD = 3;
    private static final int _ARR = 4;
    private static final int _RS = 5;
}

