/**
 *  '$RCSfile: SimpleLibrarySearcher.java,v $' '$Author: brooke $' '$Date:
 *  2004/10/22 17:25:43 $' '$Revision: 1.3 $' For Details:
 *  http://kepler.ecoinformatics.org '$RCSfile: SimpleLibrarySearcher.java,v $'
 *  For Details: http://kepler.ecoinformatics.org Copyright (c) 2004 The Regents
 *  of the University of California. All rights reserved. Permission is hereby
 *  granted, without written agreement and without license or royalty fees, to
 *  use, copy, modify, and distribute this software and its documentation for
 *  any purpose, provided that the above copyright notice and the following two
 *  paragraphs appear in all copies of this software. IN NO EVENT SHALL THE
 *  UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
 *  SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 *  SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS
 *  BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. THE UNIVERSITY OF CALIFORNIA
 *  SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 *  OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.gui;


import java.util.Iterator;
import java.util.Stack;

import javax.swing.JTree;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.NamedObj;
import ptolemy.moml.EntityLibrary;
import ptolemy.moml.ErrorHandler;


/**
 *  class to search the library. This class uses a simple name comparator to
 *  determine results. It uses a depth first traversal of the tree.
 *
 *@author     berkley
 *@created    February 17, 2005
 */
public class SimpleLibrarySearcher extends LibrarySearcher {
  private Stack pathStack;
  //the stack to create the return paths with


  /**
   *  constructor
   *
   *@param  library     Description of the Parameter
   *@param  searchPane  Description of the Parameter
   */
  public SimpleLibrarySearcher(JTree library, LibrarySearchPane searchPane) {
    super(library, searchPane);
  }


  /**
   *  search for val in the library
   *
   *@param  val  Description of the Parameter
   *@return      Description of the Return Value
   */
  public LibrarySearchResults search(String value) {
    results = new LibrarySearchResults();
    if (value.trim().equals("")) {
      return results;
    }

    // this might need optimizing at some point

    TreeModel model = library.getModel();
    //calling this method implicitly adds results
    //to the global "results" variable
    findval(value, model);

    LibrarySearcher searcher = new OntLibrarySearcher(library, searchPane);
    Iterator iter = searcher.search(value).iterator();
    while (iter.hasNext()) {
      TreePath p = (TreePath)iter.next();
      if (!results.contains(p)) {
        results.add(p);
      }
    }
    return results;
  }

  /**
   *  pre-reads the library so searches will be faster. When the momlparser hits
   *  an error in the moml, it rebuilds the model which in turn collapses the
   *  entire library JTree. This presearches the tree which causes the parse
   *  errors to get thrown before the first search so the tree won't be
   *  collapsed the first time you search.
   */
  protected void init() {
    pathStack = new Stack();
    //search("this is something that will not be found");
  }


  /**
   *  look for the val in the TreeModel
   *
   *@param  val    Description of the Parameter
   *@param  model  Description of the Parameter
   */
  private void findval(String val, TreeModel model) {
    Object o = model.getRoot();
    //start from the root
    pathStack = new Stack();
    pathStack.push(o);
    for (int i = 0; i < model.getChildCount(o); i++) {
      Object child = model.getChild(o, i);
      pathStack.push(child);
      String name = ( (NamedObj)child).getName();
      String className = ( (NamedObj)child).getClassName();
      compareVals(name, className, val);
      //go to the other findval method
      if (child instanceof EntityLibrary) {
        findval(val, (EntityLibrary)child);
      }
    }
  }


  /**
   *  look for val recursively inside entity
   *
   *@param  val     Description of the Parameter
   *@param  entity  Description of the Parameter
   */
  private void findval(String val, EntityLibrary entity) {
    for (Iterator i = entity.containedObjectsIterator(); i.hasNext(); ) {
      NamedObj e = (NamedObj)i.next();
      String name = e.getName();
      String className = e.getClassName();
      pathStack.push(e);
      //push our new candidate onto the stack

      // need to find all children and add to path
      if (e instanceof EntityLibrary) {
        //recurse into the tree
        compareVals(name, className, val);
        findval(val, (EntityLibrary)e);
        //keep searching
      } else {
        //leaf node
        compareVals(name, className, val);

        if (!pathStack.empty()) {
          pathStack.pop();
        }
        //remove the leaf node we just looked at
      }
    }

    if (!pathStack.empty()) {
      pathStack.pop();
    }
    //remove the composite node we are currently searching
  }


  /**
   *  compares two strings and adds the path to the results if they are a match
   *
   *@param  val1a  Description of the Parameter
   *@param  val1b  Description of the Parameter
   *@param  val2   Description of the Parameter
   */
  private void compareVals(String val1a, String val1b, String val2) {
    if (val1a.startsWith("_") && !val2.startsWith("_")) {
      return;
    }

    //make the comparison case insensitive
    val1a = val1a.toLowerCase();
    val1b = val1b.toLowerCase();
    val2 = val2.toLowerCase();

    if (val1a.indexOf(val2) != -1 || val1b.indexOf(val2) != -1) {
      addStackTopToResult();
    }
  }


  /**
   *  adds the path at the top of the stack to the result
   */
  private void addStackTopToResult() {
    results.add(createPathFromStack());
  }


  /**
   *  return the path as a TreePath
   *
   *@return    Description of the Return Value
   */
  private TreePath createPathFromStack() {
    return new TreePath(pathStack.toArray());
  }


  /**
   *  errorhandler that ignore all errors and continues parsing.
   *
   *@author     berkley
   *@created    February 17, 2005
   */
  private class ErrorSkipper implements ErrorHandler {
    /**
     *  Description of the Method
     *
     *@param  b  Description of the Parameter
     */
    public void enableErrorSkipping(boolean b) {
      //yeah, we're skipping everything whether you tell us to or not
    }


    /**
     *  Description of the Method
     *
     *@param  element    Description of the Parameter
     *@param  context    Description of the Parameter
     *@param  exception  Description of the Parameter
     *@return            Description of the Return Value
     */
    public int handleError(java.lang.String element, NamedObj context,
                           java.lang.Throwable exception) {
      return ErrorHandler.CONTINUE;
      //just keep it moving
    }
  }


  /**
   *  A factory that creates the searcher to search the library
   *
   *@author     berkley
   *@created    February 17, 2005
   */
  public static class Factory extends LibrarySearcherFactory {
    /**
     *  Create an factory with the given name and container.
     *
     *@param  container                     The container.
     *@param  name                          The name of the entity.
     *@exception  IllegalActionException    If the container is incompatible
     *      with this attribute.
     *@exception  NameDuplicationException  If the name coincides with an
     *      attribute already in the container.
     */
    public Factory(NamedObj container, String name) throws
      IllegalActionException, NameDuplicationException {
      super(container, name);
    }


    /**
     *  creates a ScrollBarModifier and returns it.
     *
     *@param  library        Description of the Parameter
     *@param  searchPane     Description of the Parameter
     *@return                A new LibraryPane that displays the library
     */
    public LibrarySearcher createLibrarySearcher(JTree library,
                                                 LibrarySearchPane searchPane) {
      return new SimpleLibrarySearcher(library, searchPane);
    }
  }
}
