/**
 *  '$RCSfile: KSWLibraryBuilder.java,v $'
 *  '$Author: ruland $'
 *  '$Date: 2006/02/10 03:27:41 $'
 *  '$Revision: 1.39 $'
 *
 *  For Details:
 *  http://www.kepler-project.org
 *
 *  Copyright (c) 2004 The Regents of the
 *  University of California. All rights reserved. Permission is hereby granted,
 *  without written agreement and without license or royalty fees, to use, copy,
 *  modify, and distribute this software and its documentation for any purpose,
 *  provided that the above copyright notice and the following two paragraphs
 *  appear in all copies of this software. IN NO EVENT SHALL THE UNIVERSITY OF
 *  CALIFORNIA BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL,
 *  OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 *  DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE
 *  POSSIBILITY OF SUCH DAMAGE. THE UNIVERSITY OF CALIFORNIA SPECIFICALLY
 *  DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 *  SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.moml;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringReader;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import org.ecoinformatics.util.Config;
import org.kepler.ksw.KSWFile;
import org.kepler.ksw.KSWManifest;
import org.kepler.objectmanager.cache.CacheManager;
import org.kepler.objectmanager.cache.CacheUtil;
import org.kepler.objectmanager.cache.KARCacheObject;
import org.kepler.objectmanager.library.LibraryIndex;
import org.kepler.objectmanager.lsid.KeplerLSID;

import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.StringAttribute;
import ptolemy.kernel.util.NamedObj;
import ptolemy.kernel.util.Workspace;
import ptolemy.moml.LibraryBuilder;
import EDU.oswego.cs.dl.util.concurrent.PooledExecutor;

/**
 * This class builds a ComponentEntity library from a directory of KSW files.
 */
public class KSWLibraryBuilder extends LibraryBuilder
{
  private CacheManager cache;
  private LibraryIndex libIndex;
  private File libraryDir;
  public static File actorLibSaveFile = new File(Config.getUserDirPath() + "actorLibrary");
  public static File annotationsSaveFile = new File(Config.getUserDirPath() + "annotations");
  
  /**
   * constructor.
   */
  public KSWLibraryBuilder()
  {
    super();
  }

  /**
   * build the library.  This should be built in the form of a ComponentEntity
   * See the ptolemy code if you want an example of what the ComponentEntity
   * should look like
   * @return ComponentEntity
   * @throws Exception
   */
  public CompositeEntity buildLibrary(Workspace workspace) throws Exception
  {
    try
    {
      cache = CacheManager.getInstance();
      
      PooledExecutor pool = new PooledExecutor(10);
      
      String kswDir = null;
      //get the attributes and find the _libraryDir att and get its value
      boolean directoryAttFound = false;
      List attList = getAttributes();
      Iterator itt = attList.iterator();
      Vector libraryDirs = new Vector();
      while(itt.hasNext())
      {
        String KEPLERDir = System.getProperty("KEPLER");
        StringAttribute a = (StringAttribute)itt.next();
        if(a.getName().indexOf("_libraryDir") != -1)
        {
          kswDir = KEPLERDir + File.separator + a.getExpression();
          libraryDir = new File(kswDir);
          if(!libraryDir.exists() || !libraryDir.isDirectory())
          { //make sure the directory actually exists
            System.out.println("WARNING: you specified the directory " + 
              libraryDir.getAbsolutePath() + " as a KAR repository directory " + 
              "but it either does not exist or is not a directory.  No " +
              "KAR files have been loaded from this directory.");
            continue;
          }
          libraryDirs.add(libraryDir);
          directoryAttFound = true;
        }
      }

      if(!directoryAttFound)
      {
        throw new Exception("You must provide a property called _libraryDir " +
          "that provides the relative path to the KAR directory from $KEPLER.");
      }

      System.out.println("KAR Library directories: " + libraryDirs.toString());
      //long starttime = System.currentTimeMillis();
      //now we can do some ksw processing and build the actorLibrary ComponentEntity
      for(int k=0; k<libraryDirs.size(); k++)
      {
        File kswDirFile = ((File)libraryDirs.elementAt(k));
        kswDir = kswDirFile.getAbsolutePath();
        String[] kswDirList = kswDirFile.list();
        for(int i=0; i<kswDirList.length; i++)
        {
          String filename = kswDir + File.separator + kswDirList[i];
          if(filename.indexOf(".kar") == -1)
          {
            continue;
          }
          KSWLibraryBuilderTask task = new KSWLibraryBuilderTask( filename );
          // The pooled execution of the tasks has been disabled for the Alpha 8
          // release pending examination of wierd issues during startup.
          // With pooled executions, the ActorMetadata.getActorClass() method
          // sometimes gets exceptions when processing the gamesXXX actors.
          //pool.execute( task );
          task.run();
        }
      }

      pool.shutdownAfterProcessingCurrentlyQueuedTasks();
      
      //starttime = System.currentTimeMillis() - starttime;
      //System.out.println("Time to process: " + starttime );
      libIndex = LibraryIndex.getInstance();
      Iterator dirListIterator = workspace.directoryList().iterator();
      CompositeEntity container = null;
      while(dirListIterator.hasNext())
      {
        NamedObj obj = (NamedObj)dirListIterator.next();
        if(obj.getName().equals("configuration"))
        {
          container = (CompositeEntity)obj;
        }
      }
      
      CompositeEntity actorLib = libIndex.getLibrary(workspace);
      libIndex.setActorLibrary(actorLib, workspace);
      
      writeActorLibrary(actorLib);
      return actorLib;
    }
    catch(Exception e)
    {
      e.printStackTrace();
      throw e;
    }
  }
  
  /**
   * writes the specified actorLibrary to the save file
   */
  public static void writeActorLibrary(CompositeEntity actorLibrary)
    throws IOException
  {
    FileWriter fw = new FileWriter(actorLibSaveFile);
    actorLibrary.exportMoML( fw );
    fw.close();
  }
}

/**
 * 
 * KSWLibraryBuilderTask is a runnable object which processes a single kar file.
 * This task can be used in a thread pool in order to speed initial processing of
 * the kar files.
 * 
 * @author Kevin Ruland
 *
 */
class KSWLibraryBuilderTask implements Runnable {
	
	/**
	 * The filename of the kar file this taks will process.
	 */
	private String karFilename;
	
	/** Constructs a runnable task to process a karfile.
	 * 
	 * @param karFilename The name of the kar file to process.
	 */
	
	public KSWLibraryBuilderTask( String karFilename ) {
		this.karFilename = karFilename;
	}
	
	public void run() {
		
		try {
			//System.out.println("Processing karfile = " + karFilename);
			CacheManager cache = CacheManager.getInstance();
			
	          File kswFile = new File(karFilename);
	          KSWFile kFile = new KSWFile(kswFile);
	          KSWManifest manifest = kFile.getKSWManifest();
	          String lsid = manifest.getMainAttribute("lsid");
	          /*
	           * There is a race condition below which will eventually
	           * need to be addressed.  The test and insertion need to be
	           * atomic.  One solution is to just try the insert and accomodate
	           * failures due to duplication.  But this ends up making startup
	           * with a populated cache very noisy.
	           */
	          if(!cache.isContained(new KeplerLSID(lsid)))
	          {
	            KARCacheObject kco = new KARCacheObject(kswFile);
	            cache.insertObject(kco);  
	          }
		}
		catch ( Exception e ) {
			System.out.println("Unable to process kar file: " + karFilename );
			//e.printStackTrace();
		}

	}
	
}
