/**
 *    '$RCSfile: KARCacheObject.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2005/12/31 04:03:00 $'
 *   '$Revision: 1.9 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.objectmanager.cache;

import org.kepler.objectmanager.lsid.KeplerLSID;
import org.kepler.ksw.KSWFile;
import org.kepler.objectmanager.lsid.DuplicateLSIDException;

import com.ibm.lsid.LSIDException;

import org.ecoinformatics.util.Config;

import java.io.*;
import java.util.*;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;


/**
 * This class represents a KAR file in the cache.  when the kar file
 * is read into this object, it is exploded and its contents are cached.
 * an array of the lsids of the objects inside the kar file can be returned
 * then retrieved from the cache.
 */
public class KARCacheObject extends CacheObject implements Serializable
{
  private File karFile;
  Hashtable containedLSIDs = new Hashtable();
  
  /**
   * null constructor for serialization
   */
  public KARCacheObject()
  {
    super();
  }
  
  /**
   * construct a new CacheObject
   */
  public KARCacheObject(String name, KeplerLSID lsid)
  {
    super(name, lsid);
  }
  
  /**
   * construct a karCacheObject from a kar file
   */
  public KARCacheObject(File karFile)
    throws CacheException
  {
    super();
    this.karFile = karFile;
    Vector actorMetadataVector = new Vector();
    Vector tempActorMetadataVector = new Vector();
    try
    {
      //get the name and lsid from the kar file here
      String karUnpackDirName = Config.getUserDirPath() + "kar";
      File karUnpackDir = new File(karUnpackDirName);
      if(!karUnpackDir.exists())
      {
        karUnpackDir.mkdirs();
      }
      
      //unpack the kar
      KSWFile kar = new KSWFile(karFile);
      //set the lsid and the name
      //the kar lsid is in the main attributes and is called "lsid"
      this.lsid = new KeplerLSID(
        kar.getManifest().getMainAttributes().getValue("lsid"));
      this.name = karFile.getName();
      Enumeration karFileEnum = kar.entries();
      while(karFileEnum.hasMoreElements())
      {
        JarEntry entry = (JarEntry)karFileEnum.nextElement();
        Attributes atts = entry.getAttributes();
        if(atts == null)
        { //if the entry doesn't have any attributes, skip it.
          continue;
        }

        String lsid = atts.getValue("lsid");
        String type = atts.getValue("type");
        containedLSIDs.put(lsid, type);

        try
        {
          tempActorMetadataVector = handleKSWItem(type, new KeplerLSID(lsid),
            entry, karUnpackDirName, kar);
        }
        catch(DuplicateLSIDException dlside)
        { //throw an exception because the lsids are conflicting.
          throw dlside;
        }
      }
      
      //keep track of each item extracted and it's id for easy removal later
    }
    catch(Exception e)
    {
      e.printStackTrace();
      throw new CacheException("Error creating KARCacheObject: " + e.getMessage());
    }
  }
  
  /**
   * return the java object associated with this CacheObject
   */
  public Object getObject()
  {
    return null;
  }
  
  /**
   * Return an array of lsids of a certain type from within this kar.  
   * if the type is null return all of the lsids
   * @param getType the type of the content to get
   */
  public KeplerLSID[] getContentOfType(String getType)
    throws LSIDException
  {
    Enumeration keys = containedLSIDs.keys();
    Vector returnVec = new Vector();
    while(keys.hasMoreElements())
    {
      String key = (String)keys.nextElement();
      String type = (String)containedLSIDs.get(key);
      if(getType == null || getType.equals(type))
      {
        returnVec.addElement(new KeplerLSID(key));
      }
    }
    
    KeplerLSID[] lsids = new KeplerLSID[returnVec.size()];
    for(int i=0; i<returnVec.size(); i++)
    {
      lsids[i] = (KeplerLSID)returnVec.elementAt(i);
    }
    return lsids;
  }
  
  /**
   * returns the lsids of the objects in this kar that were identified as
   * "actorMetadata"
   */
  public KeplerLSID[] getActors()
    throws LSIDException
  {
    return getContentOfType("actorMetadata");
  }
  
  /**
   * serialize this object
   */
  public void writeExternal(ObjectOutput out) 
    throws IOException
  {
    out.writeObject(lsid.toString());
    out.writeObject(name);
    out.writeObject(karFile);
    out.writeObject(containedLSIDs);
    out.flush();
  }
  
  /** 
   * deserialize this object
   */
  public void readExternal(ObjectInput in) 
    throws IOException, ClassNotFoundException
  {
    try
    {
      lsid = new KeplerLSID((String)in.readObject());
    }
    catch(Exception e)
    {
      throw new IOException("Could not create new KeplerLSID: " + e.getMessage());
    }
    name = (String)in.readObject();
    karFile = (File)in.readObject();
    containedLSIDs = (Hashtable)in.readObject();
  }
  
  /**
   * move the ksw items to the correct place and make sure the ids are right
   */
  private Vector handleKSWItem(String type, KeplerLSID lsid, JarEntry entry,
    String kswUnpackDirName, KSWFile ksw)
    throws DuplicateLSIDException, Exception
  {
    Vector tempActorMetadataVector = new Vector();
    File kswUnpackDir = new File(kswUnpackDirName);

    if(type.equals("actorMetadata"))
    {
      //copyKSWEntryToDir(kswUnpackDirName, ksw, entry);
      //File cacheFile = new File(kswUnpackDirName, entry.getName());
      ActorCacheObject aco = new ActorCacheObject(ksw.getInputStream(entry));
      CacheManager.getInstance().insertObject(aco);
    }
    else if(type.equals("jar"))
    {
    }
    else if(type.equals("class"))
    {
    }
    else if(type.equals("nativeLibrary"))
    {
    }
    else if(type.equals("workflow"))
    {
    }
    else if(type.equals("xmlMetadata"))
    {
    }
    else if(type.equals("data"))
    {
    }
    else if(type.equals("file"))
    {
    }

    return tempActorMetadataVector;
  }
  
  /**
   * This method will write an KSW entry file into cache dir(without register
   * into cache
   *
   * @param dirName
   * @param ksw
   * @param entry
   * @throws IOException
   */
  private static File copyKSWEntryToDir(String dirName, KSWFile ksw, 
    JarEntry entry)  throws IOException
  {
      File f = new File(dirName + File.separator + entry.getName());
      InputStream is = ksw.getInputStream(entry);
      FileOutputStream fos = new FileOutputStream(f);
      CacheUtil.writeInputStreamToOutputStream(is, fos);
      fos.flush();
      fos.close();
      return f;
  }
  
}
