/**
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.scia;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Menu;
import java.awt.MenuItem;
import java.awt.PopupMenu;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.swing.JCheckBoxMenuItem;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTree;
import javax.swing.JLabel;
import javax.swing.SwingUtilities;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;


/**
 * The main graphical user interface for SCIA. Through the MainUI, users can
 * select domain, source schema, target schema, select running mode and then
 * match the schemas, provide feedback including manual matches especially
 * when asked at critical points, finally save the generated mappings,
 * generate views, check and execute the generated views.
 *
 *@author Guilian Wang and Vitaliy Zavesov
 */
public class MainUI extends JFrame {
    public static String SCIA_WORK_DIR = SCIA.SCIA_WORK_DIR;
    public static String LIB_JAR_DIR = SCIA.LIB_JAR_DIR;

    // number of pixels from the line to use when selecting it
    public static int CLICKABLE_DISTANCE = 5; // pixels from the lin        

    // target schemas of this application
    HashSet targetSchemas = new HashSet();

    // all source schema of this application
    HashSet allSourceSchemas = new HashSet();

    // table of schema's document name, virtual or real
    // key = schema file name
    // content = document name
    Hashtable docTable = new Hashtable();

    // table of target schema's corresponding source schema list
    // key = target schema file name, content = list of source schemas
    Hashtable sourceSchemas = new Hashtable();

    // table of individual target-source pair views
    // key = target schema file name + source schema file name
    // content = view string
    Hashtable individualViews = new Hashtable();

    // table of target views, key = virtual target document name
    // content = integrated view (union of the corresponding target's 
    // individual views
    Hashtable targetViews = new Hashtable();
    PrintWriter cout = new PrintWriter(System.err, true);
    SwingWorker worker;

    // add new vector to hold MatchingLines
    public Vector matchingLines = new Vector();

    // this is the selected line in the list
    MatchLine selectedLine = null;
    MatchingUnit selectedUnit = null;
    int leftSelectedNode = -1;
    int rightSelectedNode = -1;

    // the function to be entered for multiple matches
    String function = new String();

    // checks if the alt key is pressed - for multiple
    // matching
    public boolean altPressed = false;

    // allows the code to wait for condition to be entered
    boolean firstCondition = false;
    boolean secondCondition = false;
    TNode conditionNode = null;
    MatchLine conditionLine = null;
    Prompt conditionPrompt = null;
    boolean sameNodeCondition = false;

    // the vector of multiple left node matchings
    Vector multipleMatchings = new Vector();
    java.util.List matchingUnits = new ArrayList();

    // the list of union matches
    java.util.List unionMatches = new ArrayList();
    public boolean unionMode = false;

    // these variables allow to add union matches of
    // local multiple matches
    boolean putInIndependent = false;
    boolean putInDependent = false;

    SchemaTree schemaJTree1 = SCIA.schemaJTree1;
    SchemaTree schemaJTree2 = SCIA.schemaJTree2;
    TNode leftSelectedTNode;
    TNode rightSelectedTNode;
    boolean treeSelectionEventHappened = false;
    String selectedReuseMap;
    JScrollPane tree1SPane = new JScrollPane(schemaJTree1);
    JScrollPane tree2SPane = new JScrollPane(schemaJTree2);
    public MyJSplitPane treesSplitPane = new MyJSplitPane(JSplitPane.HORIZONTAL_SPLIT,
            tree1SPane, tree2SPane, schemaJTree1, schemaJTree2, matchingLines);
    Arrow arr = new Arrow(schemaJTree1, schemaJTree2);

    JLabel sourceSchemaNameLabel = new JLabel("    " + schemaJTree1.treeName);
    JLabel targetSchemaNameLabel = new JLabel(schemaJTree2.treeName + "    ");

    JPanel contentPane = new JPanel(new BorderLayout());
    JMenuItem createMappingItem;

    // Display options
    public boolean showCardinality = true;
    public boolean showDescription = true;
    public boolean showType = true;
    public boolean showSimilarity = true;

    // constructor
    public MainUI() throws Exception {
        super("SCIA Prototype V0.1");

        // add window listener
        addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent e) {
                    System.exit(0);
                }
            });

        // add the keyboard listener for multiple matching
        // if the user presses and holds the control button while 
        // clicking, then we will prompt him to enter a function
        KeyAdapter kl = new KeyAdapter() {
                /** Handle the key pressed event from the text field. */
                public void keyPressed(KeyEvent e) {
                    int key = e.getKeyCode();

                    if (key == KeyEvent.VK_ALT) {
                        altPressed = true;

                        //System.err.println("Alt pressed");
                    }
                }

                /** Handle the key released event from the text field. */
                public void keyReleased(KeyEvent e) {
                    int key = e.getKeyCode();

                    if (key == KeyEvent.VK_ALT) {
                        altPressed = false;

                        if (SCIA.debug_on) {
                            System.err.println("Alt released");
                        }
                    }
                }
            };

        schemaJTree1.addKeyListener(kl);
        schemaJTree2.addKeyListener(kl);

        //expand all rows initially
        schemaJTree1.setRowNum();
        schemaJTree2.setRowNum();

        TNode n;

        for (int i = 0; i < schemaJTree1.occupancy; i++) {
            n = schemaJTree1.getTNodeFromRowNum(i);

            if ((n != null) && (n.height < 6)) {
                schemaJTree1.expandRow(i);
            }
        }

        for (int i = 0; i < schemaJTree2.occupancy; i++) {
            n = schemaJTree2.getTNodeFromRowNum(i);

            if ((n != null) && (n.height < 6)) {
                schemaJTree2.expandRow(i);
            }
        }

        // set tree selection mode
        schemaJTree1.getSelectionModel().setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
        schemaJTree2.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);

        //Listen for when the selection changes in left tree.
        final MainUI mainUI = this;
        schemaJTree1.addTreeSelectionListener(new TreeSelectionListener() {
                public void valueChanged(TreeSelectionEvent e) {
                    leftSelectedTNode = (TNode) schemaJTree1.getLastSelectedPathComponent();

                    if (leftSelectedTNode == null) {
                        return;
                    }

                    arr.manualSelect("left");

                    if (altPressed == true) {
                        multipleMatchings.add(leftSelectedTNode);

                        MatchingUnit mu = new MatchingUnit(leftSelectedTNode,
                                null, null);
                        matchingUnits.add(mu);
                        selectedUnit = mu;

                        promptForFormula(false);
                    } else {
                        function = new String();
                        multipleMatchings = new Vector();
                        matchingUnits = new ArrayList();
                    }

                    //remove existing lines
                    treesSplitPane.paintImmediately(0, 0,
                        treesSplitPane.getWidth(), treesSplitPane.getHeight());

                    if (secondCondition) {

                        ConditionPopup poptxt = new ConditionPopup(SCIA.mainUI,
                                conditionNode, leftSelectedTNode);

                        if (conditionPrompt != null) {
                            conditionPrompt.hide();
                        }

                        poptxt.show();

                        firstCondition = false;
                        secondCondition = false;
                        sameNodeCondition = false;
                        conditionNode = null;
                    }

                    if (firstCondition && !secondCondition) {
                        if (conditionPrompt != null) {
                            conditionPrompt.hide();
                        }

                        conditionPrompt = 
                            new Prompt("Please select the second node\nor enter string",
                                       false, mainUI, null, false, true);
                        conditionPrompt.show();

                        conditionNode = leftSelectedTNode;

                        firstCondition = false;
                        secondCondition = true;
                    }
                }
            });

        //Listen for when the selection changes in right tree.
        schemaJTree2.addTreeSelectionListener(new TreeSelectionListener() {
                public void valueChanged(TreeSelectionEvent e) {
                    rightSelectedTNode = (TNode) schemaJTree2.getLastSelectedPathComponent();

                    if (rightSelectedTNode == null) {
                        return;
                    }

                    //treeSelectionEventHappened = true;
                    treesSplitPane.paintImmediately(0, 0,
                        treesSplitPane.getWidth(), treesSplitPane.getHeight());

                    if ((leftSelectedTNode != null) &&
                            (rightSelectedTNode != null)) {
                        //record the user input match to the tree node in the 
                        //right tree     
                        if (treesSplitPane.highlightNodes != null) {
                            return;
                        }

                        if ((rightSelectedTNode.fromComb != null) ||
			    (rightSelectedTNode.independentUnionMatches != null) ||
			    (rightSelectedTNode.dependentUnionMatches != null)) {
                            addUnionMatch();

                            return;
                        } else {

			    // for global unions
			    unionMode = false;
			}

                        int index = matchingLines.size();
                        MatchLine matchLine = null;

			// add a new matching line
			Matching matching = new Matching(leftSelectedTNode, 1.0);
			rightSelectedTNode.fromComb = matching;

                        if ((altPressed == true) &&
                                (multipleMatchings.size() > 0)) //? only allow local 2-to-1, it works for 3-to-1
                         {
                            promptForFormula(true);

                            matchLine = new MatchLine(multipleMatchings,
                                    rightSelectedTNode, index, 1);

			    findSameLines(matchLine);

                            matchLine.function = function;

                            function = new String();
                            multipleMatchings = new Vector();

                            rightSelectedTNode.fromComb.tnode = null;
                        } else {
                            function = new String();
                            multipleMatchings = new Vector();
                            matchingUnits = new ArrayList();

                            matchLine = new MatchLine(leftSelectedTNode,
                                    rightSelectedTNode, index, 1);

			    findSameLines(matchLine);
                        }

                        // add the new match to the vector of match lines
                        matchingLines.add(matchLine);

                        //schemaJTree2.write(cout, false);
                        if (SCIA.debug_on) {
                            cout.flush();
                        }

                        schemaJTree1.clearSelection();
                        schemaJTree2.clearSelection();
                    }

                    arr.manualSelect("right");
                }
            });

        // handle action on interactive box
        ActionListener interactAL = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    JCheckBoxMenuItem item = (JCheckBoxMenuItem) ae.getSource();

                    if (item.isSelected()) {
                        SCIA.interactive_mode = true;
                    } else if (!item.isSelected()) {
                        SCIA.interactive_mode = false;
                    }
                }
            };

        // Handle action on reuse mapping box
        ActionListener reuseBoxAL = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    JCheckBoxMenuItem item = (JCheckBoxMenuItem) ae.getSource();

                    if (item.isSelected()) {
                        SCIA.reuseMap_on = true;
                    } else if (!item.isSelected()) {
                        SCIA.reuseMap_on = false;
                    }
                }
            };

        // Handle action on direction box
        ActionListener directionBoxAL = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    JCheckBoxMenuItem item = (JCheckBoxMenuItem) ae.getSource();

                    if (item.isSelected()) {
                        SCIA.bothDirection = true;
                    } else if (!item.isSelected()) {
                        SCIA.bothDirection = false;
                    }
                }
            };

        // Handle action on show cardinality menu item
        ActionListener showCardinalityAL = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    JCheckBoxMenuItem item = (JCheckBoxMenuItem) ae.getSource();

                    if (item.isSelected()) {
                        showCardinality = true;
                    } else if (!item.isSelected()) {
                        showCardinality = false;
                    }

                    treesSplitPane.paintImmediately(0, 0,
                        treesSplitPane.getWidth(), treesSplitPane.getHeight());
                }
            };

        // Handle action on show type menu item
        ActionListener showTypeAL = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    JCheckBoxMenuItem item = (JCheckBoxMenuItem) ae.getSource();

                    if (item.isSelected()) {
                        showType = true;
                    } else if (!item.isSelected()) {
                        showType = false;
                    }

                    treesSplitPane.paintImmediately(0, 0,
                        treesSplitPane.getWidth(), treesSplitPane.getHeight());
                }
            };

        // Handle action on show similarity menu item
        ActionListener showSimilarityAL = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    JCheckBoxMenuItem item = (JCheckBoxMenuItem) ae.getSource();

                    if (item.isSelected()) {
                        showSimilarity = true;
                    } else if (!item.isSelected()) {
                        showSimilarity = false;
                    }

                    treesSplitPane.paintImmediately(0, 0,
                        treesSplitPane.getWidth(), treesSplitPane.getHeight());
                }
            };

	// Handle the help button "Available Operations"
	ActionListener functionsHelpAL = new ActionListener() {
		public void actionPerformed(ActionEvent ae) {
		    
		    String functionsHelp = new String();

		    try {
			FileReader fr = new FileReader(SCIA_WORK_DIR+"/helpFiles/FunctionsHelp.txt");
			BufferedReader br = new BufferedReader(fr);
			String record = new String();
			while((record = br.readLine()) != null) {
			    functionsHelp += record;
			    functionsHelp += "\n";
			}
		    }
		    catch(IOException e) {
			e.printStackTrace();
		    }

		    HelpViewer hv = new HelpViewer("Available Operations",
						   "Please use one of the following functions", 
						   functionsHelp);
		    hv.show();
		}
	    };

        // Handle action on show description menu item
        ActionListener showDescriptionAL = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    JCheckBoxMenuItem item = (JCheckBoxMenuItem) ae.getSource();

                    if (item.isSelected()) {
                        showDescription = true;
                    } else if (!item.isSelected()) {
                        showDescription = false;
                    }

                    treesSplitPane.paintImmediately(0, 0,
                        treesSplitPane.getWidth(), treesSplitPane.getHeight());
                }
            };

        ActionListener aRM = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    JFileChooser fileChooser = new JFileChooser(SCIA_WORK_DIR +
                            "/output/mappings/");

                    int chooserResult = fileChooser.showOpenDialog(treesSplitPane);

                    String selectedReuseMap = fileChooser.getSelectedFile()
                                                         .getPath();

                    if (selectedReuseMap == null) {
                        return;
                    }

                    if (SCIA.debug_on) {
                        System.err.println("to reuse " + selectedReuseMap);
                    }

                    SAXBuilder builder = new SAXBuilder();

                    try {
                        InputStream is = new FileInputStream(selectedReuseMap);
                        Document doc = builder.build(is);

                        // check the root element name == mapping
                        Element rootElem = doc.getRootElement();
                        String rootTag = new String();

                        if (rootElem != null) {
                            rootTag = rootElem.getName();

                            if (SCIA.debug_on) {
                                System.err.println("rootTag = " + rootTag);
                            }
                        }

                        if ((rootElem == null) || (rootTag == null) ||
                                !rootTag.equals("mapping")) {
                            if (SCIA.debug_on) {
                                System.err.println("The selected mapping is " +
                                    "not appliable, root element " +
                                    "is not as supposed.");
                            }

                            return;
                        }

                        // check the map is for the right pair of schemas
                        String targetSchema = rootElem.getChildTextTrim(
                                "targetSchema");
                        String sourceSchema = rootElem.getChildTextTrim(
                                "sourceSchema");

                        if (!sourceSchema.equals(schemaJTree1.treeName) ||
                                !targetSchema.equals(schemaJTree2.treeName)) {
                            if (SCIA.debug_on) {
                                System.err.println("schemas in the selected" +
                                    " mapping document are not consistent with" +
                                    " the current task, so not appliable");
                            }

                            return;
                        }

                        /* set reuseMap to true, and initiate mapping from the
                           document, matching process will start from it
                           instead of from scratch
                        */
                        SCIA.reuseMap_on = true;
                        schemaJTree2.initMappingFromXML(rootElem, schemaJTree1);
                    } catch (Exception e) {
                        if (SCIA.debug_on) {
                            System.err.println("Processing " +
                                selectedReuseMap + ": ");
                        }

                        e.printStackTrace();
                    }

                    //reading from text file
                    StringTokenizer st;
                    String line;

                    try {
                        FileInputStream fis = new FileInputStream(selectedReuseMap);
                        BufferedReader br = new BufferedReader(new InputStreamReader(
                                    fis));

                        // check whether the mapping applicable from 
                        // the first line
                        line = br.readLine();
                        st = new StringTokenizer(line);

                        if (st.countTokens() == 4) {
                            st.nextElement(); // throw "Mapping:"

                            String sourceTreeName = (String) st.nextElement();
                            st.nextElement(); // throw " --> "

                            String targetTreeName = (String) st.nextElement();

                            if (!sourceTreeName.equals(schemaJTree1.treeName) ||
                                    !targetTreeName.equals(
                                        schemaJTree2.treeName)) {
                                if (SCIA.debug_on) {
                                    System.err.println("source tree name = " +
                                        sourceTreeName);
                                }

                                if (SCIA.debug_on) {
                                    System.err.println("target tree name = " +
                                        targetTreeName);
                                }

                                if (SCIA.debug_on) {
                                    System.err.println("tree1 name = " +
                                        schemaJTree1.treeName);
                                }

                                if (SCIA.debug_on) {
                                    System.err.println("tree2 name = " +
                                        schemaJTree2.treeName);
                                }

                                if (SCIA.debug_on) {
                                    System.err.println(
                                        "The selected mapping file " +
                                        "is not appliable");
                                }
                            } else {
                                /* set reuseMap to true, and initiate mapping
                                   from the file,
                                   matching process will start from it instead
                                   of scratch
                                */
                                SCIA.reuseMap_on = true;
                                schemaJTree2.initMappingFromFile(br,
                                    schemaJTree1);
                            }
                        }

                        br.close();
                        fis.close();
                    } catch (IOException e) {
                        if (SCIA.debug_on) {
                            System.err.println(e.getMessage());
                        }
                    }

                    // display the existing mapping on GUI
                    addMatchedLines();
                    treesSplitPane.drawMapping(treesSplitPane.getGraphics());
                }
            };

        //Create the combo box for domain input
        String[] domainStrings = {
                "Bibliography", "Biology", "Business", "Ecology", "Other"
            };

        ActionListener a1 = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    SCIA.domain = ((JMenuItem) ae.getSource()).getLabel();

                    if (SCIA.debug_on) {
                        System.err.println("domain = " + SCIA.domain);
                    }

                    SCIA.appDomain = SCIA.loadDomainThesauri(SCIA.domain);

                    if (SCIA.debug_on) {
                        System.err.println("Application Domain is " +
                            SCIA.domain);
                    }

                    if (SCIA.debug_on) {
                        System.err.println(SCIA.domain + "'s thesauri " +
                            "are used in linguistic matching.");
                    }
                }
            };

        ActionListener a2 = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    SchemaTree schemaTree1 = new SchemaTree();
                    String sourceSchemaType = null;

                    JFileChooser fileChooser = new JFileChooser(SCIA_WORK_DIR +
                            "/schemas/");
                    int chooserResult = fileChooser.showOpenDialog(treesSplitPane);

                    String sourceSchema = fileChooser.getSelectedFile().getPath();

                    if (sourceSchema == null) {
                        return;
                    }

                    if (SCIA.debug_on) {
                        System.err.println("current source schema = " +
                            sourceSchema);
                    }

                    // change the first tree
                    if (sourceSchema.endsWith("dtd") ||
                            sourceSchema.endsWith("DTD")) {
                        sourceSchemaType = "DTD";
                        schemaTree1 = new SchemaTree(sourceSchema,
                                sourceSchemaType);
                    } else if (sourceSchema.endsWith("xml") ||
                            sourceSchema.endsWith("XML")) {
                        //to do: differentiate EML from others
                        sourceSchemaType = "XMLSchema";

                        SAXBuilder builder = new SAXBuilder();

                        try {
                            InputStream is = new FileInputStream(sourceSchema);

                            Document doc = builder.build(is);
                            Element rootElem = doc.getRootElement();

                            if (rootElem != null) {
                                String rootName = rootElem.getName();

                                if (rootName.equals("eml")) {
                                    EMLParser p = new EMLParser();

                                    String extractedSchema = p.getXSD(sourceSchema);
                                    schemaTree1 = (new XMLSchemaImporter(SCIA_WORK_DIR +
                                            "/schemas/" + extractedSchema)).tree;
                                } else {
                                    schemaTree1 = new SchemaTree(sourceSchema,
                                            sourceSchemaType);
                                }
                            }
                        } catch (JDOMException e) {
                            // indicates a well-formedness error 
                            e.printStackTrace();
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                    } else if (sourceSchema.endsWith("xsd") ||
                            sourceSchema.endsWith("XSD")) {
                        sourceSchemaType = "XMLSchema";
                        schemaTree1 = (new XMLSchemaImporter(sourceSchema)).tree;
                    } else if (sourceSchema.endsWith("owl") ||
                            sourceSchema.endsWith("OWL")) {
                        sourceSchemaType = "Ontology";
                        schemaTree1 = new SchemaTree(sourceSchema, "Ontology");
                    }

                    try {
                        schemaTree1.mainUI = SCIA.mainUI;
                        schemaJTree1.rootNode = schemaTree1.rootNode;
                        schemaJTree1.occupancy = schemaTree1.occupancy;
                        schemaJTree1.treeName = schemaTree1.treeName;
                        schemaJTree1.schemaType = sourceSchemaType;
                        schemaJTree1.changeRoot(schemaTree1.rootNode);
                        schemaJTree1.mainUI = SCIA.mainUI;
                        SCIA.schemaJTree1 = schemaJTree1;
                        SCIA.sourceSchemaType = sourceSchemaType;

                        sourceSchemaNameLabel.setText("    " + 
                                                      schemaJTree1.treeName);

                        if (SCIA.debug_on) {
                            cout.flush();
                        }
                    } catch (java.lang.Exception e) {
                        if (SCIA.debug_on) {
                            System.err.println("Creating source schema tree, " +
                                e);
                        }
                    }
                }
            };

        ActionListener a3 = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    SchemaTree schemaTree2 = new SchemaTree();
                    String targetSchemaType = null;

                    JFileChooser fileChooser = new JFileChooser(SCIA_WORK_DIR +
                            "/schemas/");
                    int chooserResult = fileChooser.showOpenDialog(treesSplitPane);

                    String targetSchema = fileChooser.getSelectedFile().getPath();

                    if (targetSchema == null) {
                        return;
                    }

                    if (SCIA.debug_on) {
                        System.err.println("targetSchema = " + targetSchema);
                    }

                    // change the second tree
                    if (targetSchema.endsWith("dtd") ||
                            targetSchema.endsWith("DTD")) {
                        targetSchemaType = "DTD";
                        schemaTree2 = new SchemaTree(targetSchema,
                                targetSchemaType);
                    } else if (targetSchema.endsWith("xml") ||
                            targetSchema.endsWith("XML")) {
                        //to do: differentiate EML from others
                        targetSchemaType = "XMLSchema";

                        SAXBuilder builder = new SAXBuilder();

                        try {
                            InputStream is = new FileInputStream(targetSchema);
                            Document doc = builder.build(is);
                            Element rootElem = doc.getRootElement();

                            if (rootElem != null) {
                                String rootName = rootElem.getName();

                                if (rootName.equals("eml")) {
                                    EMLParser p = new EMLParser();
                                    String extractedSchema = p.getXSD(targetSchema);
                                    schemaTree2 = (new XMLSchemaImporter(SCIA_WORK_DIR +
                                            "/schemas/" + extractedSchema)).tree;
                                } else {
                                    schemaTree2 = new SchemaTree(targetSchema,
                                            targetSchemaType);
                                }
                            }
                        } catch (JDOMException e) {
                            // indicates a well-formedness error 
                            e.printStackTrace();
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                    } else if (targetSchema.endsWith("xsd") ||
                            targetSchema.endsWith("XSD")) {
                        targetSchemaType = "XMLSchema";
                        schemaTree2 = (new XMLSchemaImporter(targetSchema)).tree;
                        schemaTree2.schemaType = targetSchemaType;
                    }

                    try {
                        schemaTree2.mainUI = SCIA.mainUI;

                        schemaJTree2.changeRoot(schemaTree2.rootNode);
                        schemaJTree2.rootNode = schemaTree2.rootNode;
                        schemaJTree2.occupancy = schemaTree2.occupancy;
                        schemaJTree2.treeName = schemaTree2.treeName;
                        schemaJTree2.schemaType = schemaTree2.schemaType;
                        schemaJTree2.mainUI = SCIA.mainUI;
                        SCIA.schemaJTree2 = schemaJTree2;
                        SCIA.targetSchemaType = targetSchemaType;

                        targetSchemaNameLabel.setText(schemaJTree2.treeName + 
                                                      "    ");

                        //schemaJTree2.write(cout, false);
                        if (SCIA.debug_on) {
                            cout.flush();
                        }
                    } catch (java.lang.Exception e) {
                        if (SCIA.debug_on) {
                            System.err.println("From schemaList 2 action, " +
                                "schemaJTree2 construction: " + e);
                        }
                    }
                }
            };

        // add an action listener for the right-click menu
        ActionListener popupAL = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    if (ae.getSource() instanceof MenuItem) {
                        MenuItem mi = (MenuItem) ae.getSource();
                        String label = mi.getLabel();

                        if (label.equals("Enter Condition")) {
                            if (SCIA.debug_on) {
                                System.err.println(
                                    "Condition option is selected");
                            }

                            conditionLine = selectedLine;

                            conditionPrompt = new Prompt("Enter Condition:\nPlease select the first node",
                                                         true, null, null, false, false);

                            conditionPrompt.show();

                            firstCondition = true;
                        } else if (label.equals("Delete Condition")) {
                            if (selectedLine != null) {
                                selectedLine.rightTNode.fromComb.condition = null;
                                selectedLine.condition = new String();

                                //remove existing lines
                                treesSplitPane.paintImmediately(0, 0,
                                    treesSplitPane.getWidth(),
                                    treesSplitPane.getHeight());
                                treesSplitPane.drawMapping(treesSplitPane.getGraphics());
                            }
                        } else if (label.equals("Enter Operations")) {
                            worker = new SwingWorker() {
                                        public Object construct() {
                                            try {
                                                String msg = "Please enter operation";

                                                Interactive1 prompt = new Interactive1(msg);

                                                prompt.show();

                                                while (prompt.isVisible()) {
                                                    Thread.sleep(100);
                                                }

                                                if (selectedLine != null) {
                                                    selectedLine.rightTNode.fromComb.operations = prompt.data;
                                                    selectedLine.operations = prompt.data;
                                                } else {
                                                    Interactive4 prompt1 = new Interactive4(
                                                            "Select line first");
                                                    prompt1.show();
                                                }
                                            } catch (Exception e) {
                                                e.printStackTrace();
                                            }

                                            return null;
                                        }
                                    };

                            worker.start();
                        } else if (label.equals("Delete Operations")) {
                            if (selectedLine != null) {
                                selectedLine.rightTNode.fromComb.operations = new String();
                                selectedLine.operations = new String();

                                //remove existing lines
                                treesSplitPane.paintImmediately(0, 0,
                                    treesSplitPane.getWidth(),
                                    treesSplitPane.getHeight());

                                treesSplitPane.drawMapping(treesSplitPane.getGraphics());
                            }
                        } else if (label.equals("Match Formula")) {
                            Interactive4 prompt = null;

                            if (selectedLine != null) {

				String form = getFormulaString(selectedLine);

                                if (SCIA.debug_on) {
                                    System.err.println("form = " + form);
                                }

                                prompt = new Interactive4("Formula: " + form);
                            } else {
                                prompt = new Interactive4("Select line first");
                            }

                            prompt.show();
                        } else if (label.equals("Show Operations")) {
                            Interactive4 prompt = null;

                            if (selectedLine != null) {
                                prompt = new Interactive4("Operations: " +
                                        selectedLine.operations);
                            } else {
                                prompt = new Interactive4("Select line first");
                            }

                            prompt.show();
                        } else if (label.equals("Show Condition")) {
                            Interactive4 prompt = null;

                            if (selectedLine != null) {
                                prompt = new Interactive4("Condition: " +
                                        selectedLine.condition);
                            } else {
                                prompt = new Interactive4("Select line first");
                            }

                            prompt.show();
                        } else if (label.equals("Enter Grouping")) {
                            enterGroupAttrs();
                        } else if (label.equals("Enter Aggregation")) {
                            worker = new SwingWorker() {
                                        public Object construct() {
                                            try {
                                                MatchLine tmpLine = selectedLine;

                                                String aggOp = "";

                                                if (tmpLine != null) {
                                                    Interactive1 prompt = new Interactive1(
                                                            "Please enter the aggregation operator");
                                                    prompt.show();

                                                    while (prompt.isVisible()) {
                                                        Thread.sleep(100);
                                                    }

                                                    aggOp = prompt.data;
                                                } else {
                                                    Interactive4 prompt = new Interactive4(
                                                            "Select line first");
                                                    prompt.show();
                                                }

                                                if ((aggOp != null) &&
                                                        (tmpLine.rightTNode.fromComb != null)) {
                                                    tmpLine.rightTNode.fromComb.aggregateOp = aggOp;
                                                    selectedLine.aggregation = aggOp;
                                                }

                                                return null;
                                            } catch (Exception e) {
                                                if (SCIA.debug_on) {
                                                    System.err.println(
                                                        "From SchemaTreeMatch:");
                                                }

                                                e.printStackTrace();

                                                return null;
                                            }
                                        }
                                    };

                            worker.start();
                        } else if (label.equals("Show Grouping")) {
                            worker = new SwingWorker() {
                                        public Object construct() {
                                            try {
                                                Interactive4 prompt = null;

                                                if (selectedLine != null) {
                                                    String message = new String(
                                                            "Grouping Attributes: " +
                                                            selectedLine.grouping);

                                                    prompt = new Interactive4(message);
                                                } else {
                                                    prompt = new Interactive4(
                                                            "Select line first");
                                                }

                                                prompt.show();

                                                return null;
                                            } catch (Exception e) {
                                                if (SCIA.debug_on) {
                                                    System.err.println(
                                                        "From SchemaTreeMatch:");
                                                }

                                                e.printStackTrace();

                                                return null;
                                            }
                                        }
                                    };

                            worker.start();
                        } else if (label.equals("Show Aggregation")) {
                            worker = new SwingWorker() {
                                        public Object construct() {
                                            try {
                                                MatchLine tmpLine = selectedLine;

                                                Interactive4 prompt = null;

                                                if ((tmpLine != null) &&
                                                        (tmpLine.rightTNode.fromComb != null)) {
                                                    String message = new String(
                                                            "Aggregation Operator: " +
                                                            selectedLine.aggregation);

                                                    prompt = new Interactive4(message);
                                                } else {
                                                    prompt = new Interactive4(
                                                            "Select line first");
                                                }

                                                prompt.show();

                                                return null;
                                            } catch (Exception e) {
                                                e.printStackTrace();

                                                return null;
                                            }
                                        }
                                    };

                            worker.start();
                        } else if (label.equals("Delete Grouping")) {
                            if ((selectedLine != null) &&
                                    (selectedLine.rightTNode.fromComb != null)) {
                                selectedLine.rightTNode.fromComb.groupAttrs = new Vector();
                                selectedLine.grouping = new String();
                            }
                        } else if (label.equals("Delete Aggregation")) {
                            if ((selectedLine != null) &&
                                    (selectedLine.rightTNode.fromComb != null)) {
                                selectedLine.rightTNode.fromComb.aggregateOp = "";
                                selectedLine.aggregation = new String();
                            }
                        } else if (label.equals("Delete")) {
                            if (selectedLine != null) {
                                if (SCIA.debug_on) {
                                    System.err.println(
                                        "Delete match line option is selected");
                                }

                                if (selectedLine.rightTNode.dependentUnionMatches != null) {
                                    java.util.List matches = selectedLine.rightTNode.dependentUnionMatches;

                                    if (matches.size() >= 2) {
                                        // remove the selected match
                                        for (int i = 0; i < matches.size();
                                                i++) {
                                            Matching match = (Matching) matches.get(i);

                                            if ((match != null) &&
                                                    (match.tnode == selectedLine.leftTNode)) {
                                                matches.remove(i);
                                            }
                                        }

                                        // if only two, clear dependentUnionMatches, 
                                        // take the left one as fromComb
                                        if (matches.size() == 2) {
                                            selectedLine.rightTNode.fromComb = (Matching) matches.get(0);
                                            matches.clear();
                                        }
                                    }
                                } else if (selectedLine.rightTNode.independentUnionMatches != null) {
                                    java.util.List matches = selectedLine.rightTNode.independentUnionMatches;

                                    if (matches.size() >= 2) {
                                        // remove the selected match
                                        for (int i = 0; i < matches.size();
                                                i++) {
                                            Matching match = (Matching) matches.get(i);

                                            if ((match != null) &&
                                                    (match.tnode == selectedLine.leftTNode)) {
                                                matches.remove(i);
                                            }
                                        }

                                        // if only two, clear independentUnion, 
                                        // take the left one as fromComb
                                        if (matches.size() == 2) {
                                            selectedLine.rightTNode.fromComb = (Matching) matches.get(0);
                                            matches.clear();
                                        }
                                    }
                                } else if (selectedLine.rightTNode.fromComb != null) {
                                    //if exists only a simple or complex local match,
                                    //clear it
                                    selectedLine.rightTNode.fromComb = null;
                                }

                                for (int i = 0; i < selectedLine.sameLines.size(); i++) {
				    MatchLine line = (MatchLine)selectedLine.sameLines.get(i);
				    line.sameLines.remove(selectedLine);
                                }

                                matchingLines.remove(selectedLine);

                                function = new String();
                                multipleMatchings = new Vector();
                                matchingUnits = new ArrayList();
                            }

                            //remove existing lines
                            treesSplitPane.paintImmediately(0, 0,
                                treesSplitPane.getWidth(),
                                treesSplitPane.getHeight());

                            treesSplitPane.drawMapping(treesSplitPane.getGraphics());
                        } else if (label.equals("Delete Subtree")) {
                            if ((leftSelectedNode != -1) &&
                                    (schemaJTree1.getLastSelectedPathComponent() != null)) {
                                schemaJTree1.collapseRow(leftSelectedNode);

                                TNode selectedNodeElem = (TNode) schemaJTree1.getLastSelectedPathComponent();
                                selectedNodeElem.deletedChildren = selectedNodeElem.children;
                                selectedNodeElem.children = new Vector();

                                treesSplitPane.paintImmediately(0, 0,
                                    treesSplitPane.getWidth(),
                                    treesSplitPane.getHeight());

                                treesSplitPane.drawMapping(treesSplitPane.getGraphics());
                            }

                            if ((rightSelectedNode != -1) &&
                                    (schemaJTree2.getLastSelectedPathComponent() != null)) {
                                schemaJTree2.collapseRow(rightSelectedNode);

                                TNode selectedNodeElem = (TNode) schemaJTree2.getLastSelectedPathComponent();
                                selectedNodeElem.deletedChildren = selectedNodeElem.children;
                                selectedNodeElem.children = new Vector();

                                treesSplitPane.paintImmediately(0, 0,
                                    treesSplitPane.getWidth(),
                                    treesSplitPane.getHeight());

                                treesSplitPane.drawMapping(treesSplitPane.getGraphics());
                            }
                        } else if (label.equals("Undelete Subtree")) {
                            if ((leftSelectedNode != -1) &&
                                    (schemaJTree1.getLastSelectedPathComponent() != null)) {
                                TNode selectedNodeElem = (TNode) schemaJTree1.getLastSelectedPathComponent();

                                if (selectedNodeElem.deletedChildren.size() > 0) {
                                    selectedNodeElem.children = selectedNodeElem.deletedChildren;
                                    selectedNodeElem.deletedChildren = new Vector();

                                    schemaJTree1.expandRow(leftSelectedNode);

                                    treesSplitPane.paintImmediately(0, 0,
                                        treesSplitPane.getWidth(),
                                        treesSplitPane.getHeight());

                                    treesSplitPane.drawMapping(treesSplitPane.getGraphics());
                                }
                            }

                            if ((rightSelectedNode != -1) &&
                                    (schemaJTree2.getLastSelectedPathComponent() != null)) {
                                TNode selectedNodeElem = (TNode) schemaJTree2.getLastSelectedPathComponent();

                                if (selectedNodeElem.deletedChildren.size() > 0) {
                                    selectedNodeElem.children = selectedNodeElem.deletedChildren;
                                    selectedNodeElem.deletedChildren = new Vector();

                                    schemaJTree2.expandRow(rightSelectedNode);

                                    treesSplitPane.paintImmediately(0, 0,
                                        treesSplitPane.getWidth(),
                                        treesSplitPane.getHeight());

                                    treesSplitPane.drawMapping(treesSplitPane.getGraphics());
                                }
                            }
                        } else if (label.equals("Expand Recursion")) {
                            if ((leftSelectedNode != -1) &&
                                    (schemaJTree1.getLastSelectedPathComponent() != null)) {
                                TNode selectedNodeElem = (TNode) schemaJTree1.getLastSelectedPathComponent();

                                selectedNodeElem.addRefSubTree();

                                schemaJTree1.expandRow(leftSelectedNode);

                                treesSplitPane.paintImmediately(0, 0,
                                    treesSplitPane.getWidth(),
                                    treesSplitPane.getHeight());

                                treesSplitPane.drawMapping(treesSplitPane.getGraphics());
                            }

                            if ((rightSelectedNode != -1) &&
                                    (schemaJTree2.getLastSelectedPathComponent() != null)) {
                                TNode selectedNodeElem = (TNode) schemaJTree2.getLastSelectedPathComponent();

                                selectedNodeElem.addRefSubTree();

                                schemaJTree2.expandRow(rightSelectedNode);

                                treesSplitPane.paintImmediately(0, 0,
                                    treesSplitPane.getWidth(),
                                    treesSplitPane.getHeight());

                                treesSplitPane.drawMapping(treesSplitPane.getGraphics());
                            }
                        } else if (label.equals("Enter Constant Value")) {
                            worker = new SwingWorker() {
                                        public Object construct() {
                                            try {
                                                TNode node = (TNode) schemaJTree2.getLastSelectedPathComponent();

                                                String msg = "Please enter the value:";
                                                Interactive1 inter = new Interactive1(msg);

                                                inter.show();

                                                while (inter.isVisible()) {
                                                    Thread.sleep(100);
                                                }

                                                if (node != null) {
                                                    node.constantValue = inter.data;
                                                }

                                                return null;
                                            } catch (Exception e) {
                                                e.printStackTrace();

                                                return null;
                                            }
                                        }
                                    };
                            worker.start();
                        } else if (label.equals("Show Constant Value")) {
                            TNode node = (TNode) schemaJTree2.getLastSelectedPathComponent();

                            String msg = new String("Constant value: ");

                            if ((node != null) && (node.constantValue != null)) {
                                msg += node.constantValue;
                            }

                            Interactive4 prompt = new Interactive4(msg);

                            prompt.show();
                        } else if (label.equals("Clear Constant Value")) {
                            TNode node = (TNode) schemaJTree2.getLastSelectedPathComponent();

                            if (node != null) {
                                node.constantValue = new String();
                            }
                        }
			else if (label.equals("Select Line")) {

			    worker = new SwingWorker() {
				    public Object construct() {
					try {
					    MatchLine tmpLine = selectedLine;

					    Interactive4 prompt = null;

					    if (tmpLine != null) {

						String message = 
						    "Please select one of the "+
						    "following lines (functions are shown):";

						Vector choices = new Vector();
						
						choices.add(getFormulaString(tmpLine));
						for(int i = 0; 
						    i < tmpLine.sameLines.size(); i++) {

						    String form = 
							getFormulaString((MatchLine)tmpLine.sameLines.get(i));
						    choices.add(form);
						}

						final String msg = message;
						final Vector c = choices;

						final Interactive2 inter2 = 
						    new Interactive2(msg, c);

						inter2.show();

						while (inter2.isVisible()) {
						    Thread.sleep(100);
						}

						String dataEntered = inter2.data;

						if(!dataEntered.equals("1")) {

						    for(int i = 0;
							i < tmpLine.sameLines.size(); i++) {

							String iStr = String.valueOf(i+2);

							if(dataEntered.equals(iStr)) {

							    MatchLine topLine = 
								(MatchLine)tmpLine.sameLines.get(i);
							    
							    matchingLines.remove(topLine);
							    matchingLines.add(topLine);

							    tmpLine.topLine = false;
							    topLine.topLine = true;
							}
						    }
						}
					    }
					} catch (Exception e) {
					    e.printStackTrace();
					}

					treesSplitPane.paintImmediately(0, 0,
						       treesSplitPane.getWidth(),
						       treesSplitPane.getHeight());
			    
					treesSplitPane.drawMapping(treesSplitPane.getGraphics());

					return null;
				    }
				};

                            worker.start();
			}
                    }
                }
            };

        // creating the right-click menu for the left tree lines
        final PopupMenu rightClick1 = new PopupMenu();

        // making menu items
        MenuItem enterCondition1 = new MenuItem("Enter Condition");
        MenuItem delete1 = new MenuItem("Delete");
        MenuItem enterOperations1 = new MenuItem("Enter Operations");
        MenuItem deleteCondition1 = new MenuItem("Delete Condition");
        MenuItem deleteOperations1 = new MenuItem("Delete Operations");
        MenuItem showFormula1 = new MenuItem("Match Formula");
        MenuItem showOperations1 = new MenuItem("Show Operations");
        MenuItem showCondition1 = new MenuItem("Show Condition");
        MenuItem enterGrouping1 = new MenuItem("Enter Grouping");
        MenuItem showGrouping1 = new MenuItem("Show Grouping");
        MenuItem deleteGrouping1 = new MenuItem("Delete Grouping");
        MenuItem enterAggregation1 = new MenuItem("Enter Aggregation");
        MenuItem showAggregation1 = new MenuItem("Show Aggregation");
        MenuItem deleteAggregation1 = new MenuItem("Delete Aggregation");
	final MenuItem selectLine1 = new MenuItem("Select Line");

        // adding listeners to menu items
        enterCondition1.addActionListener(popupAL);
        delete1.addActionListener(popupAL);
        enterOperations1.addActionListener(popupAL);
        deleteCondition1.addActionListener(popupAL);
        deleteOperations1.addActionListener(popupAL);
        showFormula1.addActionListener(popupAL);
        showOperations1.addActionListener(popupAL);
        showCondition1.addActionListener(popupAL);
        enterGrouping1.addActionListener(popupAL);
        showGrouping1.addActionListener(popupAL);
        deleteGrouping1.addActionListener(popupAL);
        enterAggregation1.addActionListener(popupAL);
        showAggregation1.addActionListener(popupAL);
        deleteAggregation1.addActionListener(popupAL);
	selectLine1.addActionListener(popupAL);

        //grouping items into sub-menus
        Menu conditionMenu1 = new Menu("     Condition");
        Menu functionMenu1 = new Menu("Match Formula");
        Menu operationsMenu1 = new Menu("     Operations");
        Menu groupingMenu1 = new Menu("     Grouping");
        Menu aggregationMenu1 = new Menu("     Aggregation");

        //adding items to sub-menus
        conditionMenu1.add(enterCondition1);
        conditionMenu1.add(showCondition1);
        conditionMenu1.add(deleteCondition1);
        operationsMenu1.add(enterOperations1);
        operationsMenu1.add(showOperations1);
        operationsMenu1.add(deleteOperations1);
        groupingMenu1.add(enterGrouping1);
        groupingMenu1.add(showGrouping1);
        groupingMenu1.add(deleteGrouping1);
        aggregationMenu1.add(enterAggregation1);
        aggregationMenu1.add(showAggregation1);
        aggregationMenu1.add(deleteAggregation1);

        // adding menu items to the menu
        rightClick1.add(showFormula1);
        rightClick1.add(conditionMenu1);
        rightClick1.add(operationsMenu1);
        rightClick1.add(groupingMenu1);
        rightClick1.add(aggregationMenu1);
        rightClick1.add(delete1);

        // adding popup menu to the left tree
        schemaJTree1.add(rightClick1);

        // creating the right-click window for the right tree
        final PopupMenu rightClick2 = new PopupMenu();

        // making menu items
        MenuItem enterCondition2 = new MenuItem("Enter Condition");
        MenuItem delete2 = new MenuItem("Delete");
        MenuItem enterOperations2 = new MenuItem("Enter Operations");
        MenuItem deleteCondition2 = new MenuItem("Delete Condition");
        MenuItem deleteOperations2 = new MenuItem("Delete Operations");
        MenuItem showFormula2 = new MenuItem("Match Formula");
        MenuItem showOperations2 = new MenuItem("Show Operations");
        MenuItem showCondition2 = new MenuItem("Show Condition");
        MenuItem enterGrouping2 = new MenuItem("Enter Grouping");
        MenuItem showGrouping2 = new MenuItem("Show Grouping");
        MenuItem deleteGrouping2 = new MenuItem("Delete Grouping");
        MenuItem enterAggregation2 = new MenuItem("Enter Aggregation");
        MenuItem showAggregation2 = new MenuItem("Show Aggregation");
        MenuItem deleteAggregation2 = new MenuItem("Delete Aggregation");
	final MenuItem selectLine2 = new MenuItem("Select Line");

        // adding listeners to menu items
        enterCondition2.addActionListener(popupAL);
        delete2.addActionListener(popupAL);
        enterOperations2.addActionListener(popupAL);
        deleteCondition2.addActionListener(popupAL);
        deleteOperations2.addActionListener(popupAL);
        showFormula2.addActionListener(popupAL);
        showOperations2.addActionListener(popupAL);
        showCondition2.addActionListener(popupAL);
        enterGrouping2.addActionListener(popupAL);
        showGrouping2.addActionListener(popupAL);
        deleteGrouping2.addActionListener(popupAL);
        enterAggregation2.addActionListener(popupAL);
        showAggregation2.addActionListener(popupAL);
        deleteAggregation2.addActionListener(popupAL);
	selectLine2.addActionListener(popupAL);

        //grouping items into sub-menus
        Menu conditionMenu2 = new Menu("     Condition");
        Menu operationsMenu2 = new Menu("     Operations");
        Menu groupingMenu2 = new Menu("     Grouping");
        Menu aggregationMenu2 = new Menu("     Aggregation");

        //adding items to sub-menus
        conditionMenu2.add(enterCondition2);
        conditionMenu2.add(showCondition2);
        conditionMenu2.add(deleteCondition2);
        operationsMenu2.add(enterOperations2);
        operationsMenu2.add(showOperations2);
        operationsMenu2.add(deleteOperations2);
        groupingMenu2.add(enterGrouping2);
        groupingMenu2.add(showGrouping2);
        groupingMenu2.add(deleteGrouping2);
        aggregationMenu2.add(enterAggregation2);
        aggregationMenu2.add(showAggregation2);
        aggregationMenu2.add(deleteAggregation2);

        // adding menu items to the menu
        rightClick2.add(showFormula2);
        rightClick2.add(conditionMenu2);
        rightClick2.add(operationsMenu2);
        rightClick2.add(groupingMenu2);
        rightClick2.add(aggregationMenu2);
        rightClick2.add(delete2);

        // adding popup menu to the left tree
        schemaJTree2.add(rightClick2);

        // making menu items
        final MenuItem deleteSubtree1 = new MenuItem("Delete Subtree");
        final MenuItem unDeleteSubtree1 = new MenuItem("Undelete Subtree");
        final MenuItem expandRecursion1 = new MenuItem("Expand Recursion");

        // adding listeners to menu items
        deleteSubtree1.addActionListener(popupAL);
        unDeleteSubtree1.addActionListener(popupAL);
        expandRecursion1.addActionListener(popupAL);

        // making menu items
        final MenuItem deleteSubtree2 = new MenuItem("Delete Subtree");
        final MenuItem unDeleteSubtree2 = new MenuItem("Undelete Subtree");
        final MenuItem expandRecursion2 = new MenuItem("Expand Recursion");
        MenuItem enterConstant = new MenuItem("Enter Constant Value");
        MenuItem showConstant = new MenuItem("Show Constant Value");
        MenuItem deleteConstant = new MenuItem("Clear Constant Value");

        // adding listeners to menu items
        deleteSubtree2.addActionListener(popupAL);
        unDeleteSubtree2.addActionListener(popupAL);
        expandRecursion2.addActionListener(popupAL);
        enterConstant.addActionListener(popupAL);
        showConstant.addActionListener(popupAL);
        deleteConstant.addActionListener(popupAL);

        final Menu constantMenu = new Menu("Constant");

        constantMenu.add(enterConstant);
        constantMenu.add(showConstant);
        constantMenu.add(deleteConstant);

        // add MouseListener to the left tree
        MouseListener ml1 = new MouseAdapter() {
                /* right button for options(delete, etc), left button for
                   get (x, y), then from parentComponent (splitPane),
                   tell which line is clicked, in splitPane,
                   remember all lines drawn in it
                */

                // on mouse click
                public void mouseClicked(MouseEvent me) {
                    // get coordinates of the click
                    int x = me.getX();
                    int y = me.getY();

                    MatchLine foundLine = null;

                    if (matchingLines.size() > 0) {
                        // find the closest match line
                        foundLine = findClosestLine(x -
                                (int) tree1SPane.getViewport().getViewPosition()
                                                .getX() +
                                (int) tree1SPane.getLocation().getX(),
                                y -
                                (int) tree1SPane.getViewport().getViewPosition()
                                                .getY() +
                                (int) tree1SPane.getLocation().getY());
                    }

                    leftSelectedNode = -1;

                    if (foundLine == null) {
                        leftSelectedNode = findClosestNode(schemaJTree1,
                                x -
                                (int) tree1SPane.getViewport().getViewPosition()
                                                .getX() +
                                (int) tree1SPane.getViewport().getLocation()
                                                .getX(),
                                y -
                                (int) tree1SPane.getViewport().getViewPosition()
                                                .getY() +
                                (int) tree1SPane.getViewport().getLocation()
                                                .getY());
                    }

                    // right mouse button is clicked
                    if (SwingUtilities.isRightMouseButton(me)) {
                        //System.err.println("right mouse button is clicked");
                        // Display the popup for the left node if over a node
                        if (leftSelectedNode != -1) {
                            PopupMenu popup = new PopupMenu();
                            boolean showPopup = false;

                            TNode node = (TNode) schemaJTree1.getLastSelectedPathComponent();

                            if (node != null) {
                                if ((node.children != null) &&
                                        (node.children.size() != 0)) {
                                    popup.add(deleteSubtree2);
                                    showPopup = true;
                                }

                                if ((node.deletedChildren != null) &&
                                        (node.deletedChildren.size() != 0)) {
                                    popup.add(unDeleteSubtree2);
                                    showPopup = true;
                                }

                                if (node.reference != null) {
                                    popup.add(expandRecursion2);
                                    showPopup = true;
                                }

                                schemaJTree1.add(popup);

                                if (showPopup) {
                                    popup.show(me.getComponent(), me.getX(),
                                        me.getY());
                                }
                            }

                            rightSelectedNode = -1;
                        }

                        // Only show popup for a line if the mouse is over the selected
                        // line
                        if ((foundLine != null) && (foundLine.selected == true)) {

			    if(foundLine.sameLines.size() == 0 && 
			       rightClick1.getItemCount() == 7) {
				rightClick1.remove(selectLine1);
			    }
			    else if(foundLine.sameLines.size() > 0 &&
				    rightClick1.getItemCount() == 6) {
				rightClick1.insert(selectLine1, 0);
			    }

			    rightClick1.show(me.getComponent(), 
					     me.getX(),
					     me.getY());
                        }
                    } else {
                        // any other button is clicked (such as left button)
                        //System.err.println("left mouse button is clicked");
                        if (selectedLine != null) {
                            selectedLine.selected = false;
                            selectedLine = null;
                        }

                        if (foundLine != null) {
                            selectedLine = foundLine;
                            selectedLine.selected = true;
                            arr.manualSelect("middle");
                        }
                        
                        if (leftSelectedNode != -1 && conditionNode != null) {
                         
                            int conditionRow = 
                                schemaJTree1.getRowForPath(conditionNode.getTreePath());

                            if(leftSelectedNode == conditionRow && 
                               sameNodeCondition == true) {
                                
                                schemaJTree1.clearSelection();
                                schemaJTree1.setSelectionRow(conditionRow);
                            }
                            else {
                                sameNodeCondition = true;
                            }
                        }
                    }

                    //remove existing lines
                    treesSplitPane.paintImmediately(0, 0,
                        treesSplitPane.getWidth(), treesSplitPane.getHeight());

                    treesSplitPane.drawMapping(treesSplitPane.getGraphics());
                }
            };

        // assign a mouse listener to the right tree
        MouseListener ml2 = new MouseAdapter() {
                /*right button for options(delete, etc), left button for
                  get (x, y), then from parentComponent (splitPane),
                  tell which line is clicked, in splitPane,
                  remember all lines drawn in it
                */

                // on mouse click
                public void mouseClicked(MouseEvent me) {
                    // get coordinates of the click
                    int x = me.getX();
                    int y = me.getY();

                    MatchLine foundLine = null;

                    if (matchingLines.size() > 0) {
                        // find the closest match line
                        foundLine = findClosestLine(x -
                                (int) tree2SPane.getViewport().getViewPosition()
                                                .getX() +
                                (int) tree2SPane.getLocation().getX(),
                                y -
                                (int) tree2SPane.getViewport().getViewPosition()
                                                .getY() +
                                (int) tree2SPane.getLocation().getY());
                    }

                    rightSelectedNode = -1;

                    if (foundLine == null) {
                        rightSelectedNode = findClosestNode(schemaJTree2,
                                x -
                                (int) tree2SPane.getViewport().getViewPosition()
                                                .getX() +
                                (int) tree2SPane.getViewport().getLocation()
                                                .getX(),
                                y -
                                (int) tree2SPane.getViewport().getViewPosition()
                                                .getY() +
                                (int) tree2SPane.getLocation().getY());
                    }

                    // right mouse button is clicked
                    if (SwingUtilities.isRightMouseButton(me)) {
                        // Display the popup for the right node if over a node
                        if (rightSelectedNode != -1) {
                            PopupMenu popup = new PopupMenu();

                            TNode node = (TNode) schemaJTree2.getLastSelectedPathComponent();

                            if (node != null) {
                                if ((node.children != null) &&
                                        (node.children.size() != 0)) {
                                    popup.add(deleteSubtree2);
                                }

                                if ((node.deletedChildren != null) &&
                                        (node.deletedChildren.size() != 0)) {
                                    popup.add(unDeleteSubtree2);
                                }

                                if (node.reference != null) {
                                    popup.add(expandRecursion2);
                                }

                                popup.add(constantMenu);

                                schemaJTree2.add(popup);
                                popup.show(me.getComponent(), me.getX(),
                                    me.getY());
                            }

                            leftSelectedNode = -1;
                        }

                        // Only show popup for a line if the mouse is over the selected
                        // line
                        if ((foundLine != null) && (foundLine.selected == true)) {

			    if(foundLine.sameLines.size() == 0 && 
			       rightClick2.getItemCount() == 7) {
				rightClick2.remove(selectLine1);
			    }
			    else if(foundLine.sameLines.size() > 0 &&
				    rightClick2.getItemCount() == 6) {
				rightClick2.insert(selectLine1, 0);
			    }

			    rightClick2.show(me.getComponent(), 
					     me.getX(),
					     me.getY());

                        }
                    } else {
                        // any other button is clicked (such as left button)
                        if (selectedLine != null) {
                            selectedLine.selected = false;
                            selectedLine = null;
                        }

                        if (foundLine != null) {
                            selectedLine = foundLine;
                            selectedLine.selected = true;
                            arr.manualSelect("middle");
                        }
                    }

                    //remove existing lines
                    treesSplitPane.paintImmediately(0, 0,
                        treesSplitPane.getWidth(), treesSplitPane.getHeight());

                    treesSplitPane.drawMapping(treesSplitPane.getGraphics());
                }
            };

        schemaJTree1.addMouseListener(ml1);
        schemaJTree2.addMouseListener(ml2);

        ActionListener a5 = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    if (SCIA.debug_on) {
                        System.err.println("Clearing matchings");
                    }

                    while (matchingLines.size() > 0) {
                        for (int i = 0; i < matchingLines.size(); i++) {
                            MatchLine mline = (MatchLine) matchingLines.get(i);

                            // delete the corresponding Matching
                            matchingLines.remove(mline);
                        }
                    }

                    function = new String();
                    multipleMatchings = new Vector();
                    matchingUnits = new ArrayList();

                    for (int i = 0; i < schemaJTree2.getRowCount(); i++) {
                        TreePath rightTreePath = schemaJTree2.getPathForRow(i);
                        TNode rightNode = (TNode) rightTreePath.getLastPathComponent();
                        rightNode.userInputMatch = false;
                        rightNode.fromComb = null;
                        rightNode.secFromComb = null;

                        if (rightNode.independentUnionMatches != null) {
                            rightNode.independentUnionMatches = null;
                        }

                        if (rightNode.dependentUnionMatches != null) {
                            rightNode.dependentUnionMatches = null;
                        }

                        rightNode.numOfPotentialMatches = 0;
                        rightNode.numOfMatches = 0;
                        rightNode.numOfIndependentUnionMatches = 0;

                        if (rightNode.matchings != null) {
                            rightNode.matchings.clear();
                        }

                        if (rightNode.lingMatchings != null) {
                            rightNode.lingMatchings.clear();
                        }

                        if (rightNode.matchingsFromCC != null) {
                            rightNode.matchingsFromCC.clear();
                        }
                    }

                    //remove existing lines
                    treesSplitPane.paintImmediately(0, 0,
                        treesSplitPane.getWidth(), treesSplitPane.getHeight());
                }
            };

        ActionListener a4 = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    try {
                        if ((schemaJTree1.schemaType == null) ||
                                (schemaJTree2.schemaType == null)) {
                            if (SCIA.debug_on) {
                                System.err.println("schemaType == null");
                            }
                        }

                        if (schemaJTree1.schemaType.equals("XMLSchema") &&
                                schemaJTree2.schemaType.equals("XMLSchema")) {
                            SCIA.bothXMLSchemas = true;
                        } else {
                            SCIA.bothXMLSchemas = false;
                        }

                        worker = new SwingWorker() {
                                    public Object construct() {
                                        try {
                                            // deselect the arrow
                                            arr.manualSelect("none");

                                            createMappingItem.setEnabled(false);

                                            Object result = SCIA.schemaTreeMatch(SCIA.interactive_mode,
                                                    SCIA.appDomain,
                                                    schemaJTree2, schemaJTree1,
                                                    SCIA.bothXMLSchemas,
                                                    SCIA.bothDirection);

                                            createMappingItem.setEnabled(true);

                                            return result;
                                        } catch (Exception e) {
                                            if (SCIA.debug_on) {
                                                System.err.println(
                                                    "From SchemaTreeMatch:");
                                            }

                                            e.printStackTrace();
                                            createMappingItem.setEnabled(true);

                                            return null;
                                        }
                                    }
                                };
                        worker.start();
                    } catch (Exception e) {
                        if (SCIA.debug_on) {
                            System.err.println("From match button action: " +
                                e);
                        }
                    }
                }
            };

        ActionListener aMD = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    try {
                        //write mapping into an XML file with structure 
                        //specified in scia_mapping.xsd
                        String xmlFileName = schemaJTree1.treeName + "_" +
                            schemaJTree2.treeName + "_map.xml";
                        PrintWriter pw1 = new PrintWriter(new FileOutputStream(SCIA_WORK_DIR +
                                    "/output/mappings/" + xmlFileName), true);
                        schemaJTree2.writeMapToXMLDoc(pw1,
                            schemaJTree1.treeName, SCIA.context_check,
                            SCIA.NUMBER_TOP_MATCHES_TO_OUTPUT);
                        pw1.flush();

                        // write mapping into a text file
                        String fileName = schemaJTree1.treeName + "_" +
                            schemaJTree2.treeName + ".map";
                        PrintWriter pw = new PrintWriter(new FileOutputStream(SCIA_WORK_DIR +
                                    "/output/mappings/" + fileName), true);

                        pw.println("Mapping: " + schemaJTree1.treeName +
                            " --> " + schemaJTree2.treeName);
                        pw.println();
                        schemaJTree2.writeMapToFile(pw);
                        pw.flush();
                    } catch (Exception e) {
                        if (SCIA.debug_on) {
                            System.err.println(
                                " error from writing mapping::: ");
                        }

                        e.printStackTrace();
                    }
                }
            };

        ActionListener aGV = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    try {
                        /* to do: browse to get the document file name of
                           the source schema
                        */

                        // get rid of /schemas/          
                        int i = schemaJTree1.treeName.lastIndexOf("/");
                        String sourceDoc = schemaJTree1.treeName.substring(i +
                                1) + ".xml";

                        SCIA.view = schemaJTree2.generateView(SCIA_WORK_DIR +
                                "/data/" + sourceDoc);

                        if (SCIA.debug_on) {
                            System.err.println("view = \n" + SCIA.view);
                        }

                        PrintWriter pw = new PrintWriter(new FileOutputStream(SCIA_WORK_DIR +
                                    "/output/queries/" + "view.qlt"), true);
                        pw.println(SCIA.view);
                        pw.flush();

                        JFrame f = new ViewEditor("View Editor",
                                "Please check and edit the " +
                                "generated view", SCIA.view);
                        f.show();
                    } catch (Exception e) {
                        if (SCIA.debug_on) {
                            System.err.println(" error from generateView()::: " +
                                e);
                        }

                        e.printStackTrace();
                    }
                }
            };

        ActionListener aCV = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    try {
                        JFrame f = new ViewEditor("View Editor",
                                "Please check and edit the " +
                                "view definition", SCIA.view);
                        f.show();
                    } catch (Exception e) {
                        if (SCIA.debug_on) {
                            System.err.println(" error from chekViewButton::: " +
                                e);
                        }
                    }
                }
            };

        ActionListener aEV = new ActionListener() {
                public void actionPerformed(ActionEvent ae) {
                    String line;
                    Runtime rt = Runtime.getRuntime();

                    String command = " java -cp " + "\"" + LIB_JAR_DIR +
                        "/kweelt.jar;" + LIB_JAR_DIR + "/xerces.jar;" +
                        LIB_JAR_DIR + "/jakarta-regexp-1.3.jar;.\" " +
                        "xacute.quilt.Main " + SCIA_WORK_DIR +
                        "/output/queries/view.qlt";

                    boolean wrong = false;
                    Process ps = null;
                    BufferedReader br;

                    try {
                        ps = rt.exec(command);

                        int seconds = 0;
                        boolean done = false;

                        while ((seconds <= 5) && !done) {
                            Thread.sleep(1000);
                            seconds++;

                            if (ps.getInputStream().available() >= 1) {
                                done = true;
                                br = new BufferedReader(new InputStreamReader(
                                            ps.getInputStream()));

                                String result = "";

                                while ((line = br.readLine()) != null) {
                                    result = result + "\n" + line;
                                }

                                if (SCIA.debug_on) {
                                    System.err.println("query result = " +
                                        result);
                                }

                                JFrame f = new ResultViewer("Query Result Viewer",
                                        "Please check the translated" +
                                        " document", result,
                                        schemaJTree2.treeName);
                                f.show();
                            } else {
                                if (ps.getErrorStream().available() >= 1) {
                                    done = true;

                                    String error = "";
                                    br = new BufferedReader(new InputStreamReader(
                                                ps.getErrorStream()));

                                    while ((line = br.readLine()) != null) {
                                        error = error + "\n" + line;
                                    }

                                    if (SCIA.debug_on) {
                                        System.err.println(error);
                                    }

                                    JFrame f = new ResultViewer("Error from view execution",
                                            "Please check", error, null);
                                    f.show();
                                }
                            }
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            };

        JMenuItem srcSchemaItem = new JMenuItem("Source");
        srcSchemaItem.addActionListener(a2);

        JMenuItem tgtSchemaItem = new JMenuItem("Target");
        tgtSchemaItem.addActionListener(a3);

        JMenu schemasMenu = new JMenu("Schemas");
        schemasMenu.add(srcSchemaItem);
        schemasMenu.add(tgtSchemaItem);

        createMappingItem = new JMenuItem("Create");
        createMappingItem.addActionListener(a4);

        JMenuItem openMappingItem = new JMenuItem("Open");
        openMappingItem.addActionListener(aRM);

        JMenuItem saveMappingItem = new JMenuItem("Save");
        saveMappingItem.addActionListener(aMD);

        JMenuItem clearMappingItem = new JMenuItem("Clear");
        clearMappingItem.addActionListener(a5);

        JMenu mappingMenu = new JMenu("Mapping");
        mappingMenu.add(createMappingItem);
        mappingMenu.add(openMappingItem);
        mappingMenu.add(saveMappingItem);
        mappingMenu.add(clearMappingItem);

        JMenuItem createViewItem = new JMenuItem("Create");
        createViewItem.addActionListener(aGV);

        JMenuItem checkViewItem = new JMenuItem("Check");
        checkViewItem.addActionListener(aCV);

        JMenuItem executeViewItem = new JMenuItem("Execute");
        executeViewItem.addActionListener(aEV);

        JMenu viewMenu = new JMenu("View");
        viewMenu.add(createViewItem);
        viewMenu.add(checkViewItem);
        viewMenu.add(executeViewItem);

        JMenuItem enterDomainItem = new JMenuItem("Input");

        JMenu applyDomainMenu = new JMenu("Apply");

        for (int applyDomainCntr = 0; applyDomainCntr < domainStrings.length;
                applyDomainCntr++) {
            JMenuItem applyItem = new JMenuItem(domainStrings[applyDomainCntr]);
            applyDomainMenu.add(applyItem);
            applyItem.addActionListener(a1);
        }

        JMenu domainMenu = new JMenu("Domain");
        domainMenu.add(enterDomainItem);
        domainMenu.add(applyDomainMenu);

        JCheckBoxMenuItem interactiveItem = new JCheckBoxMenuItem("Interactive",
                true);
        JCheckBoxMenuItem reuseMappingItem = new JCheckBoxMenuItem("Reuse Mapping",
                false);
        JCheckBoxMenuItem bothDirectionsItem = new JCheckBoxMenuItem("Both Directions",
                false);
        JCheckBoxMenuItem showDescriptionItem = new JCheckBoxMenuItem("Show Description",
                true);
        JCheckBoxMenuItem showCardinalityItem = new JCheckBoxMenuItem("Show Cardinality",
                true);
        JCheckBoxMenuItem showTypeItem = new JCheckBoxMenuItem("Show Type", true);
        JCheckBoxMenuItem showSimilarity = new JCheckBoxMenuItem("Show Similarity",
                true);

        interactiveItem.addActionListener(interactAL);
        reuseMappingItem.addActionListener(reuseBoxAL);
        bothDirectionsItem.addActionListener(directionBoxAL);
        showDescriptionItem.addActionListener(showDescriptionAL);
        showCardinalityItem.addActionListener(showCardinalityAL);
        showTypeItem.addActionListener(showTypeAL);
        showSimilarity.addActionListener(showSimilarityAL);

        JMenuItem configurationItem = new JMenuItem("Configuration");

        JMenu optionsMenu = new JMenu("Options");
        optionsMenu.add(domainMenu);
        optionsMenu.add(interactiveItem);
        optionsMenu.add(reuseMappingItem);
        optionsMenu.add(bothDirectionsItem);
        optionsMenu.add(showCardinalityItem);
        optionsMenu.add(showTypeItem);
        optionsMenu.add(showDescriptionItem);
        optionsMenu.add(showSimilarity);
        optionsMenu.add(configurationItem);

	JMenuItem functionsItem = new JMenuItem("Available Operations");
        JMenuItem aboutItem = new JMenuItem("About SCIA");
        JMenuItem guideItem = new JMenuItem("User Guide");

	functionsItem.addActionListener(functionsHelpAL);

        JMenu helpMenu = new JMenu("Help");
	helpMenu.add(functionsItem);
        helpMenu.add(aboutItem);
        helpMenu.add(guideItem);

        JMenuBar menuBar = new JMenuBar();
        menuBar.add(schemasMenu);
        menuBar.add(mappingMenu);
        menuBar.add(viewMenu);
        menuBar.add(optionsMenu);
        menuBar.add(helpMenu);

        
        JPanel schemaNamePanel = new JPanel(new BorderLayout());
        schemaNamePanel.add(sourceSchemaNameLabel,BorderLayout.WEST);
        schemaNamePanel.add(targetSchemaNameLabel,BorderLayout.EAST);

        JPanel arrowTreesPanel = new JPanel(new BorderLayout());
        arrowTreesPanel.add(menuBar, BorderLayout.NORTH);
        arrowTreesPanel.add(arr, BorderLayout.SOUTH);

        arr.setPreferredSize(new Dimension(600, 100));
        schemaNamePanel.setPreferredSize(new Dimension(700, 30));
        treesSplitPane.setPreferredSize(new Dimension(700, 600));

        JPanel schemaPanel = new JPanel(new BorderLayout());
        schemaPanel.add(schemaNamePanel, BorderLayout.NORTH);
        schemaPanel.add(treesSplitPane, BorderLayout.SOUTH);
        // construct the content pane

        contentPane.add(arrowTreesPanel, BorderLayout.NORTH);
        contentPane.add(schemaPanel, BorderLayout.SOUTH);

        setContentPane(contentPane);
    }

    /**
     * Find the node in the given tree which is the closest
     * to the mouse location given by X and Y.  Could tell
     * which node the user is clicking on
     *
     * @param tree which the user is clicking
     * @param X current mouse position in the X direction
     * @param Y current mouse position in the Y direction
     * @param xOffset tree offset in the X direction
     * @param yOffset tree offset in the Y direction
     *
     * @return the node row number
     *
     */
    public int findClosestNode(JTree tree, int X, int Y) {
        // number of rows currently displayed
        int rowCount = tree.getRowCount();

        for (int i = 1; i <= rowCount; i++) {
            Rectangle rec = tree.getRowBounds(i);

            if ((rec != null) && (X > rec.getX()) &&
                    (X < (rec.getX() + rec.getWidth())) && (Y > rec.getY()) &&
                    (Y < (rec.getY() + rec.getHeight()))) {
                return i;
            }
        }

        // if we could not find the node in the tree, return -1 
        return -1;
    }

    /**
     * Finds which line the user is trying to select
     * by getting the current mouse coordinates and comparing
     * them to the coordinates of the lines on the GUI
     *
     * @param X current mouse position in the X direction
     * @param Y current mouse position in the Y direction
     * @return the line the user has selected
     *
     */
    public MatchLine findClosestLine(int X, int Y) {
        // the left tree offset in the MyJSplitPane
        int x1Offset = (int) schemaJTree1.getLocation().getX() +
            (int) tree1SPane.getViewport().getLocation().getX() +
            (int) tree1SPane.getLocation().getX();
        int y1Offset = (int) schemaJTree1.getLocation().getY() +
            (int) tree1SPane.getViewport().getLocation().getY() +
            (int) tree1SPane.getLocation().getY();

        int x2Offset = (int) schemaJTree2.getLocation().getX() +
            (int) tree2SPane.getViewport().getLocation().getX() +
            (int) tree2SPane.getLocation().getX();
        int y2Offset = (int) schemaJTree2.getLocation().getY() +
            (int) tree2SPane.getViewport().getLocation().getY() +
            (int) tree2SPane.getLocation().getY();

        if (matchingLines.isEmpty()) {
            return null;
        }

        for (int i = (matchingLines.size() - 1); i >= 0; i--) {
            MatchLine mline = (MatchLine) matchingLines.get(i);

            if (mline == null) {
                return null;
            }

            TNode leftNode = mline.leftTNode;
            TNode rightNode = mline.rightTNode;

            // starting coordinates
            int x1 = 0;

            // starting coordinates
            int y1 = 0;

            if (leftNode == null && mline != null &&
		mline.focusPoint != null) {
                x1 = mline.focusPoint.x;
                y1 = mline.focusPoint.y;
            } else if(leftNode != null) {
                TreePath leftTreePath = leftNode.getTreePath();

                if (schemaJTree1.isVisible(leftTreePath)) {
                    int leftRow = schemaJTree1.getRowForPath(leftTreePath);

                    if (schemaJTree1.getRowBounds(leftRow) == null) {
                        return null;
                    }

                    //end point
                    x1 = (int) schemaJTree1.getRowBounds(leftRow).getX() +
                        (int) schemaJTree1.getRowBounds(leftRow).getWidth() +
                        x1Offset;
                    y1 = (int) schemaJTree1.getRowBounds(leftRow).getY() +
                        ((int) schemaJTree1.getRowBounds(leftRow).getHeight() / 2) +
                        y1Offset;
                } else {
		    x1 = y1 = 0;
		}
            }

            // ending coordinates
            int x2 = 0;

            // ending coordinates
            int y2 = 0;

            if ((x1 != 0) && (y1 != 0)) {
                TreePath rightTreePath = rightNode.getTreePath();
                int rightRow = schemaJTree2.getRowForPath(rightTreePath);

                if (schemaJTree2.getRowBounds(rightRow) == null) {
                    return null;
                }

                // ending point
                x2 = (int) schemaJTree2.getRowBounds(rightRow).getX() +
                    x2Offset;
                y2 = (int) schemaJTree2.getRowBounds(rightRow).getY() +
                    ((int) schemaJTree2.getRowBounds(rightRow).getHeight() / 2) +
                    y2Offset;
            }

            // use the two points to find the a and b in the
            // line equation (y = ax + b)
            double a = (double) (y1 - y2) / (double) (x1 - x2);
            double b = y1 - (a * x1);

            // check if the current point is on this line
            // and that it is between the nodes
            if ((Math.abs(((a * (double) X) + b) - (double) Y) < CLICKABLE_DISTANCE) &&
                    (X >= x1) && (X <= x2)) {
                return mline;
            }
        }

        return null;
    }

    /**
     * Refreshes the screen after the matching process is complete
     * by adding the match lines it finds in the right tree
     *
     */
    public void addMatchedLines() {
        matchingLines.clear();

        //System.err.println("inside addMatchedLines");
        for (int i = 0; i < schemaJTree2.getRowCount(); i++) {
            TreePath rightTreePath = schemaJTree2.getPathForRow(i);
            TNode rightNode = (TNode) rightTreePath.getLastPathComponent();

            if ((rightNode.independentUnionMatches != null) &&
                !rightNode.independentUnionMatches.isEmpty()) {

                for(int j = 0; 
                    j < rightNode.independentUnionMatches.size(); j++) {

                    Matching localMatching = 
                        (Matching)rightNode.independentUnionMatches.get(j);

                    addSingleMatchLine(localMatching, rightNode);
                }
            }

            if ((rightNode.dependentUnionMatches != null) &&
                !rightNode.dependentUnionMatches.isEmpty()) {

                for(int j = 0; 
                    j < rightNode.dependentUnionMatches.size(); j++) {

                    Matching localMatching = 
                        (Matching)rightNode.dependentUnionMatches.get(j);

                    addSingleMatchLine(localMatching, rightNode);                
                }
            }

            if (rightNode.fromComb != null) {
                
                addSingleMatchLine(rightNode.fromComb, rightNode);
            }
        }
 
        treesSplitPane.paintImmediately(0, 0, treesSplitPane.getWidth(),
                                        treesSplitPane.getHeight());
    }

    /**
     * Adds a line to the display for a right tree node.  This could
     * be a single line or a local multiple match line
     *
     * @param matching used by the method to get parameters necessary to 
     * draw a line on the screen
     * @param rightNode a Matching class contains only the left nodes for
     * a matching.  The right node needs to be passed in separately 
     */
    void addSingleMatchLine(Matching matching, TNode rightNode) {

        MatchLine matchLine;
        
        int index = matchingLines.size();
        
        if (matching.tnode != null) {
            TNode leftNode = matching.tnode;

            //System.err.println("single match found "+
            //rightNode.getPath()+"<-"
            //                   +leftNode.getPath());
            matchLine = new MatchLine(leftNode, rightNode, index,
                                      matching.sim);
	    findSameLines(matchLine);

        } else { // multiple matchings
            
            //System.err.println("local multiple matches found : "
            //+ rightNode.getPath()+ 
            //" = "+ matching.matchingUnits.size());
            multipleMatchings = new Vector();
            function = new String();
            
            int counter = 0;
            
            for (; counter < matching.matchingUnits.size();
                 counter++) {
                MatchingUnit mu = (MatchingUnit) (matching.matchingUnits.get(counter));
                multipleMatchings.add(mu.sourceNode);
                
                if ((mu.operations != null) &&
                    (mu.operations.length() > 0)) {
                    function = function + " " + "(" +
                        mu.sourceNode.getPath() + mu.operations + ")" +
                        " " + mu.opConnectingNextUnit;
                } else {
                    function = function + " " +
                        mu.sourceNode.getPath() + " " +
                        mu.opConnectingNextUnit;
                }
                
                //System.err.println("counter= " + counter +
                //       "source node = " + mu.sourceNode.getPath());
            }
            
            //System.err.println("multipleMatchings size = " + multipleMatchings.size());
            //System.err.println("multipleMatchings =" + multipleMatchings);
            //System.exit(0);
            function = function + " -> " + rightNode.getPath();
            
            matchLine = new MatchLine(multipleMatchings, rightNode,
                                      index, 1);
	    findSameLines(matchLine);
            
            matchLine.function = "Formula: " + function;
            
            if (SCIA.debug_on) {
                System.err.println("formula = " + function);
            }
        }
        
        if (matching.condition != null) {
            Condition cond = matching.condition;
            
            if ((cond.matchNeighborNode != null) &&
                (cond.workingNeighborNode != null)) {
                matchLine.condition = "Condition: " +
                    cond.matchNeighborNode.getPath() + " " +
                    cond.compareSymbol + " " +
                    cond.workingNeighborNode.getPath();
            }
        }
        
        if (matching.operations != null) {
            matchLine.operations = matching.operations;
        }
        
        matchingLines.add(matchLine);
    }

    /**
     * this method will prompt the user to enter a function
     * for multiple matches
     *
     * @param finalNode tells the method that this is the
     * final multiple match node
     *
     */
    public void promptForFormula(boolean finalNode) {
        TNode selectedTNode;

        if (finalNode == false) {
            // get the selected node from the left tree
            selectedTNode = (TNode) schemaJTree1.getLastSelectedPathComponent();
        } else {
            // get the selected node from the right tree
            selectedTNode = (TNode) schemaJTree2.getLastSelectedPathComponent();

	    if(putInDependent) {
		Matching newMatching = new Matching((TNode)null, 1.0);
		newMatching.matchingUnits = matchingUnits;
		selectedTNode.dependentUnionMatches.add(newMatching);
	    } else if(putInIndependent) {
		Matching newMatching = new Matching((TNode)null, 1.0);
		newMatching.matchingUnits = matchingUnits;
		selectedTNode.independentUnionMatches.add(newMatching);
	    } else {
		selectedTNode.fromComb.matchingUnits = matchingUnits;
	    }
        }

        // enter function for the next node
        Prompt pr1 = new Prompt("Enter the function on the selected node",
                                false, this, selectedTNode.getPath(), true, false);

        // enter the function to connect with the next value
        Prompt pr2 = new Prompt("Enter the function to connect with the next node",
                                false, this, selectedTNode.getPath(), false, false);

        if (finalNode == false) {
            arr.can.middleColor = Color.blue;

            // highlight the middle arrow
            arr.manualSelect("middle");

            pr2.show();
            pr1.show();
        } else {
            function = function + " = " + selectedTNode.getPath();
        }
    }

    /**
     * This method is the callback for the prompt GUI
     *
     * @param func the function entered
     * @param itself tells whether this is an operation
     * (function on itself) or function on the next
     * multiple match node
     * @param nodePath the path of the currently selected
     * node
     *
     */
    public void promptCallback(String func, boolean itself, String nodePath) {
        if (func == null) {
            return;
        }

        if (itself == true) {
            if ((selectedUnit != null) && !func.equals("")) {
                selectedUnit.operations = func;
            }

            function = function + " " + "(" + nodePath + " " + func + ")";
        } else {
            if ((selectedUnit != null) && !func.equals("")) {
                selectedUnit.opConnectingNextUnit = func;
            }

            function = function + " " + func;

            arr.manualSelect("left");
        }
    }

    /**
     * This method is the callback for the condition GUI
     *
     * @param cond condition entered
     * @param n1 first node for the condition
     * @param n2 second node for the condition
     * node
     *
     */
    public void conditionCallback(String cond, TNode n1, TNode n2) {

        if(n2 != null) {
            Condition c = new Condition(n1, cond, n2);

            if (conditionLine != null) {
                conditionLine.rightTNode.fromComb.condition = c;
                conditionLine.condition = n1.getPath() + " " + cond + " " +
                    n2.getPath();
            }
        }
        else {

            int symbLoc = cond.indexOf(' ');

            String symb, rightStr;

            if(symbLoc != -1) {
                symb = cond.substring(0, symbLoc);
                rightStr = cond.substring(symbLoc);
            }
            else {
                symb = cond;
                rightStr = "";
            }

            Condition c = new Condition(n1, symb, rightStr);

            if (conditionLine != null) {
                conditionLine.rightTNode.fromComb.condition = c;
                conditionLine.condition = n1.getPath() + " " + symb + " " +
                    rightStr;
            }
        }
    }

    /**
     * This method determines if the clicked right tree node
     * is already selected (global multiple match)
     *
     * @param X mouse location in the X direction
     * @param Y mouse location in the Y direction
     * @return tells whether the node is already selected
     *
     */
    public boolean isSameRightNode(int X, int Y) {
        if (rightSelectedTNode == null) {
            return false;
        }

        TreePath rightTreePath = rightSelectedTNode.getTreePath();
        int rightRow = schemaJTree2.getRowForPath(rightTreePath);

        // get node bounds
        int x1 = (int) schemaJTree2.getRowBounds(rightRow).getX();
        int y1 = (int) schemaJTree2.getRowBounds(rightRow).getY();
        int x2 = (int) schemaJTree2.getRowBounds(rightRow).getX() +
            (int) schemaJTree2.getRowBounds(rightRow).getWidth();
        int y2 = (int) schemaJTree2.getRowBounds(rightRow).getY() +
            (int) schemaJTree2.getRowBounds(rightRow).getHeight();

        //System.err.println("X = " + X + " Y = " + Y);
        // check if our coordinates are inside the right node
        if ((X >= x1) && (X <= x2) && (Y >= y1) && (Y <= y2)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Adds a dependent or independent union match by looking
     * at whether the ancestor has a union match 
     *
     */
    public void addUnionMatch() {

        // figure out whether the union match is dependent or independent
        if (rightSelectedTNode.ancestorHasUnionMatch()) {

            // this is a dependent union match
            if ((rightSelectedTNode.dependentUnionMatches == null ||
		 rightSelectedTNode.dependentUnionMatches.isEmpty()) && 
		(rightSelectedTNode.fromComb != null)) {

                unionMatches = new ArrayList();
                unionMatches.add(rightSelectedTNode.fromComb);
		rightSelectedTNode.dependentUnionMatches = unionMatches;
	    }

	    if(altPressed == true && multipleMatchings.size() > 0) {
		putInDependent = true;
	    } else {
		rightSelectedTNode.dependentUnionMatches.
		    add(new Matching(leftSelectedTNode, 1.0));
	    }

            if (SCIA.debug_on) {
                System.err.println("dependent union matches size: " +
                    rightSelectedTNode.dependentUnionMatches.size());
            }
        } else { 

            // this is an independent union match       
            if ((rightSelectedTNode.independentUnionMatches == null ||
		 rightSelectedTNode.independentUnionMatches.isEmpty()) && 
		(rightSelectedTNode.fromComb != null)) {

                unionMatches = new ArrayList();
                unionMatches.add(rightSelectedTNode.fromComb);
		rightSelectedTNode.independentUnionMatches = unionMatches;
	    }

	    if(altPressed == true && multipleMatchings.size() > 0) {
		putInIndependent = true;
	    } else {
		rightSelectedTNode.independentUnionMatches.
		    add(new Matching(leftSelectedTNode, 1.0));
	    }

            if (SCIA.debug_on) {
                System.err.println("independent union matches size: " +
                    rightSelectedTNode.independentUnionMatches.size());
            }
        }

        int index = matchingLines.size();
	MatchLine matchLine;

	if (altPressed == true && multipleMatchings.size() > 0) {

	    promptForFormula(true);

	    matchLine = new MatchLine(multipleMatchings,
				      rightSelectedTNode, index, 1);

	    findSameLines(matchLine);

	    matchLine.function = function;

	    function = new String();
	    multipleMatchings = new Vector();

	    rightSelectedTNode.fromComb.tnode = null;
	} else {
	    function = new String();
	    multipleMatchings = new Vector();
	    matchingUnits = new ArrayList();

	    matchLine = new MatchLine(leftSelectedTNode,
				      rightSelectedTNode, index, 1);

	    findSameLines(matchLine);
	}

        // add the new match to the vector of match lines
        matchingLines.add(matchLine);
        
	putInDependent = false;
	putInIndependent = false;

        schemaJTree1.clearSelection();
        schemaJTree2.clearSelection();     

	arr.manualSelect("right");

        treesSplitPane.paintImmediately(0, 0, treesSplitPane.getWidth(),
                                        treesSplitPane.getHeight());
        treesSplitPane.drawMapping(treesSplitPane.getGraphics());
    }

    /**
     * Enters the grouping attributes
     *
     */
    public void enterGroupAttrs() {
        worker = new SwingWorker() {
		public Object construct() {
		    try {
			MatchLine tmpLine = selectedLine;
			
			Interactive4 prompt = null;

			if (tmpLine != null) {
			    prompt = new Interactive4(
						"Please enter the grouping" +
					        " attributes from the left tree");
			} else {
			    prompt = new Interactive4("Select line first");
			}

			prompt.show();
			
			while (prompt.isVisible()) {
			    Thread.sleep(100);
			}
			
			int[] selectedRows = schemaJTree1.getSelectionRows();
			
			if ((selectedRows != null) &&
			    (tmpLine.rightTNode.fromComb != null)) {
			    tmpLine.rightTNode.fromComb.groupAttrs = 
				new Vector();
			    
			    for (int i = 0; i < selectedRows.length; i++) {
				TNode node = schemaJTree1.getTNodeFromRowNum
				    (selectedRows[i]);
				
				if (node != null) {
				    if (SCIA.debug_on) {
					System.err.println(
					       "Selected node is " + node.getTag());
				    }
				    tmpLine.rightTNode.fromComb.groupAttrs.
					add(node);
				    
				    //check if cross level grouping
				    TNode rightParent = tmpLine.rightTNode.
					parent;
				    if (rightParent != null && 
					rightParent.fromComb != null &&
					rightParent.fromComb.tnode == node  &&
					tmpLine.leftTNode == node.parent) {
					if (SCIA.debug_on) {
					    System.err.println(
						       "Cross Level Grouping!");
					}
					tmpLine.rightTNode.fromComb.
					    crossLevelGrouping = true;
					rightParent.fromComb.
					    isCrossLevelGroupAttr = true;
				    }
				} else {
				    if (SCIA.debug_on) {
					System.err.println(
						   "Didn't get the selected node");
				    }
				}
			    }
			}

			if (tmpLine.rightTNode.fromComb.groupAttrs != null) {
			    tmpLine.grouping = "\n";
			    
			    if (SCIA.debug_on) {
				System.err.println(tmpLine.rightTNode.
						   fromComb.groupAttrs.size());
			    }
			    
			    for (int i = 0;
				 i < tmpLine.rightTNode.fromComb.groupAttrs.
				     size(); i++) {
				tmpLine.grouping += (((TNode) tmpLine.
					       rightTNode.fromComb.groupAttrs.get(i)).
					       getPath() + "\n");
				
				if (SCIA.debug_on) {
				    System.err.println("grpattr = " +
					       ((TNode) tmpLine.rightTNode.fromComb.
					       groupAttrs.get(i)).getPath());
				}
			    }
			} else {
			    if (SCIA.debug_on) {
				System.err.println("grpattr = null");
			    }
			}

			selectedLine = tmpLine;

			return null;
		    } catch (Exception e) {
			if (SCIA.debug_on) {
			    System.err.println("Exception:");
			}

			e.printStackTrace();
			
			return null;
		    }
		}
	    };

        worker.start();
    }

    /**
     * This method finds whether there are more then
     * one match with the same left and right nodes
     *
     */
    public void findSameLines(MatchLine mline) {

	if(mline == null) {
	    return;
	}

	for(int i = 0; i < matchingLines.size(); i++) {

	    MatchLine tmpLine = (MatchLine)matchingLines.get(i);

	    if((mline.leftTNode != null && tmpLine.leftTNode != null &&
		mline.leftTNode == tmpLine.leftTNode &&
		mline.rightTNode == tmpLine.rightTNode) ||

	       (mline.leftTNode == null && tmpLine.leftTNode == null &&
		mline.rightTNode == tmpLine.rightTNode &&
		mline.mMatchings != null && tmpLine.mMatchings != null &&
		mline.mMatchings.containsAll(tmpLine.mMatchings) &&
		tmpLine.mMatchings.containsAll(mline.mMatchings))) {

		mline.sameLines.add(tmpLine);
		tmpLine.sameLines.add(mline);
		tmpLine.topLine = false;

		//Prompt to enter condition
		if(mline.leftTNode != null) {
		    conditionLine = mline;
		    conditionPrompt = new Prompt("Please select the first node",
						 true, null, null, false, false);
		    conditionPrompt.show();
		    firstCondition = true;
		}
	    }
	}
    }

    /**
     * This method find the formula for a match
     * to be displayed to the user
     *
     * @param mline currently selected match line
     * @return gives the formula
     */
    String getFormulaString(MatchLine mline) {

	String form = new String();

	// multiple matches
	if ((mline.function != null) &&
	    (mline.function.length() != 0)) {
	    form += mline.function;
	}
	// single match
	else {
	    form += mline.leftTNode.getPath();
	    
	    if ((mline.operations != null) &&
		(mline.operations.length() != 0)) {
		form += (" " + mline.operations);
	    }
	    
	    form += " -> ";
	    form += mline.rightTNode.getPath();
	}
	
	if ((mline.condition != null) &&
	    (mline.condition.length() != 0)) {
	    form += ("   Condition: " +
		     mline.condition);
	}
	
	if ((mline.grouping != null) &&
                                        (mline.grouping.length() != 0)) {
	    form += ("   Grouping Attributes: " +
		     mline.grouping);
	}	

	if ((mline.aggregation != null) &&
	    (mline.aggregation.length() != 0)) {
	    form += ("   Aggregation Operator: " +
		     mline.aggregation);
	}

	return form;
    }

    public static void main(String[] args) throws Exception {
        if (SCIA.debug_on) {
            System.err.println("SCIA_WORK_DIR = " + SCIA_WORK_DIR);
        }

        // create a frame
        MainUI mainFrame = new MainUI();
        mainFrame.pack();
        mainFrame.setVisible(true);

        //mainFrame.commandLineSchemaMatch(args);
        //String view = mainFrame.generateViewForObsAndStudyA();
        //String view = mainFrame.generateViewForLTEROverGCE();
        //String seqOverBSMLView = mainFrame.generateViewForSeqAndBSML();
        //mainFrame.generateViewExampleForBooks();

        /*PrintWriter pw =
            new PrintWriter(new FileOutputStream("view.qlt"), true);
        pw.println(view);
        pw.flush();*/
    }
}
