/**
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.scia;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/**
 * Morphism is an implementation for the interface SchemaMorphism.
 *
 * @author Guilian Wang
 */
public class Morphism implements SchemaMorphism {
    public AbstractSchema sourceSchema;
    public AbstractSchema targetSchema;
    public Set maps;

    public Morphism(AbstractSchema ss, AbstractSchema ts, Set m) {
        sourceSchema = ss;
        targetSchema = ts;
        maps = m;
    }

    public AbstractSchema getSourceSchema() {
        return sourceSchema;
    }

    public AbstractSchema getTargetSchema() {
        return targetSchema;
    }

    public Set getMaps() {
        return maps;
    }

    // if there are multiple views with conditions or not, return union of them?
    public String getView(Type type) {
        if ((maps == null) || maps.isEmpty()) {
            return null;
        }

        Iterator it = maps.iterator();

        while (it.hasNext()) {
            Mapping map = (Mapping) it.next();

            if (map.targetType == type) {
                return map.view;
            }
        }

        return null;
    }

    // how to represent conditions uniformly?
    public String getViewCondition(Type targetType) {
        return new String();
    }

    public String execute(String sourceDataFile) {
        /**********************************************************************
        % Routine Name : Morphism::execute(public)
        % File :         Morphism.java
        %
        % Description :  execute morphism to transform the input source data
        %                into an instance of the target shcema
        % Parameters descriptions :
        % -------------------------
        %   1. sourceDataFile -- source data file to be transformed into an
        %                instance of the target schema
        %
        **********************************************************************/
        String outputFile = null;

        //both source and target are xml
        if ((targetSchema.getSchemaType().equalsIgnoreCase("XMLSchema") ||
                targetSchema.getSchemaType().equalsIgnoreCase("DTD")) &&
                (sourceSchema.getSchemaType().equalsIgnoreCase("XMLSchema") ||
                sourceSchema.getSchemaType().equalsIgnoreCase("DTD"))) {
            Document doc = null;
            outputFile = "output.xml";

            try {
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                DocumentBuilder parser = factory.newDocumentBuilder();
                doc = parser.parse(sourceDataFile);

                //build target instance from the source document
                Type t = targetSchema.getTopType();
                Document outputDoc = parser.newDocument();
                Element root = outputDoc.createElement(t.typeName);

                ///to do in aother way 
                //doc.setRootElement(root);
                while (t != null) {
                    //simple types
                    if (t.isAttribute &&
                            (!t.typeType.equalsIgnoreCase("list") ||
                            !t.typeType.equalsIgnoreCase("product"))) {
                        Element elem = outputDoc.createElement(t.typeName);

                        Iterator it = getMaps().iterator();

                        while (it.hasNext()) {
                            Mapping map = (Mapping) it.next();

                            if (map.targetType == t) {
                                String view = map.getView();
                            }
                        }
                    }
                }
            } catch (Exception e) {
                if (SCIA.debug_on) {
                    System.err.println("Exception: " + e.getMessage());
                }
            }

            //printDOM(doc);
            writeXMLDocToFile(outputFile, doc);
        }

        return outputFile;
    }

    public void writeXMLDocToFile(String filename, Document document) {
        /**********************************************************************
        % Routine Name : Morphism::writeXMLDocToFile(public)
        % File :         Morphism.java
        %
        % Description :  write a DOM document to a file
        %
        % Parameters descriptions :
        % -------------------------
        %   1. filename -- the file to hold the document
        %   2. document -- the document to be written into a file
        **********************************************************************/
        try {
            // Prepare the DOM document for writing
            Source source = new DOMSource(document);

            // Prepare the output file
            File file = new File(filename);
            javax.xml.transform.Result result = new StreamResult(file);

            // Write the DOM document to the file
            // Get Transformer
            Transformer xformer = TransformerFactory.newInstance()
                                                    .newTransformer();

            // Write to a file
            xformer.transform(source, result);
        } catch (TransformerConfigurationException e) {
            if (SCIA.debug_on) {
                System.err.println("TransformerConfigurationException: " + e);
            }
        } catch (TransformerException e) {
            if (SCIA.debug_on) {
                System.err.println("TransformerException: " + e);
            }
        }
    }

    public Set listToSet(List l) {
        return new HashSet();
    }

    public List setToList(Set s) {
        return new ArrayList();
    }

    public void printDOM(Node node) {
        int type = node.getNodeType();

        switch (type) {
        // print the document element
        case Node.DOCUMENT_NODE: {
            if (SCIA.debug_on) {
                System.err.println("<?xml version=\"1.0\" ?>");
            }

            printDOM(((Document) node).getDocumentElement());

            break;
        }

        // print element with attributes
        case Node.ELEMENT_NODE: {
            if (SCIA.debug_on) {
                System.err.print("<");
            }

            if (SCIA.debug_on) {
                System.err.print(node.getNodeName());
            }

            NamedNodeMap attrs = node.getAttributes();

            for (int i = 0; i < attrs.getLength(); i++) {
                Node attr = attrs.item(i);

                if (SCIA.debug_on) {
                    System.err.print(" " + attr.getNodeName().trim() + "=\"" +
                        attr.getNodeValue().trim() + "\"");
                }
            }

            if (SCIA.debug_on) {
                System.err.println(">");
            }

            NodeList children = node.getChildNodes();

            if (children != null) {
                int len = children.getLength();

                for (int i = 0; i < len; i++)
                    printDOM(children.item(i));
            }

            break;
        }

        // handle entity reference nodes
        case Node.ENTITY_REFERENCE_NODE: {
            if (SCIA.debug_on) {
                System.err.print("&");
            }

            if (SCIA.debug_on) {
                System.err.print(node.getNodeName().trim());
            }

            if (SCIA.debug_on) {
                System.err.print(";");
            }

            break;
        }

        // print cdata sections
        case Node.CDATA_SECTION_NODE: {
            if (SCIA.debug_on) {
                System.err.print("");
            }

            break;
        }

        // print text
        case Node.TEXT_NODE: {
            if (SCIA.debug_on) {
                System.err.print(node.getNodeValue().trim());
            }

            break;
        }

        // print processing instruction
        case Node.PROCESSING_INSTRUCTION_NODE: {
            if (SCIA.debug_on) {
                System.err.print("");
            }

            break;
        }
        }

        if (type == Node.ELEMENT_NODE) {
            if (SCIA.debug_on) {
                System.err.println();
            }

            if (SCIA.debug_on) {
                System.err.print("");
            }
        }
    }

    public static void main(String[] args) {
        AbstractSchema s = new SchemaTree("book1.dtd", "DTD");
        AbstractSchema t = new SchemaTree("book3.dtd", "DTD");

        if (SCIA.debug_on) {
            System.err.println("s top type = " + s.getTopType().typeName);
        }

        if (SCIA.debug_on) {
            System.err.println("t top type = " + t.getTopType().typeName);
        }

        Set c = new HashSet();
        Mapping map = new Mapping();
        Type type = new Type();

        type = t.getTypeFromTag("bookstore");

        if (type == null) {
            if (SCIA.debug_on) {
                System.err.println("type bookstore == null");
            }
        }

        map = new Mapping(type, "bib");
        c.add(map);

        type = t.getTypeFromTag("book");

        if (type == null) {
            if (SCIA.debug_on) {
                System.err.println("type book == null");
            }
        }

        map = new Mapping(type, "book");
        c.add(map);

        type = t.getTypeFromTag("title");

        if (type == null) {
            if (SCIA.debug_on) {
                System.err.println("type title == null");
            }
        }

        map = new Mapping(type, "title");
        c.add(map);

        SchemaMorphism m = new Morphism(s, t, c);

        if (SCIA.debug_on) {
            System.err.println("Schema Morphism = ");
        }

        Iterator it = m.getMaps().iterator();

        while (it.hasNext()) {
            map = (Mapping) it.next();

            if (SCIA.debug_on) {
                System.err.println(map.targetType.typeName + " --> " +
                    map.view);
            }
        }

        String output = m.execute("book3.xml");

        if (output != null) {
            if (SCIA.debug_on) {
                System.err.println("output file = " + output);
            }
        }
    }
}
