/**
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.scia;

import java.awt.event.KeyEvent;
import java.io.PrintWriter;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import org.w3c.rdf.model.Model;
import org.w3c.rdf.model.Resource;

import com.interdataworking.mm.alg.MapPair;
import com.interdataworking.mm.alg.SFMatcher;
import com.interdataworking.mm.alg.StringMatcher;


/**
 * The SCIA internal representation of an schema element (attribute or field),
 * as a tree node in the relevant schema tree.
 * Each TNode holds a schema element's name, data type and matching results.
 * It also provides methods for manipulating especially matching schema tree
 * nodes.
 *
 * @author GuilianWang
 */
public class TNode implements TreeNode {
    public int keyEntered;
    public String dataEntered;
    public SchemaNode data;

    // for display in GUI, including compositor nodes
    public Vector children;
    public TNode parent;

    //for mapping, without compositor nodes
    public Vector realChildren;
    public TNode realParent;
    public Vector deletedChildren = new Vector();
    public SchemaTree tree;
    public TNode reference;

    // the row number in the JTree, each node displayed as a row
    public int rowNum;
    public Matching fromNode;
    public Matching fromPath;
    public Matching fromPurePath;
    public String constantValue;
    public String functionOutput;

    // the best from structural similarity through graph match
    public Matching fromGraph;

    // the best match from combination of name, path and structure
    public Matching fromComb;
    public Matching secFromComb; // the second best match

    // the best match from name and path similarity
    public Matching fromNameAndPath;
    public List matchings = new ArrayList();
    public List lingMatchings; // matching from only linguistic and type

    // matchings immediately from context check
    public List matchingsFromCC = new ArrayList();
    public boolean userInputMatch;
    public long matchedTimes;
    public long numOfPotentialMatches;
    public long numOfMatches; // num of user input matches for this node 
    public long subTreeSize; // num of nodes in the subtree rooted at this node
    public boolean hasAncestorWithUserInput;

    // has ancestor with multiple user input matching paths
    public boolean hasAncestorWithMultiUserInput;
    public long userInputAncestorNumMatches;
    public TNode ancestorWithUserInput;
    public short numOfIndependentUnionMatches;
    public List independentUnionMatches;

    // for each independent union match (a subtree) of a tnode's ancestor, 
    // this tnode has a list of candidate matches in that subtree 
    public List subTreeMatchesList; // order is in ancestor's 
                                    // independentUnionMatches list
    public List subTreeMatchings; // ?????????

    // the list of the best or final subtreeMatch in each subtree
    // length = length of ancestor's independentUnionMatches
    public List dependentUnionMatches;

    // root's height=0, for other nodes = 1 + height of parent
    public long height;
    DecimalFormat format = new DecimalFormat("#.##");

    public TNode() {
        // initialize the tree node content
        height = 0;
        userInputMatch = false;
        matchedTimes = 0;
        numOfPotentialMatches = 0;
    }

    public TNode(SchemaNode element) {
        // initialize the tree node content
        data = element;
        rowNum = 0;
        height = 0;
        userInputMatch = false;
        matchedTimes = 0;
        numOfPotentialMatches = 0;
    }

    public TNode(SchemaNode element, TNode parentNode) {
        // initialize the tree node content
        data = element;
        parent = parentNode;

        // attach this new TNode as one child of the node 
        if (parentNode.children == null) {
            parentNode.children = new Vector();
        }

        parentNode.children.addElement(this);

        if (parentNode.data.nodeType.equals("leafElem")) {
            parentNode.data.nodeType = "parentElem";
        }

        rowNum = 0;
        height = parentNode.height + 1;
        userInputMatch = false;
        matchedTimes = 0;
        numOfPotentialMatches = 0;
        tree = parentNode.tree;
    }

    //make a copy of  TNode as a child of 
    public TNode(TNode node, TNode newParent) {
        data = node.data;
        height = node.height;

        if (newParent != null) {
            parent = newParent;

            if (newParent.children == null) {
                newParent.children = new Vector();
            }

            newParent.children.add(this);
        }
    }

    public int getIndex(TreeNode node) {
        return rowNum;
    }

    public TreeNode getParent() {
        return parent;
    }

    public Enumeration children() {
        Enumeration e = children.elements();

        return e;
    }

    public boolean isLeaf() {
        return true;
    }

    public boolean getAllowsChildren() {
        return true;
    }

    public String getTag() {
        return data.getTag();
    }

    public String getDescription() {
        return data.getDescription();
    }

    public String getDataType() {
        return data.dataType;
    }

    public String getNodeType() {
        return data.nodeType;
    }

    public void getBestNodeForAll(TNode working1) {
        Matching bestRes = new Matching();
        getBestNode(working1, bestRes);
        this.fromNode = bestRes;

        //if (bestRes.tnode != null) {
        //    if(SCIA.debug_on) System.err.print(this.getPath() +": " );
        //    if(SCIA.debug_on) System.err.println(bestRes.tnode.getPath()+ 
        //                " sim = " + bestRes.sim);
        //}
        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.getBestNodeForAll(working1);
                }
            }
        }
    }

    public void getBestNode(TNode working1, // another dtd tree working node
        
    //double maxSim,  // max similarity got so far
    //TNode tnode, // best match so far
    Matching bestRes) // best res so far
     {
        StringMatcher sm = new StringMatcher();

        String path = this.getPath();
        String path1 = working1.getPath();

        String tag = this.data.getTag();
        String tag1 = working1.data.getTag();

        //double maxSim = 0.0;
        //TNode tnode = new TNode();
        double sim = sm.computeStringSimilarity(tag, tag1);

        //if(SCIA.debug_on) System.err.print("\nThe similarity is ");
        //if(SCIA.debug_on) System.err.print(sim);
        //String key = "(" +path + "," + path1 +")";
        //simHash.put((Object)key, (Object)(new Double (sim)));
        if (sim > bestRes.sim) {
            bestRes.sim = sim;
            bestRes.tnode = working1;
        }

        /*
        if(SCIA.debug_on) System.err.println("With " + tag1 + " sim = " + sim +
                           " maxSim = " + bestRes.sim);
        if (bestRes.tnode!= null){
            if(SCIA.debug_on) System.err.println("tnode path = " + bestRes.tnode.getPath());
        }
        */
        if ((working1.children != null) && !working1.children.isEmpty()) {
            for (int i = 0; i < working1.children.size(); i++) {
                getBestNode((TNode) working1.children.get(i), bestRes);
            }
        }
    }

    public void getFromNameAndPathForAll(TNode working1,
        Hashtable nameSimTable, //name sim table
        Hashtable pathSimTable) //path sim table
        throws Exception {
        Matching bestRes = new Matching();
        getFromNameAndPath(working1, bestRes, nameSimTable, pathSimTable);

        if (bestRes.sim > 1.0) {
            bestRes.sim = 1.0;
        }

        this.fromNameAndPath = bestRes;

        /*
        if (bestRes.tnode != null) {
            if(SCIA.debug_on) System.err.print(this.getPath() +": " );
            if(SCIA.debug_on) System.err.println(bestRes.tnode.getPath()+
                       " sim = " + bestRes.sim);
        }
        */
        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.getFromNameAndPathForAll(working1, nameSimTable,
                        pathSimTable);
                }
            }
        }
    }

    public void getFromNameAndPath(TNode working1, // another tree working node
        Matching bestRes, // best res so far
        Hashtable nameSimTable, //name similarity table
        Hashtable pathSimTable) //path similarity table
        throws Exception {
        if (data.nodeType.equals("compositor")) {
            return;
        }

        String path = this.getPath();

        if (!working1.data.nodeType.equals("compositor")) {
            String path1 = working1.getPath();

            String tag = this.data.getTag();
            String tag1 = working1.data.getTag();

            String key = "(" + path + "," + path1 + ")";
            double nsim = ((Double) nameSimTable.get(key)).doubleValue();
            double psim = ((Double) pathSimTable.get(key)).doubleValue();

            double lsim = (nsim * SCIA.NAME_WEIGHT_LINGUISTIC) +
                (psim * SCIA.PATH_WEIGHT_LINGUISTIC);

            /*
              if (path.equalsIgnoreCase("/ExcelOrder/InvoiceTo/Address/street1") &&
              (path1.endsWith("/street1") || path1.endsWith("/street4"))) {
              if(SCIA.debug_on) System.err.println();
              if(SCIA.debug_on) System.err.println("EO/Header/Add/street1 With " + path1 +
              " name sim = " + nsim +
              " path sim = " + psim +
              " linguistic sim = " + lsim +
              " maxSim = " + bestRes.sim);
              if(SCIA.debug_on) System.err.println();

              BufferedReader BR =
              new BufferedReader(new InputStreamReader(System.in));
              if(SCIA.debug_on) System.err.println("Please choose one option: go ahead (y/n)");
              char choice = (char)BR.read();
              if (choice == 'n') {
              System.exit(0);
              }
              }
            */
            /*
              if (path.equalsIgnoreCase("/ExcelOrder/Header/Address/street1") &&
              (path1.endsWith("/street") || path1.endsWith("/OrderDesc"))) {
              if(SCIA.debug_on) System.err.println();
              if(SCIA.debug_on) System.err.println("after increasing, EO/Header/OD With " + path1 +
              " linguistic sim = " + lsim);
              if(SCIA.debug_on) System.err.println();
              }
            */
            if (lsim > bestRes.sim) {
                bestRes.sim = lsim;
                bestRes.tnode = working1;
            }
        }

        if ((working1.children != null) && !working1.children.isEmpty()) {
            for (int i = 0; i < working1.children.size(); i++) {
                getFromNameAndPath((TNode) working1.children.get(i), bestRes,
                    nameSimTable, pathSimTable);
            }
        }
    }

    public void getBestPathForAll() {
        if (this.height >= 2) { // not root, not root's children

            if (this.fromNode != null) {
                if (this.fromNode.tnode != null) {
                    TNode best = this.fromNode.tnode;

                    if (this.parent.fromNode != null) {
                        if (this.parent.fromNode.tnode != null) {
                            TNode parentBest = this.parent.fromNode.tnode;

                            // check whether parent matched to any ancestor
                            if ((parentBest != null) &&
                                    !parentBest.isAncestorOf(best)) {
                                if (this.parent.fromNode.sim >= 0.8) {
                                    this.getBestPath(parentBest);
                                } else if (this.parent.parent.fromNode != null) {
                                    if (this.parent.parent.fromNode.tnode != null) {
                                        TNode grandBest = this.parent.parent.fromNode.tnode;

                                        if (this.parent.parent.fromNode.sim >= 0.8) {
                                            this.getBestPath(grandBest);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

                if (children != null) {
                    for (int i = 0; i < children.size(); i++) {
                        TNode child = (TNode) children.get(i);

                        if (child != null) {
                            child.getBestPathForAll();
                        }
                    }
                }
            }
        } else {
            if (children != null) {
                for (int i = 0; i < children.size(); i++) {
                    TNode child = (TNode) children.get(i);

                    if (child != null) {
                        child.getBestPathForAll();
                    }
                }
            }
        }
    }

    public void getBestPath(TNode subTreeRoot) {
        Matching res = new Matching();
        getBestNode(subTreeRoot, res);

        if (res.tnode != null) {
            this.fromPath = res;

            if (SCIA.debug_on) {
                System.err.println(this.getPath() + ": " +
                    this.fromPath.tnode.getPath() + "  sim = " +
                    this.fromPath.sim);
            }
        }
    }

    public void getBestPurePathForAll(TNode working1) {
        Matching bestRes = new Matching();
        getBestPurePath(working1, bestRes);
        this.fromPurePath = bestRes;

        if (bestRes.tnode != null) {
            if (SCIA.debug_on) {
                System.err.print(this.getPath() + ": ");
            }

            if (SCIA.debug_on) {
                System.err.println(bestRes.tnode.getPath() + " sim = " +
                    bestRes.sim);
            }
        }

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.getBestPurePathForAll(working1);
                }
            }
        }
    }

    public void getBestPurePath(TNode working1, // another dtd tree working node
        Matching bestRes) // best res so far
     {
        StringMatcher sm = new StringMatcher();

        String path = this.getPath();
        String path1 = working1.getPath();

        double sim = sm.computeStringSimilarity(path, path1);

        //if(SCIA.debug_on) System.err.print("\nThe similarity is ");
        //if(SCIA.debug_on) System.err.print(sim);
        //String key = "(" +path + "," + path1 +")";
        //simHash.put((Object)key, (Object)(new Double (sim)));
        if (sim > bestRes.sim) {
            bestRes.sim = sim;
            bestRes.tnode = working1;
        }

        /*
        if(SCIA.debug_on) System.err.println("With " + tag1 + " sim = " + sim +
                           " maxSim = " + bestRes.sim);
        if (bestRes.tnode!= null){
            if(SCIA.debug_on) System.err.println("tnode path = " + bestRes.tnode.getPath());
        }
        */
        if ((working1.children != null) && !working1.children.isEmpty()) {
            for (int i = 0; i < working1.children.size(); i++) {
                getBestPurePath((TNode) working1.children.get(i), bestRes);
            }
        }
    }

    public void getNameAndPathSimTablesForAll(Hashtable nameSimTable,
        Hashtable pathSimTable, TNode working1, Domain dm) {
        Matching bestFromName = new Matching();
        Matching bestFromPurePath = new Matching();
        getNameAndPathSimTables(nameSimTable, pathSimTable, working1,
            bestFromName, bestFromPurePath, dm);
        this.fromPurePath = bestFromPurePath;
        this.fromNode = bestFromName;

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.getNameAndPathSimTablesForAll(nameSimTable,
                        pathSimTable, working1, dm);
                }
            }
        }
    }

    public void getNameAndPathSimTables(Hashtable nameSimTable,
        Hashtable pathSimTable, 
    // another dtd tree working node
    TNode working1, Matching bestFromName, Matching bestFromPurePath, Domain dm) {
        /* Modification: at Oct.13, 2003, taking lower case string or name or path
           in order to be consistent with the normalization process that does not
           differentiate uppercase from lowercase, but the StringMatcher does care
           uppercase or lowercase
        */

        //if(SCIA.debug_on) System.err.println("getNameAndPathSimTables() for " + getTag());
        if (working1 == null) {
            if (SCIA.debug_on) {
                System.err.println("working1 == null");
            }

            return;
        }

        if (data.nodeType.equals("compositor")) {
            return;
        }

        String name = data.getTag();

        if (!working1.data.nodeType.equals("compositor")) {
            String name1 = working1.data.getTag();

            /* when using Rondo StringMatcher
               change to lower case to be uniform for thesauri hashtable key
               name = name.toLowerCase();
               name1 = name1.toLowerCase();
            */
            double nsim = 0; // name similarity
            double psim = 0; // path similarity
            String path = this.getPath();
            String path1 = working1.getPath();
            String normalizedPath1 = new String();

            //StringMatcher sm = new StringMatcher();
            String key = "(" + path + "," + path1 + ")";

            if (dm == null) {
                // no domain specified, then no thesauri provided
                // do name and path matching directly             

                /* using Rondo StringMatcher
                   nsim = sm.computeStringSimilarity(name.toLowerCase(),
                   name1.toLowerCase());
                   psim = sm.computeStringSimilarity(path.toLowerCase(),
                   path1.toLowerCase());
                */

                //use adjacent pair string comparison
                nsim = LetterPairSimilarity.compareStrings(name, name1);
                psim = LetterPairSimilarity.compareStrings(path, path1);
            } else {
                // use this domain's thesauri to normalize name and path
                // then do name and path matching
                Normalizer nm = new Normalizer(dm);

                if (nm.isSynonym(name, name1) || nm.isAbbrev(name, name1)) {
                    //if(SCIA.debug_on) System.err.println(name + " =~= " + name1);
                    nsim = 1.0;

                    // normalize path1, replace name1 with name
                    //normalizedPath1 = working1.parent.getPath() + "/" + name;
                    // modified at Oct.12, 2003
                    working1.data.setNormalizedTag(name);

                    //if(SCIA.debug_on) System.err.println(name1 + " changes to " + 
                    //working1.data.getNormalizedTag());
                    normalizedPath1 = working1.getNormalizedPath();

                    //if(SCIA.debug_on) System.err.println(path1 + " changes to " + normalizedPath1);
                    //System.exit(0);
                    // to do: set this node as normalized and give coeff = 1.0
                    /*psim = sm.computeStringSimilarity(path.toLowerCase(),
                      normalizedPath1.toLowerCase());
                    */
                    psim = LetterPairSimilarity.compareStrings(path,
                            normalizedPath1);
                } else if (nm.isHypernym(name, name1)) { //name > name1

                    //name1 --> name is small --> big or special --> general
                    nsim = 1.0;
                    working1.data.setNormalizedTag(name);
                    normalizedPath1 = working1.getNormalizedPath();

                    /*  psim = sm.computeStringSimilarity(path.toLowerCase(),
                        normalizedPath1.toLowerCase());
                    */
                    psim = LetterPairSimilarity.compareStrings(path,
                            normalizedPath1);
                } else if (nm.isHypernym(name1, name)) { //name < name1

                    // big --> small, sometimes it maybe wrong
                    nsim = SCIA.HYPERNYM_SIMILARITY;

                    // to do: set this node as normalized and give coeff = 0.8
                    // normalizedPath1 = working1.parent.getPath() + "/" + name;
                    // modified at Oct.12, 2003
                    working1.data.setNormalizedTag(name);
                    normalizedPath1 = working1.getNormalizedPath();

                    /*psim = SCIA.PATH_HYPERNYM_COEFF *
                      sm.computeStringSimilarity(path.toLowerCase(),
                      normalizedPath1.toLowerCase());
                    */
                    psim = SCIA.PATH_HYPERNYM_COEFF * LetterPairSimilarity.compareStrings(path,
                            normalizedPath1);
                } else {
                    // not found in thesauri, compute name and path sim directly

                    /*nsim = sm.computeStringSimilarity(name.toLowerCase(),
                      name1.toLowerCase());
                    */
                    nsim = LetterPairSimilarity.compareStrings(name, name1);
                    normalizedPath1 = working1.getNormalizedPath();

                    /*psim = sm.computeStringSimilarity(path.toLowerCase(),
                      normalizedPath1.toLowerCase());
                    */
                    psim = LetterPairSimilarity.compareStrings(path,
                            normalizedPath1);

                    int nameLen = name.length();
                    char nameLastChar = name.charAt(nameLen - 1);
                    int name1Len = name1.length();
                    char name1LastChar = name1.charAt(name1Len - 1);
                    Character test = new Character(nameLastChar);

                    if (test.isDigit(nameLastChar) &&
                            test.isDigit(name1LastChar)) {
                        if (nameLastChar != name1LastChar) {
                            nsim = nsim * SCIA.DIFF_LAST_DIGIT_COEFF;
                            psim = psim * SCIA.DIFF_LAST_DIGIT_COEFF;
                        }
                    }

                    // increase nsim and psim for nodes with the same data type 
                    if (SCIA.bothXMLSchemas) {
                        if ((data.dataType != null) &&
                                (working1.data.dataType != null) &&
                                data.dataType.equalsIgnoreCase(
                                    working1.data.dataType)) {
                            nsim = SCIA.SAME_DATA_TYPE_SIM + nsim;
                            psim = SCIA.SAME_DATA_TYPE_SIM + psim;
                        }
                    }
                }
            }

            /*
              if (path.endsWith("/Contact") &&
              path1.endsWith("/Contact")) {
              if(SCIA.debug_on) System.err.println();
              if(SCIA.debug_on) System.err.println(path + " with " + path1 +
              " normalized to" + normalizedPath1 +
              " name sim = " + nsim +
              " path sim = " + psim);
              if(SCIA.debug_on) System.err.println();
              }
            */
            /*
              psim = sm.computeStringSimilarity("/ExcelOrder/InvoiceTo/Contact",
              "/ParagonOrder/InvoiceTo/Contact");
              if(SCIA.debug_on) System.err.println("IT to IT, psim = " + psim );
              psim = sm.computeStringSimilarity("/ExcelOrder/InvoiceTo/Contact",
              "/ParagonOrder/invoiceto/Contact");
              if(SCIA.debug_on) System.err.println("IT to it, psim = " + psim );
              psim = sm.computeStringSimilarity("/excelorder/invoiceto/contact",
              "/paragonorder/invoiceto/contact");
              if(SCIA.debug_on) System.err.println("it to it, psim = " + psim );
              psim = sm.computeStringSimilarity("/ExcelOrder/InvoiceTo/Contact",
              "/ParagonOrder/ShipTo/Contact");
              if(SCIA.debug_on) System.err.println("IT to ST, psim = " + psim );
              psim = sm.computeStringSimilarity("/ExcelOrder/InvoiceTo/Contact",
              "/ParagonOrder/ShipTo/Contact");
              if(SCIA.debug_on) System.err.println("IT to st, psim = " + psim );
              psim = sm.computeStringSimilarity("/excelorder/invoiceto/contact",
              "/paragonorder/shipto/contact");
              if(SCIA.debug_on) System.err.println("it to st, psim = " + psim );
            */
            /* to do: check StringMatcher, why return sim value > 1.0 for two paths like
               "/Bsml/Definitions/Sequences/Sequence/Seq-data", and
               "/Sequences/Sequence/Seq-data"

               then remove the following if
            */
            //if (psim > 1.0) {
            //    psim = 1.0;
            //}
            nameSimTable.put((Object) key, (Object) (new Double(nsim)));
            pathSimTable.put((Object) key, (Object) (new Double(psim)));

            if ((bestFromName == null) || (bestFromPurePath == null)) {
                if (SCIA.debug_on) {
                    System.err.println(
                        "bestFromName or BestfromPurePath == null");
                }
            }

            if (nsim > bestFromName.sim) {
                bestFromName.sim = nsim;
                bestFromName.tnode = working1;
            }

            if (psim > bestFromPurePath.sim) {
                bestFromPurePath.sim = psim;
                bestFromPurePath.tnode = working1;
            }
        }

        if ((working1.children != null) && !working1.children.isEmpty()) {
            for (int i = 0; i < working1.children.size(); i++) {
                getNameAndPathSimTables(nameSimTable, pathSimTable,
                    (TNode) working1.children.get(i), bestFromName,
                    bestFromPurePath, dm);
            }
        }
    }

    public Matching bestMatchingPath(SchemaTree localTree, TNode localTreeRoot) {
        TNode working = this;
        String workingPath = getPath();
        String localPath;
        Vector localLeaves = new Vector();
        StringMatcher sm = new StringMatcher();
        String masterTag;
        String localTag;
        String masterParentTag;
        String localParentTag;
        String masterGrandParentTag;
        String localGrandParentTag;

        double maxSim = 0.0;
        TNode tnode = new TNode();
        double sim = 0;

        localTree.getLeaves(localLeaves, localTreeRoot);
        masterTag = working.data.getTag();

        if ((localLeaves != null) && !localLeaves.isEmpty()) {
            for (int i = 0; i < localLeaves.size(); i++) {
                TNode localWorking = (TNode) localLeaves.get(i);
                localPath = localWorking.getPath();

                //String key = "(" + workingPath + "," + localPath + ")";
                localTag = localWorking.data.getTag();

                sim = sm.computeStringSimilarity(masterTag, localTag);

                //if(SCIA.debug_on) System.err.println("with " + localTag + " sim = " + sim);
                if ((working.parent != null) && (localWorking.parent != null)) {
                    masterParentTag = working.parent.data.getTag();
                    localParentTag = localWorking.parent.data.getTag();
                    sim = sim +
                        sm.computeStringSimilarity(masterParentTag,
                            localParentTag);
                }

                if ((working.parent.parent != null) &&
                        (localWorking.parent.parent != null)) {
                    masterGrandParentTag = working.parent.parent.data.getTag();
                    localGrandParentTag = localWorking.parent.parent.data.
                        getTag();
                    sim = sim + sm.computeStringSimilarity
                        (masterGrandParentTag, localGrandParentTag);
                }

                if (sim > maxSim) {
                    maxSim = sim;
                    tnode = localWorking;
                }
            }
        }

        return new Matching(tnode, maxSim);
    }

    /**
     * Determine whether this TNode has an ancestor that has an independent 
     * union match.
     */
    public boolean ancestorHasUnionMatch() {
        boolean res = false;
        if (parent != null) {
            if (parent.independentUnionMatches != null) {
                res = true;
            } else {
                res = parent.ancestorHasUnionMatch();
            }
        }
        return res;
    }

    public boolean isAncestorOf(TNode working) {
        boolean res = false;

        if (working!= null && working.parent != null) {
            if (this == working.parent) {
                res = true;

                return res;
            } else {
                res = isAncestorOf(working.parent);
            }
        }

        return res;
    }

    public boolean isDecendentOf(TNode working) {
        boolean res = false;

        if ((parent != null) && (working != null)) {
            if (parent == working) {
                res = true;
            } else if (parent.parent != null) {
                res = parent.isDecendentOf(working);
            }
        } else {
            res = false;
        }

        return res;
    }

    /*
    public String getPath() {

        String path = new String();
        if (data != null) {
            if (data.nodeType.equals("attribute")) {
                path = "/@" + data.getTag();
            } else {
                path = "/" + data.getTag();
            }
            if (parent != null) { // not root
                if (parent.parent != null) { // not root's child
                    path = parent.getPath() + path;
                } else {
                    path = "/" + parent.data.getTag() + path;
                }
            }
        }
        return path;
    }
    */
    public String getPath() {
        String path = new String();

        if (data != null) {
            if (data.nodeType.equals("attribute")) {
                path = "/@" + data.getTag();
            } else if (!data.nodeType.equals("compositor")) {
                path = "/" + data.getTag();
            }

            if (parent != null) { // not root

                if (parent.parent != null) { // not root's child
                    path = parent.getPath() + path;
                } else {
                    path = "/" + parent.data.getTag() + path;
                }
            }
        }

        return path;
    }

    public String getNormalizedPath() {
        //if (nodeType == "attribute")
        String path = new String();

        if (data != null) {
            if (!data.nodeType.equals("compositor")) {
                path = "/" + data.getNormalizedTag();
            }

            if (parent != null) { // not root

                if (parent.parent != null) { // not root's child
                    path = parent.getNormalizedPath() + path;
                } else { // root's child
                    path = "/" + parent.data.getNormalizedTag() + path;
                }
            }
        }

        //if(SCIA.debug_on) System.err.println("******" + getTag() + " normalized = " + path);
        return path;
    }

    // compute the path from an ancestor node to this node
    public String getPath(TNode working) {
        if (working == null) {
            if (SCIA.debug_on) {
                System.err.println("In getPath(TNode), input TNode == null");
            }
        }

        String path = new String();

        // if the working node is not an ancestor of this node, there is no path
        if (!isDecendentOf(working)) {
            if (SCIA.debug_on) {
                System.err.println(getPath() + " isn't " + working.getPath() +
                    "'s decendent");
            }

            return null;
        } else if (data != null) {
            if (data.nodeType.equals("attribute")) {
                path = "/@" + data.getTag();
            } else if (!data.nodeType.equals("compositor")) {
                path = "/" + data.getTag();
            }

            if ((parent != null) && (parent != working)) { // working is not parent of this node
                path = parent.getPath(working) + path;
            }

            /*
            if (getTag().equals("author")) {
                if(SCIA.debug_on) System.err.println(getPath() + " is " +
                working.getPath() + "'s decendent, \n path between = " + path);
                System.exit(0);
            }
            */
        }

        return path;
    }

    public TreePath getTreePath() {
        //if (nodeType == "attribute")
        TreePath treePath = new TreePath(this);

        if (parent != null) { // not root
            treePath = (parent.getTreePath()).pathByAddingChild(this);
        }

        return treePath;
    }

    public void toRDFModelForAll(Resource parentRe, Model model,
        Hashtable resHash) {
        try {
            String tag = getTag();

            if ((data.nodeType != null) && !data.nodeType.equals("compositor")) {
                Resource re = toRDFModel(parentRe, model, resHash);

                if (children != null) {
                    for (int i = 0; i < children.size(); i++) {
                        TNode child = (TNode) children.get(i);

                        if (child != null) {
                            child.toRDFModelForAll(re, model, resHash);
                        }
                    }
                }
            } else { //skip compositor nodes

                if (children != null) {
                    for (int i = 0; i < children.size(); i++) {
                        TNode child = (TNode) children.get(i);

                        if (child != null) {
                            child.toRDFModelForAll(parentRe, model, resHash);
                        }
                    }
                }
            }
        } catch (java.lang.Exception e) {
            if (SCIA.debug_on) {
                System.err.println("From TNode.toRDFModel():");
            }

            e.printStackTrace();
        }
    }

    public Resource toRDFModel(Resource parentRe, Model model, Hashtable resHash)
        throws Exception {
        //if(SCIA.debug_on) System.err.println("in TNode: toRDFModel top dealing with " + getPath());
        String path = this.getPath();
        Resource re = SchemaTree.nf.createResource(path);
        resHash.put(path, re);

        if (parentRe != null) {
            model.add(SchemaTree.nf.createStatement(re,
                    SchemaTree.parent[(int) height], parentRe));
            model.add(SchemaTree.nf.createStatement(parentRe,
                    SchemaTree.child[(int) height], re));
        }

        if (data.nodeType == "rootElem") {
            // connect to rootElem node
            model.add(SchemaTree.nf.createStatement(re, SchemaTree.nodeType,
                    SchemaTree.rootElem));

            // also connect to parentElem node
            model.add(SchemaTree.nf.createStatement(re, SchemaTree.nodeType,
                    SchemaTree.parentElem));
        } else if (data.nodeType == "parentElem") {
            model.add(SchemaTree.nf.createStatement(re, SchemaTree.nodeType,
                    SchemaTree.parentElem));
        } else if (data.nodeType == "leafElem") {
            model.add(SchemaTree.nf.createStatement(re, SchemaTree.nodeType,
                    SchemaTree.leafElem));
        } else if (data.nodeType == "attribute") {
            model.add(SchemaTree.nf.createStatement(re, SchemaTree.nodeType,
                    SchemaTree.attr));

            //also connect to leafElem, since leafElem maybe matched to attribute
            model.add(SchemaTree.nf.createStatement(re, SchemaTree.nodeType,
                    SchemaTree.leafElem));
        }

        if (data.dataType != null) {
            if (data.dataType == "CDATA") {
                model.add(SchemaTree.nf.createStatement(re,
                        SchemaTree.dataType, SchemaTree.CDATA));
            } else if (data.dataType == "PCDATA") {
                model.add(SchemaTree.nf.createStatement(re,
                        SchemaTree.dataType, SchemaTree.PCDATA));
            } else if (data.dataType == "ID") {
                model.add(SchemaTree.nf.createStatement(re,
                        SchemaTree.dataType, SchemaTree.ID));
            } else if (data.dataType.equalsIgnoreCase("String")) {
                model.add(SchemaTree.nf.createStatement(re,
                        SchemaTree.dataType, SchemaTree.STRING));
            } else if (data.dataType.equalsIgnoreCase("Int") ||
                    data.dataType.equalsIgnoreCase("Integer")) {
                model.add(SchemaTree.nf.createStatement(re,
                        SchemaTree.dataType, SchemaTree.INT));
            } else if (data.dataType.equalsIgnoreCase("Date")) {
                model.add(SchemaTree.nf.createStatement(re,
                        SchemaTree.dataType, SchemaTree.DATE));
            } else if (data.dataType.equalsIgnoreCase("Float")) {
                model.add(SchemaTree.nf.createStatement(re,
                        SchemaTree.dataType, SchemaTree.FLOAT));
            } else if (data.dataType.equalsIgnoreCase("Number")) {
                model.add(SchemaTree.nf.createStatement(re,
                        SchemaTree.dataType, SchemaTree.NUMBER));
            }
        }

        //if(SCIA.debug_on) System.err.println("leaving TNode: toRDFModel()");
        return re;
    }

    public void getInitMapForAll(Hashtable thisResHash, Hashtable resHash,
        List initMap) throws Exception {
        getInitMap(thisResHash, resHash, initMap);

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.getInitMapForAll(thisResHash, resHash, initMap);
                }
            }
        }
    }

    public void getInitMap(Hashtable thisResHash, Hashtable resHash,
        List initMap) throws Exception {
        String path1 = getPath();
        Resource r1 = (Resource) thisResHash.get(path1);

        if ((fromPath != null) && (fromPath.tnode != null) &&
                (fromPath.sim >= SCIA.LINGUISTIC_THRESHOLD)) {
            String path2 = fromPath.tnode.getPath();
            Resource r2 = (Resource) resHash.get(path2);
            MapPair map = new MapPair(r1, r2, fromPath.sim);
            initMap.add(map);

            return;
        } else if ((fromNode != null) && (fromNode.tnode != null) &&
                (fromNode.sim >= SCIA.LINGUISTIC_THRESHOLD)) {
            String path2 = fromNode.tnode.getPath();
            Resource r2 = (Resource) resHash.get(path2);
            MapPair map = new MapPair(r1, r2, fromNode.sim);
            initMap.add(map);
        }
    }

    public void getInitMapFromLinguisticForAll(Hashtable thisResHash,
        Hashtable resHash, List initMap) throws Exception {
        getInitMapFromLinguistic(thisResHash, resHash, initMap);

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.getInitMapFromLinguisticForAll(thisResHash, resHash,
                        initMap);
                }
            }
        }
    }

    public void getInitMapFromLinguistic(Hashtable thisResHash,
        Hashtable resHash, List initMap) throws Exception {
        String path1 = getPath();
        Resource r1 = (Resource) thisResHash.get(path1);

        // take of user input match, possibly multiple matches, first
        if (userInputMatch) {
            if ((matchings != null) && !matchings.isEmpty()) {
                for (int i = 0; i < matchings.size(); i++) {
                    /*
                    if (i > numOfMatches) {
                        String message = "input matchings size > numOfMatches \n" +
                            "Go ahead?";
                        final Interactive inter = new Interactive(message, keyEntered);

                        inter.show();

                        tree.mainUI.treesSplitPane.paintImmediately
                            (0,
                             0,
                             tree.mainUI.treesSplitPane.getWidth(),
                             tree.mainUI.treesSplitPane.getHeight());
                        tree.mainUI.treesSplitPane.
                            drawMapping(tree.mainUI.treesSplitPane.
                                        getGraphics());

                        // wait for the option to close
                        while(inter.isVisible()) {
                            Thread.sleep(100);}

                        keyEntered = inter.keyEntered;

                        if (keyEntered == KeyEvent.VK_N) {
                            //System.exit(0);
                            return;
                        }
                    }
                    */
                    Matching inputMatch = (Matching) matchings.get(i);
                    String path2 = inputMatch.tnode.getPath();
                    Resource r2 = (Resource) resHash.get(path2);

                    // make sure the matched path is in the corresponding subtree
                    if (r2 != null) {
                        MapPair map = new MapPair(r1, r2, inputMatch.sim);
                        initMap.add(map);
                    }
                }
            }

            return;
        }

        if ((fromNameAndPath != null) && (fromNameAndPath.tnode != null) &&
                (fromNameAndPath.sim >= SCIA.LINGUISTIC_THRESHOLD)) {
            String path2 = fromNameAndPath.tnode.getPath();
            Resource r2 = (Resource) resHash.get(path2);
            MapPair map = new MapPair(r1, r2, fromNameAndPath.sim);
            initMap.add(map);
        }
    }

    public void getInitMapFromCombForAll(Hashtable thisResHash,
        Hashtable resHash, List initMap, TNode subTreeRoot)
        throws Exception {
        getInitMapFromComb(thisResHash, resHash, initMap, subTreeRoot);

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.getInitMapFromCombForAll(thisResHash, resHash,
                        initMap, subTreeRoot);
                }
            }
        }
    }

    public void getInitMapFromComb(Hashtable thisResHash, Hashtable resHash,
        List initMap, TNode subTreeRoot) throws Exception {
        // sort the matchings of this TNode first
        if ((matchings == null) || matchings.isEmpty()) {
            if (SCIA.debug_on) {
                System.err.println(getPath() +
                    "'s matchings is null or empty, can not get initMapFromComb for it");
            }

            return;
        }

        sortMatchings(matchings);

        String path1 = getPath();
        Resource r1 = (Resource) thisResHash.get(path1);
        Matching tmpMatch = new Matching();
        TNode tmpTNode = new TNode();

        for (int i = 0; i < matchings.size(); i++) {
            tmpMatch = (Matching) matchings.get(i);
            tmpTNode = tmpMatch.tnode;

            if (data.getTag().equalsIgnoreCase("segment")) {
                if (SCIA.debug_on) {
                    System.err.println("segment's matching =" +
                        tmpTNode.getPath() + "sim = " + tmpMatch.sim);
                }
            }

            if ((tmpTNode == subTreeRoot) ||
                    tmpTNode.isDecendentOf(subTreeRoot)) {
                // found the matching tnode in subtree with higher similarity
                break;
            }
        }

        if (tmpMatch.sim > SCIA.SUBTREE_COMB_THRESHOLD) {
            String path2 = tmpTNode.getPath();
            Resource r2 = (Resource) resHash.get(path2);
            MapPair map = new MapPair(r1, r2, tmpMatch.sim);
            initMap.add(map);
        }

        return;
    }

    public TNode getTNodeFromPathForAll(String path) {
        //if(SCIA.debug_on) System.err.println("in for all");
        TNode tnode = getTNodeFromPath(path);

        if (tnode == null) {
            if (children != null) {
                for (int i = 0; i < children.size(); i++) {
                    TNode child = (TNode) children.get(i);

                    if (child != null) {
                        tnode = child.getTNodeFromPathForAll(path);

                        if (tnode != null) {
                            return tnode;
                        }
                    }
                }
            }
        }

        return tnode;
    }

    public TNode getTNodeFromPath(String path) {
        //if(SCIA.debug_on) System.err.println("in getTNodeFromPath");
        if (getPath().equals(path)) {
            TNode tnode = this;

            //if(SCIA.debug_on) System.err.println("this node's path = " + getPath() +
            //                         " tnode's path = " + tnode.getPath());
            return tnode;
        }

        return null;
    }

    /**
     * Add this schema tree node's existing match and all decendent nodes' matches
     * with higher similarity values into
     * the input List goodInputMap for reuse in the further matching process.
     *
     * @param thisResHash The Hashtable for retrieving this node's corresponding
     * resource from its path
     * @param reHash The Hashtable for retrieving this node's matching nodes'
     * resources from their paths
     * @goodInputMap The list of good matches to reuse
     */
    public void addGoodInputMapForAll(Hashtable thisResHash, Hashtable resHash,
        List goodInputMap) {
        addGoodInputMap(thisResHash, resHash, goodInputMap);

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.addGoodInputMapForAll(thisResHash, resHash,
                        goodInputMap);
                }
            }
        }
    }

    /**
     * Add this schema tree node's existing local n-to-1 matches
     * with higher similarity values into
     * the input List goodInputMap for reuse in the further matching process.
     *
     * @param match The match of this schema tree node to add
     * @param r1  The resource for this node
     * @param reHash The Hashtable for retrieving this node's matching nodes'
     * resources from their paths
     * @goodInputMap The list of good matches to reuse
     */
    public void addGoodInputMap_local_n_to_1(Matching match, Resource r1,
        Hashtable resHash, List goodInputMap) {
        for (int i = 0; i < match.matchingUnits.size(); i++) {
            MatchingUnit unit = (MatchingUnit) match.matchingUnits.get(i);
            String path2 = unit.sourceNode.getPath();
            Resource r2 = (Resource) resHash.get(path2);
            MapPair map = new MapPair(r1, r2, match.sim);
            goodInputMap.add(map);
        }
    }

    /**
     * Add this schema tree node's existing local n-to-1 matches
     * with higher similarity values into
     * the input List goodInputMap for reuse in the further matching process.
     *
     * @param match The match of this schema tree node to add
     * @param r1  The resource for this node
     * @param reHash The Hashtable for retrieving this node's matching nodes'
     * resources from their paths
     * @goodInputMap The list of good matches to reuse
     */
    public void addGoodInputMap_1_to_1(Matching match, Resource r1,
        Hashtable resHash, List goodInputMap) {
        String path2 = match.tnode.getPath();
        Resource r2 = (Resource) resHash.get(path2);
        MapPair map = new MapPair(r1, r2, match.sim);
        goodInputMap.add(map);
    }

    /**
     * Add this schema tree node's existing match with higher similarity values into
     * the input List goodInputMap for reuse in the further matching process.
     *
     * @param thisResHash The Hashtable for retrieving this node's corresponding
     * resource from its path
     * @param reHash The Hashtable for retrieving this node's matching nodes'
     * resources from their paths
     * @goodInputMap The list of good matches to reuse
     */
    public void addGoodInputMap(Hashtable thisResHash, Hashtable resHash,
        List goodInputMap) {
        String path1 = getPath();
        Resource r1 = (Resource) thisResHash.get(path1);

        if ((fromComb != null) && (fromComb.sim > SCIA.REUSE_SIM_THRESHOLD)) {
            // Set this schema tree node as user input match, further matching process
            // won't change its matches, though using it to help with neighbors
            userInputMatch = true;

            // Add one-to-one match
            if (fromComb.tnode != null) {
                addGoodInputMap_1_to_1(fromComb, r1, resHash, goodInputMap);

                // Add local n-to-1 matches
            } else if ((fromComb.matchingUnits != null) &&
                    !fromComb.matchingUnits.isEmpty()) {
                addGoodInputMap_local_n_to_1(fromComb, r1, resHash, goodInputMap);
            }
        }

        // Add global n-to-1 matches                
        List unionMatches = null;

        if ((independentUnionMatches != null) &&
                !independentUnionMatches.isEmpty()) {
            unionMatches = dependentUnionMatches;
        } else if ((dependentUnionMatches != null) &&
                !dependentUnionMatches.isEmpty()) {
            unionMatches = dependentUnionMatches;
        }

        if (unionMatches != null) {
            for (int i = 0; i < unionMatches.size(); i++) {
                Matching m = (Matching) unionMatches.get(i);

                // if it is one-to-one match
                if (m.tnode != null) {
                    addGoodInputMap_1_to_1(m, r1, resHash, goodInputMap);

                    // Add local n-to-1 matches
                } else if ((m.matchingUnits != null) &&
                        !m.matchingUnits.isEmpty()) {
                    addGoodInputMap_local_n_to_1(m, r1, resHash, goodInputMap);
                }
            }
        }
    }

    /**
     * Compute the best matching candidate of this schema node TNode and all
     * nodes in the subtree rooted at this node by
     * combining all the matching results of linguistic, structural and
     * optional context check. Asks user input if
     * decision cannot be made at critical points in the interactive mode.
     *
     * @param mapFromGraph The output from graph matching
     * @param nameSimTable The Hashtable that stores the computed similarity
     * for any pair of a node of this SchemaTree and a node of anotherTree
     * @param pathSimTable The Hashtable that stores the computed similarity
     * for any pair of path of this SchemaTree and a path of anotherTree
     * @param descSimTable The Hashtable that stores the computed similarity
     * for any pair of node description of this SchemaTree and a node description
     * of anotherTree
     * @param dataTypeSimTable The Hashtable that stores the similarity
     * for any pair of common data types
     * @param resHash The Hashtable holding the resources for paths
     * @param tree The SchemaTree is being matched with this SchemaTree
     * @param fromContextCheck The boolean value indicating this method is
     * called from context check process or not
     * @param fromUserInputSubTreeMatch The boolean value indicating this method
     * is called in the process of matching subtrees of user input match or not
     * @param interactive_on The boolean value indicating user interaction is
     * expected or not, i.e., in interactive mode or non-intercative mode
     */
    public void getFromCombForAll(Map mapFromGraph, Hashtable nameSimTable,
        Hashtable pathSimTable, Hashtable descSimTable,
        Hashtable dataTypeSimTable, Hashtable resHash, //resource hash for this tree
        SchemaTree tree, // another tree to match
        boolean fromContextCheck, boolean fromUserInputSubTreeMatch,
        boolean interactive_on) throws Exception {
        if (!userInputMatch) {
            getFromComb(mapFromGraph, nameSimTable, pathSimTable, descSimTable,
                dataTypeSimTable, resHash, tree, fromContextCheck,
                fromUserInputSubTreeMatch, interactive_on);
        }

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.getFromCombForAll(mapFromGraph, nameSimTable,
                        pathSimTable, descSimTable, dataTypeSimTable, resHash,
                        tree, fromContextCheck, fromUserInputSubTreeMatch,
                        interactive_on);
                }
            }
        }
    }

    /**
     * Compute the best matching candidate of this schema node TNode by
     * combining all the matching results of linguistic, structural and
     * optional context check. Asks user input if
     * decision cannot be made at critical points in the interactive mode.
     *
     * @param mapFromGraph The output from graph matching
     * @param nameSimTable The Hashtable that stores the computed similarity
     * for any pair of a node of this SchemaTree and a node of anotherTree
     * @param pathSimTable The Hashtable that stores the computed similarity
     * for any pair of path of this SchemaTree and a path of anotherTree
     * @param descSimTable The Hashtable that stores the computed similarity
     * for any pair of node description of this SchemaTree and a node description
     * of anotherTree
     * @param dataTypeSimTable The Hashtable that stores the similarity
     * for any pair of common data types
     * @param resHash The Hashtable holding the resources for paths
     * @param tree The SchemaTree is being matched with this SchemaTree
     * @param fromContextCheck The boolean value indicating this method is
     * called from context check process or not
     * @param fromUserInputSubTreeMatch The boolean value indicating this method
     * is called in the process of matching subtrees of user input match or not
     * @param interactive_on The boolean value indicating user interaction is
     * expected or not, i.e., in interactive mode or non-intercative mode
     */
    public void getFromComb(Map mapFromGraph, Hashtable nameSimTable,
        Hashtable pathSimTable, Hashtable descSimTable,
        Hashtable dataTypeSimTable, Hashtable resHash, SchemaTree tree,
        boolean fromContextCheck, boolean fromUserInputSubTreeMatch,
        boolean interactive_on) throws Exception {
        if (getTag().equals(".sequence.") || getTag().equals(".choice.") ||
                getTag().equals(".all.")) {
            return;
        }

        if (!fromContextCheck && !fromUserInputSubTreeMatch) {
            matchings = new ArrayList();
        } else if (fromContextCheck) { // update everytime for context check??? 
            matchingsFromCC = new ArrayList();
        }

        String path1 = getPath();
        Resource r1 = (Resource) resHash.get(path1);
        List pairs = new ArrayList();
        pairs = (List) mapFromGraph.get(r1);

        double maxSim = 0;
        double secMaxSim = 0;
        double lsimOfBestNode = 0;
        double psimOfBestNode = 0;
        TNode best = new TNode();
        TNode secBest = new TNode();

        if ((pairs == null) || pairs.isEmpty()) {
            return;
        }

        for (int i = 0; i < pairs.size(); i++) {
            MapPair pair = (MapPair) pairs.get(i);

            //if(SCIA.debug_on) System.err.println("pair= " + pair);
            Resource r2 = (Resource) pair.r2;

            // make sure it is a valid mapping pair
            if (r2 != null) {
                String path2 = r2.getLabel();

                if (path2 == null) {
                    if (SCIA.debug_on) {
                        System.err.println("path2 == null");
                    }
                }

                //to do: put following code into autoCombine();
                String key = "(" + path1 + "," + path2 + ")";

                TNode tnode = tree.getTNodeFromPath(path2);
                Double psimD = (Double) pathSimTable.get(key);
                Double nsimD = (Double) nameSimTable.get(key);
                Double dsimD = (Double) descSimTable.get(key);

                // Data type similarity 
                String dt1 = getDataType();
                String dt2 = tnode.getDataType();
                Double dtsimD = null;

                if ((dt1 != null) && !dt1.equals("") && (dt2 != null) &&
                        !dt2.equals("")) {
                    String dtkey = "(" + dt1 + "," + dt2 + ")";
                    dtsimD = (Double) dataTypeSimTable.get(dtkey);
                }

                // Semantic type similarity
                //Double stsimD = 
                double psim = 0.0;
                double nsim = 0.0;
                double lsim = 0.0;
                double dtsim = 0.0;
                double dsim = 0.0;

                if ((psimD != null) && (nsimD != null)) {
                    psim = psimD.doubleValue();
                    nsim = nsimD.doubleValue();
                    lsim = (psim * SCIA.PATH_WEIGHT_LINGUISTIC) +
                        (nsim * SCIA.NAME_WEIGHT_LINGUISTIC);
                }

                if (dtsimD != null) {
                    dtsim = dtsimD.doubleValue();
                }

                if (dsimD != null) {
                    dsim = dsimD.doubleValue();
                }

                double sim = (pair.sim * SCIA.GRAPH_WEIGHT_COMB) +
                    (lsim * SCIA.LINGUITIC_WEIGHT_COMB) +
                    (dtsim * SCIA.DATATYPE_WEIGHT_COMB) +
                    (dsim * SCIA.DESCRIPTION_WEIGHT_COMB);

                if (tnode != null) {
                    if (!fromContextCheck) {
                        matchings.add(new Matching(tnode, sim));
                    } else {
                        matchingsFromCC.add(new Matching(tnode, sim));
                    }

                    if (sim > maxSim) {
                        secMaxSim = maxSim;
                        secBest = best;
                        maxSim = sim;
                        best = tnode;
                        lsimOfBestNode = lsim;
                        psimOfBestNode = psim;
                    } else if ((sim == maxSim) || (sim > secMaxSim)) {
                        secMaxSim = sim;
                        secBest = tnode;
                    }

                    /* bug: don't go through telephone
                    if (getPath().equalsIgnoreCase
                        ("/ExcelOrder/Header/Contact/telephone")) {
                        if(SCIA.debug_on) System.err.println("\n##" + getPath() +
                                           " with sim = " +
                                           sim + " to " + tnode.getPath() +
                                           " \nmaxSim = " + maxSim +
                                           " best = " + best.getPath() +
                                           " best node's lsim = " +
                                           lsimOfBestNode +
                                           " \nsecMaxSim = " + secMaxSim +
                                           " secBest = " +
                                           secBest.getPath());
                        BufferedReader BR =
                            new BufferedReader(new InputStreamReader
                            (System.in));
                        if(SCIA.debug_on) System.err.println("Go ahead(y/n)");
                        char choice = (char)BR.read();
                        if (choice == 'n') {
                            System.exit(0);
                        }
                    }
                    */
                }
            }
        }

        if (!fromContextCheck && !fromUserInputSubTreeMatch) {
            if ((best != null) && (secBest != null)) {
                if ((best.getPath() != null) && (secBest.getPath() != null)) {
                    if ((secMaxSim > SCIA.SIGNIFICANT_SIM_THRESHOLD) &&
                            ((maxSim - secMaxSim) < SCIA.DIFFERENCE_THRESHOLD)) {
                        secFromComb = new Matching(secBest, secMaxSim);
                    }

                    fromComb = new Matching(best, maxSim);
                }
            }
        }

        /* record the good matches from one subtree into matchings field
           if from user input subtree match, there may be multiple matched
           subtrees
        */
        if (fromUserInputSubTreeMatch) {
            if (SCIA.debug_on) {
                System.err.println(
                    "\n ######## fromUserInputSubTreeMatch #####");
            }

            if ((best != null) && (secBest != null)) {
                if ((best.getPath() != null) && (secBest.getPath() != null)) {
                    if ((secMaxSim > SCIA.SIGNIFICANT_SIM_THRESHOLD) &&
                            ((maxSim - secMaxSim) < SCIA.DIFFERENCE_THRESHOLD)) {
                        /*
                        if(SCIA.debug_on) System.err.println();
                        if(SCIA.debug_on) System.err.println("**************************");
                        if(SCIA.debug_on) System.err.println("I can't differentiate " +
                                           best.getPath() +
                                           " from" + secBest.getPath());
                        */
                        secFromComb = new Matching(secBest, secMaxSim);

                        //System.exit(0);
                    }

                    fromComb = new Matching(best, maxSim);
                }
            }

            if (subTreeMatchings == null) {
                subTreeMatchings = new ArrayList();
            }

            if (fromComb != null) {
                if (SCIA.debug_on) {
                    System.err.println(getPath() + "fromComb = " +
                        fromComb.tnode.getPath() + " sim = " + fromComb.sim);
                }

                subTreeMatchings.add(fromComb);

                if (secFromComb != null) {
                    if (SCIA.debug_on) {
                        System.err.println(getPath() + "secFromComb = " +
                            secFromComb.tnode.getPath() + " sim = " +
                            secFromComb.sim);
                    }

                    subTreeMatchings.add(secFromComb);
                }

                if (SCIA.debug_on) {
                    System.err.println("\n#########      current matchings = " +
                        matchings);
                }
            }

            return;
        }

        //Sort the matchings in decending order based on the similarity value
        sortMatchings(matchings);

        if (fromContextCheck) {
            getFromCombFromCC(nameSimTable, pathSimTable, maxSim, secMaxSim,
                psimOfBestNode, best, secBest, path1, interactive_on);
        }
    }

    /**
     * Compute the best matching candidates of this schema node TNode by
     * combining all the matching
     * results of linguistic, structural and context check. Asks user input if
     * decision cannot be made at critical points in the interactive mode.
     *
     * @param nameSimTable The Hashtable that stores the computed similarity
     * for any pair of a node of this SchemaTree and a node of anotherTree
     * @param pathSimTable The Hashtable that stores the computed similarity
     * for any pair of path of this SchemaTree and a path of anotherTree
     * @param maxSim The maximum similarity value of matching candidate that
     * this node gets so far
     * @param secMaxSim The second maximum similarity value of matching
     * candidate that this node gets so far
     * @param psimOfBestNode The path similarity value of the best matching
     * candidate so far
     * @param best The best matching node that this node gets so far
     * @param secBest The second best matching node that this node gets so far
     * @param interactive_on The boolean value indicating user interaction is
     * expected or not, i.e., in interactive mode or non-intercative mode
     */
    public void getFromCombFromCC(Hashtable nameSimTable,
                                  Hashtable pathSimTable, 
                                  double maxSim, 
                                  double secMaxSim,
                                  double psimOfBestNode, 
                                  TNode best, 
                                  TNode secBest, 
                                  String path1,
                                  boolean interactive_on) throws Exception {
        // from context check, combine w/o context check and w/ context check
        if (fromComb != null) {
            double fromCCSim = 0.0;
            Matching tmp;
            List combined = new ArrayList();

            for (int j = 0; j < matchings.size(); j++) {
                Matching currMatching = (Matching) matchings.get(j);

                for (int k = 0; k < matchingsFromCC.size(); k++) {
                    if (currMatching.tnode == ((Matching) matchingsFromCC.get(k)).tnode) {
                        fromCCSim = ((Matching) matchingsFromCC.get(k)).sim;
                    }

                    tmp = new Matching(currMatching.tnode,
                            currMatching.sim + fromCCSim);
                    combined.add(tmp);
                }
            }

            sortMatchings(combined);

            /* w/o context fromComb is equally or less similar than
               w/ context check, update fromComb and matchings
            */
            if ((fromComb.sim <= maxSim) && (fromComb.tnode != best)) {
                // got multiple equally good ones
                if (maxSim >= SCIA.CORE_ACCEPT_THRESHOLD) {
                    if ((secBest != null) &&
                            (secMaxSim > SCIA.CORE_ACCEPT_THRESHOLD) &&
                            ((maxSim - secMaxSim) < SCIA.DIFFERENCE_THRESHOLD)) {
                        fromComb = new Matching(best, maxSim);
                        secFromComb = new Matching(secBest, secMaxSim);
                        resolveMultiMatches(interactive_on);
                    }
                } else if ((secBest != null) &&
                        (secMaxSim > SCIA.SIGNIFICANT_SIM_THRESHOLD) &&
                        ((maxSim - secMaxSim) < SCIA.DIFFERENCE_THRESHOLD)) {
                    if (fromComb.sim <= secMaxSim) {
                        if ((secFromComb == null) ||
                                (secFromComb.sim < secMaxSim)) {
                            secFromComb = new Matching(secBest, secMaxSim);
                        }
                    } else {
                        secFromComb = fromComb;
                    }

                    fromComb = new Matching(best, maxSim);
                }

                /* the best from context check is much less similar than
                   w/o context fromComb that has very high sim value
                */
            } else if ((fromComb.sim > maxSim) &&
                    (((fromComb.sim - maxSim) >= 0.2) || (fromComb.sim > 0.85))) {
                /* The previous best from without context check has higher
                   similarity value than the best from context check. If they
                   are not the same node, this indicates condition may be
                   needed,
                   propose condition from 2 contexts connection point, e.g.,
                   genomeref to genome@id,
                   and ask users
                */
                if (fromComb.tnode != best) {
                    resolveContextConflict();
                }

                // original fromComb is better, but not much better,
                // user input might be needed
            } else if ((fromComb.sim > maxSim) && (fromComb.tnode != best) &&
                    ((fromComb.sim - maxSim) < 0.2)) {
                if (SCIA.debug_on) {
                    System.err.println("***fromComb = " +
                        fromComb.tnode.getPath());
                }

                if (SCIA.debug_on) {
                    System.err.println("***best = " + best.getPath() +
                        " sim = " + maxSim);
                }

                if (SCIA.debug_on) {
                    System.err.println("***secBest = " + secBest.getPath() +
                        " sim = " + secMaxSim);
                }

               
                if (parent.fromComb.tnode != null) {
                    TNode parentMatchNode = parent.fromComb.tnode;

                    if ((parentMatchNode == best.parent) ||
                            best.parent.isDecendentOf(parentMatchNode)) {
                        if (!fromComb.tnode.isDecendentOf(parentMatchNode)) {
                            // if parent context is consistent, take the better one
                            secFromComb = fromComb;
                            fromComb = new Matching(best, maxSim);

                            if ((secBest != null) &&
                                    (parentMatchNode == secBest.parent) &&
                                    ((fromComb.sim - secMaxSim) < SCIA.DIFFERENCE_THRESHOLD)) {
                                secFromComb = new Matching(secBest, secMaxSim);
                            }

                            // ask user to solve multiple matches             
                            resolveMultiMatches(interactive_on);

                            //this node's new best from context check is out of 
                            //this node's parent context
                        } else {
                            secFromComb = new Matching(best, maxSim);

                            if (SCIA.debug_on) {
                                System.err.println("***after secFromComb****");
                            }

                            //resolveContextConflict();
                            resolveMultiMatches(interactive_on);
                        }
                    }
                }

                //if there are multiple better matches from w/o CC than best
                // from CC, it means????
                if ((secFromComb != null) && (secFromComb.tnode != null) &&
                        (fromComb != null) && (fromComb.tnode != null) &&
                        (secFromComb.tnode != fromComb.tnode) &&
                        (secFromComb.sim > maxSim)) {
                    // keep original fromComb and secFromComb 
                    // if non-interactive
                    if (interactive_on &&
                            (hasDeepSubTree() || hasBigSubTree())) {
                        String message = getPath() + " have multiple matches?";
                        final Interactive inter = new Interactive(message,
                                keyEntered);

                        //user interaction
                        Thread.sleep(200);
                        inter.show();

                        Thread.sleep(200);
                        tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                            tree.mainUI.treesSplitPane.getWidth(),
                            tree.mainUI.treesSplitPane.getHeight());
                        tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                        // wait for the option to close
                        while (inter.isVisible()) {
                            Thread.sleep(100);
                        }

                        keyEntered = inter.keyEntered;

                        if (keyEntered == KeyEvent.VK_N) {
                        } else if (keyEntered == KeyEvent.VK_Y) {
                            message = fromComb.tnode.getPath() + " sim = " +
                                format.format(fromComb.sim) +
                                " is a good match?";

                            final Interactive inter1 = new Interactive(message,
                                    keyEntered);

                            Thread.sleep(200);
                            inter1.show();

                            Thread.sleep(200);
                            tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                                tree.mainUI.treesSplitPane.getWidth(),
                                tree.mainUI.treesSplitPane.getHeight());
                            tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                            // wait for the option to close
                            while (inter1.isVisible()) {
                                Thread.sleep(100);
                            }

                            keyEntered = inter1.keyEntered;

                            if (keyEntered == KeyEvent.VK_Y) {
                                fromComb.sim = 1.0;
                                userInputMatch = true;
                            } else if (keyEntered == KeyEvent.VK_N) {
                                fromComb = null;
                            }

                            message = secFromComb.tnode.getPath() + " sim = " +
                                format.format(secFromComb.sim) +
                                " is a good match?";

                            final Interactive inter2 = new Interactive(message,
                                    keyEntered);

                            Thread.sleep(200);
                            inter2.show();

                            Thread.sleep(200);
                            tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                                tree.mainUI.treesSplitPane.getWidth(),
                                tree.mainUI.treesSplitPane.getHeight());
                            tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                            // wait for the option to close
                            while (inter2.isVisible()) {
                                Thread.sleep(100);
                            }

                            keyEntered = inter2.keyEntered;

                            if (keyEntered == KeyEvent.VK_Y) {
                                secFromComb.sim = 1.0;
                                userInputMatch = true;

                                if (fromComb == null) {
                                    fromComb = secFromComb;
                                }
                            } else if (keyEntered == KeyEvent.VK_N) {
                                secFromComb = null;
                            }

                            message = "More good matches?";

                            final Interactive inter3 = new Interactive(message,
                                    keyEntered);

                            Thread.sleep(200);
                            inter3.show();

                            Thread.sleep(200);
                            tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                                tree.mainUI.treesSplitPane.getWidth(),
                                tree.mainUI.treesSplitPane.getHeight());
                            tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                            // wait for the option to close
                            while (inter3.isVisible()) {
                                Thread.sleep(100);
                            }

                            keyEntered = inter3.keyEntered;

                            while (keyEntered == KeyEvent.VK_Y) {
                                message = "Please input one more good" +
                                    "match: ";

                                final Interactive inter4 = new Interactive(message,
                                        keyEntered);

                                Thread.sleep(200);
                                inter4.show();

                                Thread.sleep(200);
                                tree.clearSelection();

                                while (tree.getSelectionCount() == 0) {
                                }

                                ;

                                dataEntered = tree.getSelectionPath().toString();

                                if (SCIA.debug_on) {
                                    System.err.println("input match is " +
                                        dataEntered);
                                }

                                TNode inputMatchTNode = (TNode) tree.getLastSelectedPathComponent();

                                // set thirdFromComb, etc.
                                Thread.sleep(200);
                                tree.mainUI.treesSplitPane.paintImmediately(0,
                                    0, tree.mainUI.treesSplitPane.getWidth(),
                                    tree.mainUI.treesSplitPane.getHeight());
                                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                                // wait for the option to close
                                while (inter3.isVisible()) {
                                    Thread.sleep(100);
                                }
                            }
                        }
                    }
                } else if (interactive_on && hasDeepSubTree()) {
                    //user interaction
                    String message = "I can't determine where to match to " +
                        getPath() + "\nPlease choose one option: ";

                    Vector choices = new Vector();

                    choices.add("Follow original fromComb: " +
                        fromComb.tnode.getPath() + " sim = " + fromComb.sim);
                    choices.add("Follow  the best from context check: " +
                        best.getPath() + " sim = " + maxSim);
                    choices.add("No good match");
                    choices.add("I will tell you where to match");

                    final String msg = message;
                    final Vector c = choices;

                    final Interactive2 inter2 = new Interactive2(msg, c);

                    Thread.sleep(200);
                    inter2.show();

                    Thread.sleep(200);
                    tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                        tree.mainUI.treesSplitPane.getWidth(),
                        tree.mainUI.treesSplitPane.getHeight());
                    tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                    // wait for the option to close
                    while (inter2.isVisible()) {
                        Thread.sleep(100);
                    }

                    dataEntered = inter2.data;

                    if (dataEntered.equals("2")) {
                        fromComb = new Matching(best, maxSim);

                        //update matchings?
                    } else if (dataEntered.equals("3")) {
                        fromComb = null;
                    } else if (dataEntered.equals("4")) {
                        // get user input
                    }
                } else if (!interactive_on || !hasDeepSubTree()) {
                    // decision is based on linguistic similarity or only path 
                    //similarity???
                    if ((fromComb != null) && (fromComb.tnode != null)) {
                        String key = "(" + path1 + "," +
                            fromComb.tnode.getPath() + ")";
                        Double psimD = (Double) pathSimTable.get(key);
                        Double nsimD = (Double) nameSimTable.get(key);

                        if ((psimD != null) && (nsimD != null)) {
                            double psim = ((Double) pathSimTable.get(key)).doubleValue();
                            double nsim = ((Double) nameSimTable.get(key)).doubleValue();
                            double lsim = (psim * SCIA.PATH_WEIGHT_LINGUISTIC) +
                                (nsim * SCIA.NAME_WEIGHT_LINGUISTIC);

                            if (SCIA.bothXMLSchemas &&
                                    this.data.dataType.equalsIgnoreCase(
                                        fromComb.tnode.data.dataType)) {
                                lsim = lsim + SCIA.SAME_DATA_TYPE_SIM;
                            }

                            //based on linguis. sim 
                            //if (lsim < lsimOfBestNode) {
                            // based on path sim 
                            if (psim < psimOfBestNode) {
                                if ((secBest != null) &&
                                        (secMaxSim > SCIA.SIGNIFICANT_SIM_THRESHOLD) &&
                                        ((maxSim - secMaxSim) < SCIA.DIFFERENCE_THRESHOLD)) {
                                    if (fromComb.sim <= secMaxSim) {
                                        if ((secFromComb == null) ||
                                                (secFromComb.sim < secMaxSim)) {
                                            secFromComb = new Matching(secBest,
                                                    secMaxSim);
                                        }
                                    } else {
                                        secFromComb = fromComb;
                                    }
                                }

                                fromComb = new Matching(best, maxSim);
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * Compute the linguistic matchings for this node and all nodes in the
     * subtree rooted at this node.
     *
     * @param working1 The working TNode in another tree
     * @param nameSimTable The Hashtable holding the similarity values between
     * any pair of one node in this tree and one node in another tree
     * @param The Hashtable holding the similarity values between
     * any pair of one path in this tree and one path in another tree
     */
    public void getLingMatchingsForAll(TNode working1, Hashtable nameSimTable, //name sim table
        Hashtable pathSimTable) //path sim table
        throws Exception {
        getLingMatchings(working1, nameSimTable, pathSimTable);

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.getLingMatchingsForAll(working1, nameSimTable,
                        pathSimTable);
                }
            }
        }
    }

    public void getNumOfPotentialMatchesForAll(SchemaTree anotherTree,
        Hashtable nameSimTable, Hashtable pathSimTable, Hashtable descSimTable,
        Hashtable dataTypeSimTable, boolean interactive_on)
        throws Exception {
        getNumOfPotentialMatches(anotherTree, nameSimTable, pathSimTable,
            descSimTable, dataTypeSimTable, interactive_on);

        if ((children != null) && !children.isEmpty()) {
            for (int i = 0; i < children.size(); i++) {
                ((TNode) children.get(i)).getNumOfPotentialMatchesForAll(anotherTree,
                    nameSimTable, pathSimTable, descSimTable, dataTypeSimTable,
                    interactive_on);
            }
        }
    }

    public void getNumOfPotentialMatches(SchemaTree anotherTree,
        Hashtable nameSimTable, Hashtable pathSimTable, Hashtable descSimTable,
        Hashtable dataTypeSimTable, boolean interactive_on)
        throws Exception {
        if ((lingMatchings != null) && !lingMatchings.isEmpty()) {
            sortMatchings(lingMatchings);

            if (getPath().equalsIgnoreCase("/ExcelOrder/Header/Contact")) {
                for (int j = 0; j < 2; j++) {
                    Matching temp = (Matching) lingMatchings.get(j);

                    if (SCIA.debug_on) {
                        System.err.println(j + "th best lingMatch = " +
                            temp.tnode.getPath() + " sim = " + temp.sim);
                    }
                }

                //System.exit(0);
            }

            double maxSim = ((Matching) lingMatchings.get(0)).sim;

            if (maxSim > SCIA.MULTIPLE_LING_MATCHES_THRESHOLD) {
                for (int j = 1; j < lingMatchings.size(); j++) {
                    double tempSim = ((Matching) lingMatchings.get(j)).sim;

                    if ((tempSim < SCIA.MULTIPLE_LING_MATCHES_THRESHOLD) ||
                            ((maxSim - tempSim) > SCIA.DIFFERENCE_THRESHOLD)) {
                        break;
                    }

                    numOfPotentialMatches++;
                }
            }
        }

        // potential multiple matches exist, for important node with big subtree,
        // user input would be helpful
        if (interactive_on && (numOfPotentialMatches > 1) &&
                (hasBigSubTree() || hasDeepSubTree())) {
            String message = getPath() +
                " might have multiple matches, give input?";

            final Interactive inter = new Interactive(message, keyEntered);

            //user interaction
            Thread.sleep(200);
            inter.show();

            Thread.sleep(200);
            tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                tree.mainUI.treesSplitPane.getWidth(),
                tree.mainUI.treesSplitPane.getHeight());
            tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

            // wait for the option to close
            while (inter.isVisible()) {
                Thread.sleep(100);
            }

            keyEntered = inter.keyEntered;

            if (keyEntered == KeyEvent.VK_Y) {
                userInputMatch = true;
                matchings = new ArrayList();
                numOfMatches = 0;

                for (int j = 0; j < lingMatchings.size(); j++) {
                    TNode tempNode = ((Matching) lingMatchings.get(j)).tnode;
                    double tempSim = ((Matching) lingMatchings.get(j)).sim;

                    message = tempNode.getPath() + " sim = " +
                        format.format(tempSim) + " is a good match?";

                    final Interactive inter1 = new Interactive(message,
                            keyEntered);

                    Thread.sleep(200);
                    inter1.show();

                    Thread.sleep(200);
                    tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                        tree.mainUI.treesSplitPane.getWidth(),
                        tree.mainUI.treesSplitPane.getHeight());
                    tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                    // wait for the option to close
                    while (inter1.isVisible()) {
                        Thread.sleep(100);
                    }

                    keyEntered = inter1.keyEntered;

                    if (keyEntered == KeyEvent.VK_Y) {
                        numOfMatches++;
                        matchings.add(new Matching(tempNode, 1.0));
                    }
                }

                message = "More matches to add?";

                final Interactive inter2 = new Interactive(message, keyEntered);

                Thread.sleep(200);
                inter2.show();

                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                // wait for the option to close
                while (inter2.isVisible()) {
                    Thread.sleep(100);
                }

                keyEntered = inter2.keyEntered;

                while (keyEntered == KeyEvent.VK_Y) {
                    message = "Please input the new match path in" +
                        " another tree:\n";

                    final Interactive inter3 = new Interactive(message,
                            keyEntered);

                    Thread.sleep(200);
                    inter3.show();

                    anotherTree.clearSelection();

                    while (anotherTree.getSelectionCount() == 0) {
                    }

                    ;

                    dataEntered = anotherTree.getSelectionPath().toString();

                    if (SCIA.debug_on) {
                        System.err.println("input match is " + dataEntered);
                    }

                    TNode inputMatchTNode = (TNode) anotherTree.getLastSelectedPathComponent();

                    numOfMatches++;
                    matchings.add(new Matching(inputMatchTNode, 1.0));

                    Thread.sleep(200);
                    tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                        tree.mainUI.treesSplitPane.getWidth(),
                        tree.mainUI.treesSplitPane.getHeight());
                    tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                    // wait for the option to close
                    while (inter2.isVisible()) {
                        Thread.sleep(100);
                    }
                }

                // inform its children,they would have matches from multiple subtrees
                if ((numOfMatches >= 1) &&
                        (hasBigSubTree() || hasDeepSubTree())) {
                    informDecendentsUserInput(this);

                    // do subtree matching
                    if ((matchings != null) && !matchings.isEmpty()) {
                        for (int i = 0; i < matchings.size(); i++) {
                            Matching match = (Matching) matchings.get(i);

                            if (SCIA.debug_on) {
                                System.err.println("match = " +
                                    match.tnode.getPath() + " sim = " +
                                    match.sim);
                            }

                            // update fromNameAndPath field for this subtree, 
                            // to recompute from the corresponding
                            // subtree in another tree
                            getFromNameAndPathForAll(match.tnode, 
                            // corresponding subtree root
                            nameSimTable, pathSimTable);
                            subTreeMatching(anotherTree, match.tnode,
                                nameSimTable, pathSimTable, descSimTable,
                                dataTypeSimTable, interactive_on);
                            write_AllTNodes(new PrintWriter(System.err, true),
                                "    ", false);
                        }
                    }
                }
            }
        }
    }

    public void informDecendentsUserInput(TNode ancestorHasUserInput) {
        if ((children != null) && !children.isEmpty()) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);
                child.hasAncestorWithUserInput = true;

                if (ancestorHasUserInput.numOfMatches > 1) {
                    hasAncestorWithMultiUserInput = true;
                }

                userInputAncestorNumMatches = ancestorHasUserInput.numOfMatches;
                ancestorWithUserInput = ancestorHasUserInput;

                child.informDecendentsUserInput(ancestorHasUserInput);
            }
        }
    }

    /**
     * Compute the linguistic matchings for this node.
     *
     * @param working1 The working TNode in another tree
     * @param nameSimTable The Hashtable holding the similarity values between
     * any pair of one node in this tree and one node in another tree
     * @param The Hashtable holding the similarity values between
     * any pair of one path in this tree and one path in another tree
     */
    public void getLingMatchings(TNode working1, // another tree working node
        Hashtable nameSimTable, //name sim table
        Hashtable pathSimTable) //path sim table
        throws Exception {
        if (data.nodeType.equals("compositor") ||
                working1.data.nodeType.equals("compositor")) {
            return;
        }

        String path = this.getPath();
        String path1 = working1.getPath();
        String key = "(" + path + "," + path1 + ")";
        double nsim = ((Double) nameSimTable.get(key)).doubleValue();
        double psim = ((Double) pathSimTable.get(key)).doubleValue();

        double lsim = (nsim * SCIA.NAME_WEIGHT_LINGUISTIC) +
            (psim * SCIA.PATH_WEIGHT_LINGUISTIC);

        /* type is already considered when computing the name and path sim
        tables
        if (SCIA.bothXMLSchemas &&
            this.data.dataType.equalsIgnoreCase(working1.data.dataType)) {

            lsim = SCIA.SAME_DATA_TYPE_SIM + lsim;
            }*/
        if (lsim >= SCIA.LINGUISTIC_THRESHOLD) {
            if (lingMatchings == null) {
                lingMatchings = new ArrayList();
            }

            lingMatchings.add(new Matching(working1, lsim));
        }

        if ((working1.children != null) && !working1.children.isEmpty()) {
            for (int i = 0; i < working1.children.size(); i++) {
                getLingMatchings((TNode) working1.children.get(i),
                    nameSimTable, pathSimTable);
            }
        }
    }

    public void adjustMatchWithHierForAll() {
        adjustMatchWithHier();

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.adjustMatchWithHierForAll();
                }
            }
        }
    }

    public void adjustMatchWithHier() {
        //if(SCIA.debug_on) System.err.println("in tnode adjust, this path =" + getPath());
        double maxSim = 0;
        TNode best = new TNode();
        boolean adjusted = false;

        if ((height > 2) && !matchings.isEmpty()) {
            for (int i = 0; i < matchings.size(); i++) {
                Matching target = (Matching) matchings.get(i);

                if ((parent != null) && (target.tnode.parent != null)) {
                    adjusted = true;

                    for (int j = 0; j < parent.matchings.size(); j++) {
                        Matching tmp = (Matching) parent.matchings.get(j);

                        if (tmp.tnode == target.tnode.parent) {
                            if (parent.height == 3) {
                                target.sim = (target.sim * 0.5) +
                                    (tmp.sim * 0.5);
                            }

                            break;
                        }
                    }

                    if (target.sim > maxSim) {
                        maxSim = target.sim;
                        best = target.tnode;
                    }
                }
            }
        }

        if (adjusted) {
            if (SCIA.debug_on) {
                System.err.println("this path = " + getPath() +
                    " best path = " + best.getPath() + " sim = " + maxSim);
            }
        } else {
            if (SCIA.debug_on) {
                System.err.println("this path = " + getPath() +
                    " best path = " + fromComb.tnode.getPath() + " sim = " +
                    fromComb.sim);
            }
        }
    }

    public void removeCompositorNodesInSubTree(TNode working) {
        TNode newWorking = new TNode();

        if (!data.nodeType.equals("compositor")) {
            TNode node = new TNode(this, working);
            newWorking = node;
        } else {
            newWorking = working;
        }

        if ((children != null) && !children.isEmpty()) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);
                child.removeCompositorNodesInSubTree(newWorking);
            }
        }
    }

    /**
     * Check the context consistency and adjust if needed.
     */
    public void adjustMatchWithCoreLevel(SchemaTree anotherTree,
        Hashtable nameSimTable, Hashtable pathSimTable, Hashtable descSimTable,
        Hashtable dataTypeSimTable, boolean interactive_on)
        throws Exception {


        if (SCIA.debug_on) {
            System.err.println("adjusting " + getTag());
        }

        if (fromComb == null || userInputMatch == true) {
            return;
        }

        if (fromComb.sim >= SCIA.CORE_ACCEPT_THRESHOLD) {
            
            if ((secFromComb != null) &&
                    (secFromComb.sim >= SCIA.CORE_ACCEPT_THRESHOLD)) {
                //to do:  if there exist multiple good ones
                if (SCIA.debug_on) {
                    System.err.println(
                        "********** multiple good matches exist!!!");
                }
            }
            
            /* if this node at the core level is matched with high similarity,
               but its context is not consistent, and cross level matching 
               occurs, it indicates regrouping may be involved, and condition 
               may be needed.
            */    
            checkCrossLevelGrouping();            

            // Adjust its subtree matching 
            if ((children != null) && !children.isEmpty()) {
                if (SCIA.debug_on) {
                    System.err.println("adjusting " + getTag());
                }
                Hashtable resHash1 = new Hashtable();
                Hashtable resHash2 = new Hashtable();
                Model mod1 = SchemaTree.rf.createModel();
                Model mod2 = SchemaTree.rf.createModel();
                List initMap = new ArrayList();

                toRDFModelForAll(null, mod1, resHash1);
                fromComb.tnode.toRDFModelForAll(null, mod2, resHash2);

                try {
                    getInitMapFromCombForAll(resHash1, resHash2, initMap,
                        fromComb.tnode);
                } catch (Exception e) {
                    if (SCIA.debug_on) {
                        System.err.println("From getInitMapFromCombForAll():" +
                            e.getMessage());
                    }
                }

                if (SCIA.debug_on) {
                    System.err.println(initMap);
                }

                SFMatcher sfm = new SFMatcher();

                MapPair[] res = sfm.getMatch(mod1, mod2, initMap);

                MapPair.sortGroup(res);

                //print out details of map pairs
                //sfm.dump(res);
                Map mapFromGraph = MapPair.sortedCandidates(res, false);

                // update fromComb field, the best matching 
                // from combination of without context check and with context check
                //matchings = null; 
                getFromCombForAll(mapFromGraph, nameSimTable, pathSimTable,
                    descSimTable, dataTypeSimTable, resHash1, anotherTree,
                    true, false, interactive_on);

                //update mapping in the gui
                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.addMatchedLines();
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());
            }
        } else if (!interactive_on) { 
            // sim < accept_threshold and not in interactive mode

            // deal with the non-trustable matches in its shallow subtree
            if (!hasDeepSubTree()) {
                if (fromComb.sim < SCIA.CORE_LEAF_ACCEPT_THRESHOLD) {
                    // if data types are compatible, especially when they are 
                    //date, int, 
                    // number, keep them and inform users the uncertainty
                    if (SCIA.bothXMLSchemas &&
                            this.data.dataType.equalsIgnoreCase(
                                fromComb.tnode.data.dataType) &&
                            (this.data.dataType.equalsIgnoreCase("date") ||
                            this.data.dataType.equalsIgnoreCase("int")) &&
                            (fromComb.sim >= SCIA.TYPE_MATCH_CORE_LEAF_THRESHOLD)) {
                        // one more field in TNode, check when writing out????
                        // remove second best match, 
                        secFromComb = null;
                    } else {
                        //throw the non-trustable matches in its shallow subtree
                        resetFromCombForAll();
                    }
                }
            } else {
                /* it has deep subtree, ask user what strategy to use:
                 * 1) throw whole subtree's matches,
                      e.g. article element of book2.dtd (arts)
                 * 2) only throw this node and its leaf children's matches,
                      and check and adjust its non-leaf children,
                      e.g. feature-tables element of BSML
                 */

                //user interaction
                String message = "I am not sure about whether " +
                    fromComb.tnode.getPath() + " is a good match (sim=" +
                    format.format(fromComb.sim) + ")" + " to " + getPath() +
                    " but it has a deep subtree, " +
                    "please choose a strategy to deal " +
                    "with it and its subtree";

                Vector choices = new Vector();

                choices.add("  to throw whole subtree's untrustable " +
                    "matches");
                choices.add("  to throw only the trustable matches " +
                    "of itself and leaf-children's, continue " +
                    "to check its non-leaf children");
                choices.add("  no suggestion, trust your prediction");

                final String msg = message;
                final Vector c = choices;

                final Interactive2 inter2 = new Interactive2(msg, c);

                Thread.sleep(200);
                inter2.show();

                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                // wait for the option to close
                while (inter2.isVisible()) {
                    Thread.sleep(100);
                }

                dataEntered = inter2.data;

                if (dataEntered.equals("1")) {
                    resetFromCombForAll();
                } else if (dataEntered.equals("2")) {
                    resetFromComb();

                    // adjust its non-leaf children, recursively
                    if ((children != null) && !children.isEmpty()) {
                        for (int i = 0; i < children.size(); i++) {
                            TNode child = (TNode) children.get(i);

                            if ((child.children != null) &&
                                    !child.children.isEmpty()) {
                                child.adjustMatchWithCoreLevel(anotherTree,
                                    nameSimTable, pathSimTable, descSimTable,
                                    dataTypeSimTable, interactive_on);
                            } else {
                                // throw leaf child's matches
                                child.resetFromComb();
                            }
                        }
                    }
                }
            }
        } else if (interactive_on) { 
            // sim < accept_threshold and in interactive mode 

            // not matched well, ask for user input
            // then adjust its subtree matching

            //tree.write(cout, true);
            // highlight working target node and poor matching nodes in gui
            Vector nodes = new Vector();
            nodes.add(this);
            nodes.add(fromComb.tnode);
            anotherTree.mainUI.treesSplitPane.highlightNodes = nodes;
            anotherTree.mainUI.treesSplitPane.drawHighlight(nodes,
                anotherTree.mainUI.treesSplitPane.getGraphics());

            //left or right side of match lines attached to highlighted node 
            //might not be shown
            //tree.mainUI.addMatchedLines();
            tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

            //user interaction
            String message = "";

            if (secFromComb != null) {
                message = "For " + getPath() +
                    ", the predicted best match is " +
                    fromComb.tnode.getPath() + "  sim = " +
                    format.format(fromComb.sim) +
                    " , the second best match is " +
                    secFromComb.tnode.getPath() + "  sim = " +
                    format.format(secFromComb.sim) +
                    ", please provide input and choose one option:";
            } else {
                message = "For " + getPath() +
                    ", the predicted best matching is " +
                    fromComb.tnode.getPath() + "  sim = " +
                    format.format(fromComb.sim) +
                    ", please provide input and choose one option:";
            }

            Vector choices = new Vector();

            choices.add("No match for this path");
            choices.add("No match for this path, " +
                "I will tell you its constant value");
            choices.add("No match for this path, " +
                "but there exist match for its decendent(s)");
            choices.add("Your prediction is good, take the best");
            choices.add("Your prediction is good, take the second");
            choices.add("Take the best, I will add information");
            choices.add("Take the second, I will add information");
            choices.add("I will tell where to match");

            final String msg = message;
            final Vector c = choices;

            final Interactive2 inter2 = new Interactive2(msg, c);

            Thread.sleep(200);
            inter2.show();

            Thread.sleep(200);
            tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                tree.mainUI.treesSplitPane.getWidth(),
                tree.mainUI.treesSplitPane.getHeight());
            tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

            // wait for the option to close
            while (inter2.isVisible()) {
                Thread.sleep(100);
            }

            // unhighlight nodes
            int selectionRow = anotherTree.mainUI.schemaJTree1.getMinSelectionRow();

            if (selectionRow != 0) {
                anotherTree.mainUI.schemaJTree1.removeSelectionRow(selectionRow);
            }

            selectionRow = anotherTree.mainUI.schemaJTree2.getMinSelectionRow();

            if (selectionRow != 0) {
                anotherTree.mainUI.schemaJTree2.removeSelectionRow(selectionRow);
            }

            // set arrow to be un-selected
            tree.mainUI.arr.manualSelect("none");

            dataEntered = inter2.data;

            if (SCIA.debug_on) {
                System.err.println("option is " + dataEntered);
            }

            if (dataEntered.equals("1")) {
                // no matching path, no fixed value available
                // reset the best matches, the fromComb field, of its subtree
                fromComb = null;
                resetFromCombForAll();

                //redraw mapping in the gui
                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.addMatchedLines();
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());
            } else if (dataEntered.equals("2")) {
                // no matching path, but fixed value exists
                // reset the best matches, the fromComb field, of its subtree
                resetFromCombForAll();

                //redraw mapping in the gui
                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.addMatchedLines();
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                final String msg1 = "Please enter the value:";
                final Interactive1 inter3 = new Interactive1(msg1);

                Thread.sleep(200);
                inter3.show();

                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                // wait for the option to close
                while (inter3.isVisible()) {
                    Thread.sleep(100);
                }

                constantValue = inter3.data;
            } else if (dataEntered.equals("3")) {
                // no match, but there exist match for its decendent(s)
                // reset the best match, the fromComb field, only for this node
                resetFromComb();

                // adjust its children, recursively
                if ((children != null) && !children.isEmpty()) {
                    for (int i = 0; i < children.size(); i++) {
                        TNode child = (TNode) children.get(i);
                        child.adjustMatchWithCoreLevel(anotherTree,
                            nameSimTable, pathSimTable, descSimTable,
                            dataTypeSimTable, interactive_on);
                    }
                } else {
                    if (SCIA.debug_on) {
                        System.err.println("Sorry, it has no decendents!");
                    }
                }
            } else if (dataEntered.equals("4")) {
                // keep the best, set sim = 1 and userInputMatch = true
                fromComb.sim = 1.0;
                userInputMatch = true;

                //update ancetors' and subtree's match similarity values
                updateAncestorMatch();
                updateDecendentMatch();

                //redraw mapping in the gui
                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());

                tree.mainUI.schemaJTree1.setSelectionPath(null);
                tree.mainUI.schemaJTree2.setSelectionPath(null);
                tree.mainUI.treesSplitPane.highlightNodes = null;

                //if(SCIA.debug_on) System.err.println("in L2496");
                //tree.write(cout, true);
                //cout.flush();
                tree.mainUI.addMatchedLines();
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());
            } else if (dataEntered.equals("5")) {
                // set secFromComb as fromComb
                fromComb = secFromComb;
                fromComb.sim = 1.0;
                userInputMatch = true;
            } else if (dataEntered.equals("6")) {
                // keep the best, set sim = 1 and userInputMatch = true
                fromComb.sim = 1.0;
                userInputMatch = true;

                //update ancetors' and subtree's match similarity values
                updateAncestorMatch();
                updateDecendentMatch();

                message = "Add information to this match";

                final Interactive4 inter = new Interactive4(message);

                Thread.sleep(200);
                inter.show();

                while (inter.isVisible()) {
                    Thread.sleep(100);
                }

                //redraw mapping in the gui
                //tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                //tree.mainUI.treesSplitPane.getWidth(), 
                //tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.schemaJTree1.setSelectionPath(null);
                tree.mainUI.schemaJTree2.setSelectionPath(null);
                tree.mainUI.treesSplitPane.highlightNodes = null;

                tree.mainUI.addMatchedLines();
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());
            } else if (dataEntered.equals("7")) {
                // set second as best, set sim = 1 and userInputMatch = true
                fromComb = secFromComb;
                fromComb.sim = 1.0;
                userInputMatch = true;

                //update ancetors' and subtree's match similarity values
                updateAncestorMatch();
                updateDecendentMatch();

               //redraw mapping in the gui
                //tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                //tree.mainUI.treesSplitPane.getWidth(), 
                //tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.schemaJTree1.setSelectionPath(null);
                tree.mainUI.schemaJTree2.setSelectionPath(null);
                tree.mainUI.treesSplitPane.highlightNodes = null;

                tree.mainUI.addMatchedLines();
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());
                message = "Add information to this match";

                final Interactive4 inter = new Interactive4(message);

                Thread.sleep(200);
                inter.show();

                while (inter.isVisible()) {
                    Thread.sleep(100);
                }

        } else if (dataEntered.equals("8")) { // input matching path
                fromComb = null;
                independentUnionMatches = null;
                dependentUnionMatches = null;

                //redraw mapping in the gui
                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());

                tree.mainUI.schemaJTree1.setSelectionPath(null);
                tree.mainUI.schemaJTree2.setSelectionPath(null);
                tree.mainUI.treesSplitPane.highlightNodes = null;

                //if(SCIA.debug_on) System.err.println("in L2496");
                //tree.write(cout, true);
                //cout.flush();
                tree.mainUI.addMatchedLines();
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());
                message = "Please enter matches for this path: " + getPath();

                final Interactive4 inter = new Interactive4(message);

                Thread.sleep(200);
                inter.show();

                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                while (inter.isVisible()) {
                    Thread.sleep(100);
                }

                userInputMatch = true;
                updateAncestorMatch();

                if ((fromComb != null) && (fromComb.matchingUnits != null) &&
                        !fromComb.matchingUnits.isEmpty()) {
                    for (int i = 0; i < fromComb.matchingUnits.size(); i++) {
                        MatchingUnit unit = (MatchingUnit) fromComb.matchingUnits.get(i);

                        if (SCIA.debug_on) {
                            System.err.println(getPath() + " match unit" + i +
                                " = " + unit.sourceNode.getPath());
                        }
                    }
                }

                if ((this.children != null) && !this.children.isEmpty() &&
                        (fromComb.tnode.children != null) &&
                        !fromComb.tnode.children.isEmpty()) {
                    Hashtable resHash1 = new Hashtable();
                    Hashtable resHash2 = new Hashtable();
                    Model mod1 = SchemaTree.rf.createModel();
                    Model mod2 = SchemaTree.rf.createModel();
                    List initMap = new ArrayList();

                    toRDFModelForAll(null, mod1, resHash1);

                    fromComb.tnode.toRDFModelForAll(null, mod2, resHash2);
                    getInitMapFromCombForAll(resHash1, resHash2, initMap,
                        fromComb.tnode);

                    //if(SCIA.debug_on) System.err.println(initMap);
                    SFMatcher sfm = new SFMatcher();
                    MapPair[] res = sfm.getMatch(mod1, mod2, initMap);
                    MapPair.sortGroup(res);

                    if (SCIA.debug_on) {
                        System.err.println("after sortGroup");
                    }

                    //sfm.dump(res);
                    Map mapFromGraph = MapPair.sortedCandidates(res, false);

                    if (SCIA.debug_on) {
                        System.err.println("after mapfromgraph");
                    }

                    //update subtree's fromComb field, the best matching 
                    //from combination of methods with user input match
                    matchings = null; // reset matchings field for this tnode 
                    getFromCombForAll(mapFromGraph, nameSimTable, pathSimTable,
                        descSimTable, dataTypeSimTable, resHash1, anotherTree,
                        true, false, interactive_on);

                    if (SCIA.debug_on) {
                        System.err.println("after getFromCombForall");
                    }

                    //PrintWriter cout = new PrintWriter (if(SCIA.debug_on) System.err, true);
                    //write_AllTNodes(cout, "    ", false);
                    //System.exit(0);
                    // to check: why subtree's matches are not updated with 
                    // user input at the subtree root
                }
            }
        }

	// un-highlighting the nodes and removing selection from the trees
	int selectionRow = tree.mainUI.schemaJTree1.getMinSelectionRow();
	tree.mainUI.schemaJTree1.removeSelectionRow(selectionRow);
	selectionRow = tree.mainUI.schemaJTree2.getMinSelectionRow();
	tree.mainUI.schemaJTree2.removeSelectionRow(selectionRow);

	anotherTree.mainUI.treesSplitPane.highlightNodes = null;
	anotherTree.mainUI.treesSplitPane.removeHighlight();
	anotherTree.mainUI.addMatchedLines();

	Thread.sleep(200);
	anotherTree.mainUI.treesSplitPane.paintImmediately(0, 0,
		    anotherTree.mainUI.treesSplitPane.getWidth(),
		    anotherTree.mainUI.treesSplitPane.getHeight());

        //cleanMatchingsWithCoreConceptsForAll(); 
    }

    public boolean hasDeepSubTree() {
        boolean res = false;

        if ((children != null) && !children.isEmpty()) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if ((child.children != null) && !child.children.isEmpty()) {
                    res = true;

                    break;
                }
            }
        }

        return res;
    }

    public boolean hasBigSubTree() {
        boolean res = false;

        if (subTreeSize > SCIA.BIG_SUBTREE_LOWER_LIMIT) {
            res = true;
        }

        return res;
    }

    public void resetFromCombForAll() {
        resetFromComb();

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.resetFromCombForAll();
                }
            }
        }
    }

    public void resetFromComb() {
        fromComb = null;
    }

    public void updateTreeFieldForAll(SchemaTree t) {
        updateTreeField(t);

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.updateTreeFieldForAll(t);
                }
            }
        }
    }

    public void updateTreeField(SchemaTree t) {
        tree = t;
    }

    public TNode getTNodeFromRowNumForAll(int num) {
        TNode tnode = getTNodeFromRowNum(num);

        if (tnode != null) {
            return tnode;
        } else {
            if (children != null) {
                for (int i = 0; i < children.size(); i++) {
                    TNode child = (TNode) children.get(i);

                    if (child != null) {
                        tnode = child.getTNodeFromRowNumForAll(num);

                        if (tnode != null) {
                            return tnode;
                        }
                    }
                }
            }
        }

        return null;
    }

    public TNode getTNodeFromRowNum(int num) {
        if (rowNum == num) {
            return this;
        }

        return null;
    }

    public void setRowNumForAll() {
        setRowNum();

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.setRowNumForAll();
                }
            }
        }
    }

    public void setRowNum() {
        rowNum = SchemaTree.currentRowNum;
        SchemaTree.currentRowNum++;
    }

    public boolean sortMatchings(List matchings) {
        boolean res = true;

        if ((matchings == null) || matchings.isEmpty()) {
            res = false;

            return res;
        }

        int size = matchings.size();
        int index = 0;
        Matching[] sorted = new Matching[size];

        /*
        for (int i = 0; i < size; i++) {
            Matching match = (Matching)matchings.get(i);

            if (getPath().equalsIgnoreCase("/bookstore/book/author")) {
                if(SCIA.debug_on) System.err.println(match.tnode.getPath() + " sim = " +
                                   match.sim);
            }

        }
        */
        for (int i = 0; i < size; i++) {
            double maxSim = 0;

            for (int j = 0; j < matchings.size(); j++) {
                if (((Matching) matchings.get(j)).sim >= maxSim) {
                    maxSim = ((Matching) matchings.get(j)).sim;
                    index = j;
                }
            }

            sorted[i] = (Matching) matchings.get(index);
            matchings.remove(index);
        }

        //copy the sorted back to matchings
        //matchings = new ArrayList(); 
        for (int i = 0; i < size; i++) {
            matchings.add(i, sorted[i]);

            Matching match = (Matching) matchings.get(i);
        }

        return res;
    }

    public PrintWriter write(PrintWriter stream, String headSpace,
        boolean writeMapping) {
        //stream.println("In TNode Write()");         
        stream.print("at height:  " + height + "  ");
        data.write(stream, headSpace);
        stream.print("\n");
        stream.print("                  " + headSpace);

        // write local multiple matches
        if ((fromComb != null) && (fromComb.matchingUnits != null) &&
                !fromComb.matchingUnits.isEmpty()) {
            stream.print(" =~= ");

            for (int i = 0; i < fromComb.matchingUnits.size(); i++) {
                MatchingUnit unit = (MatchingUnit) fromComb.matchingUnits.get(i);

                if (writeMapping) {
                    unit.sourceNode.matchedTimes++;
                }

                stream.print(unit.sourceNode.getPath());

                if (unit.operations != null) {
                    stream.print(" " + unit.operations);
                }

                if (unit.opConnectingNextUnit != null) {
                    stream.print(" " + unit.opConnectingNextUnit + " ");
                }

                //stream.println("     (sim: " + format.format(match.sim) + ")");
            }

            stream.print("\n");
        } else if (data.getTag().equalsIgnoreCase("Density")) {
            stream.print("NO MATCHINGUNITS \n");
        }

        if (userInputMatch) {
            for (int i = 0; i < numOfMatches; i++) {
                Matching inputMatch = (Matching) matchings.get(i);

                if (writeMapping) {
                    inputMatch.tnode.matchedTimes++;
                }

                stream.print(" =~= " + inputMatch.tnode.getPath());
                stream.println("     (sim: " + format.format(inputMatch.sim) +
                    ")");
                stream.print("\n");
            }

            stream.print("\n");
            stream.flush();

            return stream;
        }

        if (hasAncestorWithUserInput) {
            for (int i = 0; i < subTreeMatchings.size(); i++) {
                Matching match = (Matching) subTreeMatchings.get(i);

                if (writeMapping) {
                    match.tnode.matchedTimes++;
                }

                stream.print(" =~= " + match.tnode.getPath());
                stream.println("     (sim: " + format.format(match.sim) + ")");
                stream.print("\n");
            }

            stream.print("\n");
            stream.flush();

            return stream;
        }

        // if a node got multiple similar matches with realtive low similarity,
        // it might means there is no good match at all, throw them, but output 
        // information, allow user to pick up if they are good ones
        if (((fromComb != null) && (fromComb.sim > 0.15)) &&
                ((secFromComb != null) &&
                ((fromComb.sim - secFromComb.sim) < SCIA.DIFFERENCE_THRESHOLD)) &&
                (fromComb.sim < SCIA.MULTI_BAD_MATCHES_UPPER_SIM)) {
            stream.print(
                " Multiple low simiratity matches were found, but thrown: \n");
            stream.print("                  " + headSpace);
            stream.print(fromComb.tnode.getPath());
            stream.println("     (sim: " + format.format(fromComb.sim) + ")");
            stream.print("                  " + headSpace);
            stream.print(secFromComb.tnode.getPath());
            stream.println("     (sim: " + format.format(secFromComb.sim) +
                ")");
        } else if ((fromComb != null) && (fromComb.tnode != null) &&
                (fromComb.sim > 0.10)) {
            if (writeMapping) {
                if (fromComb.tnode == null) {
                    if (SCIA.debug_on) {
                        System.err.println("tnode = null");
                    }
                }

                fromComb.tnode.matchedTimes++;
            }

            stream.print(" =~= " + fromComb.tnode.getPath());

            /*
            if (!(fromComb.tnode.getPath()).equals(fromComb.tnode.
            getNormalizedPath())) {
                stream.print("  normalized to " + fromComb.tnode.
                getNormalizedPath());
            }
            */
            stream.print("\n");

            stream.print("                  " + headSpace);

            stream.println("     (sim: " + format.format(fromComb.sim) + ")");

            //stream.print("to check fromCOmb.tnode.matchedTimes");
            //stream.flush();
            if (fromComb.tnode.matchedTimes > 1) {
                stream.print("                  " + headSpace);
                stream.println("Please check, matched to multiple paths!");
            }

            stream.print("\n");

            //stream.print("to check secFromCOmb");
            //stream.flush();
            stream.print("                  " + headSpace);

            if (secFromComb != null) {
                if ((fromComb.sim - secFromComb.sim) < SCIA.DIFFERENCE_THRESHOLD) {
                    stream.print(" =~= " + secFromComb.tnode.getPath());
                    stream.print("\n");

                    if (secFromComb.sim > 1.0) {
                        secFromComb.sim = 1.0;
                    }

                    stream.print("                  " + headSpace);
                    stream.println("     (sim: " +
                        format.format(secFromComb.sim) + ")");
                    stream.print("                  " + headSpace);
                    stream.println("Please check the above similar matches.");
                }
            }
        }

        stream.print("\n");
        stream.flush();

        return stream;
    }

    public PrintWriter write_AllTNodes(PrintWriter stream, String headSpace,
        boolean writeMapping) {
        //stream.println ("In TNode Write_AllTNodes()");
        write(stream, headSpace, writeMapping);
        headSpace = headSpace + "    ";

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.write_AllTNodes(stream, headSpace, writeMapping);
                }
            }
        }

        return stream;
    }

    public void writeMapToFile(PrintWriter stream) {
        //if(SCIA.debug_on) System.err.println("In TNode writeMapToFile()");         
        if ((fromComb != null) && (fromComb.tnode != null)) {
            stream.print(fromComb.tnode.getPath() + " --> ");
            stream.print(getPath() + "  ");
            stream.print(format.format(fromComb.sim));
            stream.print("\n");
        }
    }

    public void writeAllMapToFile(PrintWriter stream) {
        //if(SCIA.debug_on) System.err.println("In TNode writeALlMapToFile()");
        writeMapToFile(stream);

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.writeAllMapToFile(stream);
                }
            }
        }
    }

    /**
     * Write the local n-to-1 match of this schema tree node
     * into a stream to an XML document with structure specified in
     * scia_mapping.xsd.
     @param stream The name of the PrintWriter to the XML document
     */
    public void writeLocal_n_to_1(PrintWriter stream) {
        stream.println("<local-n-to-1>");

        for (int i = 1; i <= fromComb.matchingUnits.size(); i++) {
            MatchingUnit unit = (MatchingUnit) fromComb.matchingUnits.get(i -
                    1);
            stream.println("<unit" + i + ">");
            stream.println("<sourcePath> " + unit.sourceNode.getPath() +
                "</sourcePath>");

            if (unit.operations != null) {
                stream.println("<operations> " + unit.operations +
                    "</operations>");
            }

            if (unit.opConnectingNextUnit != null) {
                stream.println("<combOperation> " + unit.opConnectingNextUnit +
                    " </combOperation>");
            }

            stream.println("</unit" + i + ">");
        }

        //condition
        if (fromComb.condition != null) {
            stream.println("<condition> " + fromComb.condition.toString() +
                " </condition>");
        }

        stream.println("<similarity> " + fromComb.sim + "</similarity>");
        stream.println("</local-n-to-1>");
    }

    /**
     * Write the basic 1-to-1 match, containing only a single  source path and
     * a similarity value, of this schema tree node
     * into a stream to an XML document with structure specified in
     * scia_mapping.xsd.
     @param stream The name of the PrintWriter to the XML document
     @param match The name of the Matching to be written
     */
    public void writeBasic_1_to_1(PrintWriter stream, Matching match) {
        stream.println("<sourcePath> " + match.tnode.getPath() +
            " </sourcePath>");
        stream.println("<similarity> " + format.format(match.sim) +
            " </similarity>");
    }

    /**
     * Write the 1-to-1 match of this schema tree node
     * into a stream to an XML document with structure specified in
     * scia_mapping.xsd.
     @param stream The name of the PrintWriter to the XML document
     */
    public void write_1_to_1(PrintWriter stream) {
        stream.println("<one-to-one>");
        stream.println("<sourcePath> " + fromComb.tnode.getPath() +
            " </sourcePath>");
        stream.println("<similarity> " + format.format(fromComb.sim) +
            " </similarity>");

        if (fromComb.operations != null) {
            stream.println("<operations> " + fromComb.operations +
                " </operations>");
        }

        if (fromComb.condition != null) {
            stream.println("<condition> " + fromComb.condition.toString() +
                " </condition>");
        }

        if (fromComb.groupAttrs != null) {
            stream.println("<groupAttrs>");

            for (int i = 0; i < fromComb.groupAttrs.size(); i++) {
                stream.println("<groupAttr> " +
                    ((TNode) fromComb.groupAttrs.get(i)).getPath() +
                    " </groupAttr>");
            }

            stream.println("</groupAttrs>");
        } else if (fromComb.aggregateOp != null) {
            stream.println("<aggregator> " + fromComb.aggregateOp +
                " </aggregator>");
        }

        stream.println("</one-to-one>");
    }

    /**
     * Write the mapping of this schema tree node
     * into a stream to an XML document with structure specified in
     * scia_mapping.xsd.
     @param stream The name of the PrintWriter to the XML document
     @param contextCheck The boolean value whether the mapping is computed
                using context check or not
     @param k The number of matches to be written
     */
    public void writeMapToXMLDoc(PrintWriter stream, boolean contextCheck, int k) {
        stream.println("<component>");

        stream.println("<targetPath> " + getPath() + " </targetPath>");

        //write constant match
        if (constantValue != null) {
            stream.println("<constant> " + constantValue + "</constant>");

            //write function output match
        } else if (functionOutput != null) {
            stream.println("<functionOutput> " + functionOutput +
                "</functionOutput>");

            // write local n-to-1 match
        } else if ((fromComb != null) && (fromComb.matchingUnits != null) &&
                !fromComb.matchingUnits.isEmpty()) {
            writeLocal_n_to_1(stream);

            //write global n-to-1 union matches
        } else if ((independentUnionMatches != null) &&
                !independentUnionMatches.isEmpty()) {
            stream.println("<global-n-to-1>");

            for (int i = 0; i < independentUnionMatches.size(); i++) {
                Matching match = (Matching) independentUnionMatches.get(i);

                if ((match != null) && (match.tnode != null)) {
                    write_1_to_1(stream);
                } else if ((match != null) && (match.matchingUnits != null)) {
                    writeLocal_n_to_1(stream);
                }
            }

            stream.println("</global-n-to-1>");
        } else if ((dependentUnionMatches != null) &&
                !dependentUnionMatches.isEmpty()) {
            stream.println("<global-n-to-1>");

            for (int i = 0; i < dependentUnionMatches.size(); i++) {
                Matching match = (Matching) dependentUnionMatches.get(i);

                if ((match != null) && (match.tnode != null)) {
                    write_1_to_1(stream);
                } else if ((match != null) && (match.matchingUnits != null)) {
                    writeLocal_n_to_1(stream);
                }
            }

            stream.println("</global-n-to-1>");

            // write one-to-one match
        } else if ((fromComb != null) && (fromComb.tnode != null) &&
                (fromComb.sim >= SCIA.SIGNIFICANT_SIM_THRESHOLD)) {
            write_1_to_1(stream);

            /* Handle non-user input 1-to-1 match during or after context check
               get at most the best and the second, since matchings with context
               check and matchings without context check are not fully combined,
               only the top two are extracted.*/
            if (!userInputMatch && contextCheck) {
                /* if a node got more than one similar matches with relative low
                   similarity, output them, allow user to pick up if they are
                   good
                */
                if ((secFromComb != null) &&
                        ((fromComb.sim - secFromComb.sim) < SCIA.DIFFERENCE_THRESHOLD)) {
                    if (secFromComb.sim > SCIA.LOWEST_SIGNIFICANT_SIM_THRESHOLD) {
                        stream.println("<match_2>");
                        writeBasic_1_to_1(stream, secFromComb);
                        stream.println("</match_2>");
                    }
                }

                //handle non-user input 1-to-1 match before context check
            } else if (!userInputMatch && !contextCheck) {
                if ((matchings != null) && (matchings.size() > 1)) {
                    int i = 1;
                    Matching m = (Matching) matchings.get(i);

                    while ((m != null) && (i < k) &&
                            (m.sim > SCIA.LOWEST_SIGNIFICANT_SIM_THRESHOLD)) {
                        int j = i + 1;
                        stream.println("<match_" + j + ">");
                        writeBasic_1_to_1(stream, m);
                        stream.println("</match_" + j + ">");
                        i++;

                        if (matchings.size() > i) {
                            m = (Matching) matchings.get(i);
                        } else {
                            m = null;
                        }
                    }
                }
            }
        }

        stream.println("</component>");
    }

    /**
     * Write the mapping of all the nodes in the subtree rooted at this
     * node into a stream to an XML document with structure specified in
     * scia_mapping.xsd.
     @param stream The name of the PrintWriter to the mapping XML document
     @param contextCheck The boolean value whether the mapping is computed
                using context check or not
     @param k The number of matches to be written
     */
    public void writeAllMapToXMLDoc(PrintWriter stream, boolean contextCheck,
        int k) {
        writeMapToXMLDoc(stream, contextCheck, k);

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.writeAllMapToXMLDoc(stream, contextCheck, k);
                }
            }
        }
    }

    /**
     * Write out matches of this node and all subtree nodes considering both
     * directions. E.g., (S1:e1, S2:e2) is
     * accepted only if S2:e2 can not find better ei in S1 from the other
     * direction of matching.
     *
     * @param stream The PrintWriter of the outpu to be sent
     * @param writeMapping The boolean value indicating whether to write
     * final mapping or not
     * @param tree1 The SchemaTree to which this SchemaTree is matched with
     *
     * @return the PrintWriter
     */
    public PrintWriter writeBothForAll(PrintWriter stream, String headSpace,
        boolean writeMapping, SchemaTree tree1) {
        writeBoth(stream, headSpace, writeMapping, tree1);
        headSpace = headSpace + "    ";

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.writeBothForAll(stream, headSpace, writeMapping, tree1);
                }
            }
        }

        return stream;
    }

    /**
     * Write out matches of this node considering both
     * directions. E.g., (S1:e1, S2:e2) is
     * accepted only if S2:e2 can not find better ei in S1 from the other
     * direction of matching.
     *
     * @param stream The PrintWriter of the outpu to be sent
     * @param writeMapping The boolean value indicating whether to write
     * final mapping or not
     * @param tree1 The SchemaTree to which this SchemaTree is matched with
     *
     * @return the PrintWriter
     */
    public PrintWriter writeBoth(PrintWriter stream, String headSpace,
        boolean writeMapping, SchemaTree tree1) {
        //stream.println("In TNode writeBoth()");         
        stream.print("at height:  " + height + "  ");
        data.write(stream, headSpace);
        stream.print("\n");
        stream.print("                  " + headSpace);

        if (userInputMatch) {
            for (int i = 0; i < matchings.size(); i++) {
                Matching match = (Matching) matchings.get(i);

                if (writeMapping) {
                    match.tnode.matchedTimes++;
                }

                stream.print(" =~= " + match.tnode.getPath());
                stream.println("     (sim: " + format.format(match.sim) + ")");
                stream.print("\n");
            }

            stream.print("\n");
            stream.flush();

            return stream;
        }

        if (hasAncestorWithUserInput) {
            for (int i = 0; i < subTreeMatchings.size(); i++) {
                Matching match = (Matching) subTreeMatchings.get(i);

                if (writeMapping) {
                    match.tnode.matchedTimes++;
                }

                stream.print(" =~= " + match.tnode.getPath());
                stream.println("     (sim: " + format.format(match.sim) + ")");
                stream.print("\n");
            }

            stream.print("\n");
            stream.flush();

            return stream;
        }

        if ((fromComb != null) &&
                (fromComb.sim > SCIA.OUTPUT_ACCEPT_THRESHOLD)) {
            if (userInputMatch ||
                    ((fromComb.tnode.fromComb != null) &&
                    (fromComb.tnode.fromComb.tnode == this)) ||
                    ((fromComb.tnode.secFromComb != null) &&
                    (fromComb.tnode.secFromComb.tnode == this) &&
                    ((fromComb.tnode.fromComb.sim -
                    fromComb.tnode.secFromComb.sim) < SCIA.DIFFERENCE_THRESHOLD)) ||
                    (fromComb.sim >= SCIA.NO_CARE_DIRECTION_THRESHOLD)) {
                if (writeMapping) {
                    fromComb.tnode.matchedTimes++;
                }

                stream.print(" =~= " + fromComb.tnode.getPath());

                stream.print("\n");

                stream.print("                  " + headSpace);

                if (fromComb.sim > 1.0) {
                    fromComb.sim = 1.0;
                }

                stream.println("     (sim: " + format.format(fromComb.sim) +
                    ")");

                //stream.print("to check fromCOmb.tnode.matchedTimes");
                //stream.flush();
                if (fromComb.tnode.matchedTimes > 1) {
                    stream.print("                  " + headSpace);
                    stream.println("Please check,  matched to multiple paths!");
                }

                stream.print("\n");

                //stream.print("to check secFromCOmb");
                //stream.flush();
                stream.print("                  " + headSpace);

                if ((secFromComb != null) &&
                        ((fromComb.sim - secFromComb.sim) < SCIA.DIFFERENCE_THRESHOLD) &&
                        (secFromComb.sim > SCIA.OUTPUT_ACCEPT_THRESHOLD)) {
                    if (userInputMatch ||
                            ((secFromComb.tnode.fromComb != null) &&
                            (secFromComb.tnode.fromComb.tnode == this)) ||
                            ((secFromComb.tnode.secFromComb != null) &&
                            (secFromComb.tnode.secFromComb.tnode == this) &&
                            ((secFromComb.tnode.fromComb.sim -
                            secFromComb.tnode.secFromComb.sim) < SCIA.DIFFERENCE_THRESHOLD)) ||
                            (secFromComb.sim >= SCIA.NO_CARE_DIRECTION_THRESHOLD)) {
                        stream.print(" =~= " + secFromComb.tnode.getPath());
                        stream.print("\n");

                        if (secFromComb.sim > 1.0) {
                            secFromComb.sim = 1.0;
                        }

                        stream.print("                  " + headSpace);
                        stream.println("     (sim: " +
                            format.format(secFromComb.sim) + ")");
                        stream.print("                  " + headSpace);
                        stream.println(
                            "Please check the above similar matches!");
                    }
                }

                //more matches, thirdFromComb for user input multiple matches 
            }
        }

        stream.print("\n");
        stream.flush();

        return stream;
    }

    public long getSubTreeSize() {
        subTreeSize = 1;

        if ((children != null) && !children.isEmpty()) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);
                subTreeSize = child.getSubTreeSize() + subTreeSize;
            }
        }

        return subTreeSize;
    }

    public void clearMatchedTimesForAll() {
        clearMatchedTimes();

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.clearMatchedTimesForAll();
                }
            }
        }
    }

    public void clearMatchedTimes() {
        matchedTimes = 0;
    }

    /* methods required for being tree nodes in JTree */
    public int getChildCount() {
        if ((children != null) && !children.isEmpty()) {
            return children.size();
        }

        return 0;
    }

    public TreeNode getChildAt(int index) {
        if ((children != null) && !children.isEmpty() &&
                (index < children.size())) {
            return (TNode) children.get(index);
        }

        return null;
    }

    public int getIndexOfChild(TNode child) {
        if ((children != null) && !children.isEmpty()) {
            for (int i = 0; i < children.size(); i++) {
                TNode tmp = (TNode) children.get(i);

                if (child == tmp) {
                    return i;
                }
            }
        }

        return -1;
    }

    public String toString() {
        String tag = data.getTag();
        String dt = data.dataType;
        String cardi = getCardinality();
        String desc = data.description;
        String res = tag;

        if (!data.nodeType.equals("compositor")) {
            res = res + " (";

            if ((dt != null) && !dt.equals("")) {
                res = res + dt;
            }

            if ((cardi != null) && !cardi.equals("")) {
                if (res.endsWith("(")) {
                    res = res + cardi;
                } else {
                    res = res + " " + cardi;
                }
            }

            if ((desc != null) && !desc.equals("")) {
                if (res.endsWith("(")) {
                    res = res + desc;
                } else {
                    res = res + " " + desc;
                }
            }

            res = res + ")";

            if (res.endsWith("()")) {
                res = res.substring(0, res.length() - 2);
            }
        }

        return res;
    }

    /* end of required methods for being tree nodes in JTree */
    public String getCardinality() {
        String cardi = null;
        String min = data.minOccurs;
        String max = data.maxOccurs;

        if ((min != null) && (max != null)) {
            if (min.equals("0") &&
                    (max.equals("*") || max.equals("unbounded"))) {
                cardi = "*";
            } else if (min.equals("1") &&
                    (max.equals("*") || max.equals("unbounded"))) {
                cardi = "+";
            } else if (min.equals("0") && max.equals("1")) {
                cardi = "?";
            } else if (min.equals("1") && max.equals("1")) {
                cardi = "1";
            } else {
                cardi = "minOccurs: " + min + "maxOccurs: " + max;
            }
        } else if ((min == null) && (max != null)) {
            cardi = "maxOccurs: " + max;
        } else if ((min != null) && (max == null)) {
            cardi = "minOccurs: " + min;
        }

        return cardi;
    }

    public void subTreeMatching(SchemaTree tree1, 
    //the schema tree where another subtree from
    TNode subTree1Root, 
    // the root of another subtree to match
    Hashtable nameSimTable, Hashtable pathSimTable, Hashtable descSimTable,
        Hashtable dataTypeSimTable, boolean interactive_on)
        throws Exception {
        TNode subTreeRoot = this;
        Hashtable resHash1 = new Hashtable();
        Hashtable resHash2 = new Hashtable();
        Model mod1 = SchemaTree.rf.createModel();
        Model mod2 = SchemaTree.rf.createModel();
        List initMap = new ArrayList();

        toRDFModelForAll(null, mod1, resHash1);

        if (SCIA.debug_on) {
            System.err.println("******resHash1 = " + resHash1);
        }

        subTree1Root.toRDFModelForAll(null, mod2, resHash2);

        if (SCIA.debug_on) {
            System.err.println("***** resHash2 = " + resHash2);
        }

        getInitMapFromLinguisticForAll(resHash1, resHash2, initMap);

        // add typeMap???

        /*
        java.util.List typeMap = new ArrayList();
        typeMap.add(new MapPair(parentElem, parentElem, 1.0));
        typeMap.add(new MapPair(leafElem, leafElem, 1.0));
        typeMap.add(new MapPair(rootElem, rootElem, 1.0));
        typeMap.add(new MapPair(CDATA, CDATA, 1.0));
        typeMap.add(new MapPair(PCDATA, PCDATA, 1.0));
        typeMap.add(new MapPair(INT, INT, 1.0));
        typeMap.add(new MapPair(FLOAT, FLOAT, 1.0));
        typeMap.add(new MapPair(DATE, DATE, 1.0));
        typeMap.add(new MapPair(STRING, STRING, 1.0));
        typeMap.add(new MapPair(NUMBER, NUMBER, 1.0));
        initMap.addAll(typeMap);
        */
        if (SCIA.debug_on) {
            System.err.println(initMap);
        }

        SFMatcher sfm = new SFMatcher();
        MapPair[] res = sfm.getMatch(mod1, mod2, initMap);
        MapPair.sortGroup(res);
        sfm.dump(res);

        Map mapFromGraph = MapPair.sortedCandidates(res, false);

        // update fromComb field, the best matching and
        // record the good matches of this subtree from the subtree1 
        // into subTreematchings
        // should record it is from which subtree in the source tree
        //matchings = null; 
        getFromCombForAll(mapFromGraph, nameSimTable, pathSimTable,
            descSimTable, dataTypeSimTable, resHash1, tree1, false, true,
            interactive_on);
    }

    /**
     * Compute the view expression in Quilt for this leaf node from the
     * acquired matching information, handling one local match at one time.
     *
     * @param documentFile The name of the data file, i.e., an XML document file
     * @param match The Matching of this leaf node to be handled
     * @param currentVar The name of the current variable being used in the
     * Quilt expression
     * @param headSpace The blank spaces for indentation
     * @param parentVar The name of the parrent variable being used in the
     * Quilt expression
     * @param closestAncestorMatchNode The closest ancestor match node, for
     * computing the end path from the closest ancestor match node to this
     * node's match node
     * @param lastStr The String as the last part of the generated view
     * for this leaf node from this match
     * @param parentAttributeStr The StringBuffer that holds the expression
     * the parent node's attrbute.
     * @param queryHead The StringBuffer that holds the head part of the
     * generated view expression.
     * udfDefs The StringBuffer that holds the expression part for needed user
     * defined functions in the final view expression.
     */
    public String leafViewFromLocalMatch(
        String documentFile, 
        // one local match of this leaf
        Matching match, 
        String currentVar, 
        String headSpace, 
        String parentVar,        
        // for computing the end path from the
        //closest ancestor match node to this 
        // node's match node
        TNode closestAncestorMatchNode, String lastStr,
        StringBuffer parentAttributeStr, StringBuffer queryHead,
        StringBuffer udfDefs) {

        String firstStr = "";
        String middleStr = "";
        String opsOnItself = "";
        String pathEnd = "";
        String tag = getTag();

        if ((parent != null) && parent.data.nodeType.equals("compositor")) {
            if (realParent == null) {
                realParent = getRealParent();
            }
        }

        // 1:k local match where k > 1

        /* A MatchingUnit is a basic unit in a local 1:k match.
           It has a source node, a list of operations on this source node,
           and an operation for this source node and the next source node
           E.g., for density = count/(area/5000), there are 2 matching
           units, 1) (count, null, div) and 2) (area, div (5000), null)
        */
        if (data.nodeType.equals("compositor")) {
            return "";
        }

        /* check if the match's context is consistent with
           closestAncestorMatchNode

           for leaf nodes with single match, but has ancestor with multiple
           matches

           if not consistent, skip it by returning an empty string
        */
        if ((match != null) && (match.tnode != null) &&
                (match.condition == null) && // should not skip conditional match
                (closestAncestorMatchNode != null)) {
            if (!match.tnode.isDecendentOf(closestAncestorMatchNode)) {
                //if(SCIA.debug_on) System.err.println("#####skipped");
                return "";
            }
        }

        if (data.nodeType.equals("attribute")) {
            if ((match.matchingUnits != null) &&
                    !match.matchingUnits.isEmpty()) {
                // assume the k nodes are siblings
                // assume no condition involved
                parentAttributeStr = parentAttributeStr.append("\n" +
                        headSpace + tag + " = ");

                String endParenthesis = "";
                String openParenthesis = "";

                for (int i = 0; i < match.matchingUnits.size(); i++) {
                    MatchingUnit unit = (MatchingUnit) match.matchingUnits.get(i);

                    if (SCIA.debug_on) {
                        System.err.println("unit =" + unit);
                    }

                    String opWithNextNode = "";

                    if (unit.operations != null) {
                        opsOnItself = unit.operations;
                    }

                    if (unit.opConnectingNextUnit != null) {
                        opWithNextNode = unit.opConnectingNextUnit;

                        if (SCIA.debug_on) {
                            System.err.println("opWithNextNode =" +
                                opWithNextNode);
                        }

                        openParenthesis = " (";
                        endParenthesis = endParenthesis + ")";
                    } else {
                        openParenthesis = "";
                    }

                    // to change, allow non-siblings
                    if (unit.sourceNode.data.nodeType.equals("attribute")) {
                        pathEnd = "/@" + unit.sourceNode.data.getTag();
                    } else {
                        pathEnd = "/" + unit.sourceNode.data.getTag();
                    }

                    parentAttributeStr = parentAttributeStr.append(parentVar +
                            pathEnd + "/text() " + opsOnItself +
                            opWithNextNode + openParenthesis);
                }

                parentAttributeStr = parentAttributeStr.append(endParenthesis);
            } else { // attribute has 1:1 local match

                if (match.condition == null) {
                } else { // condition is involved
                }

                pathEnd = match.tnode.getPath(closestAncestorMatchNode);

                parentAttributeStr = parentAttributeStr.append("\n" +
                        headSpace + tag + " = ");

                if (match.operations != null) {
                    String fName = match.operations;
                    String fImport = (String) SCIA.udfImportTable.get(fName);
                    String fDef = (String) SCIA.udfDefTable.get(fName);

                    if ((fImport != null) && (fDef != null)) {
                        //get user defined function definitions
                        queryHead = queryHead.append(fImport);
                        udfDefs = udfDefs.append(fDef);
                        opsOnItself = fName;
                        parentAttributeStr = parentAttributeStr.append(opsOnItself +
                                "(" + parentVar + pathEnd + ")");

                        return "";
                    }

                    if (SCIA.debug_on) {
                        System.err.println(tag + "'s op name =" + fName +
                            " queryhead = " + queryHead + " udfDefs = " +
                            udfDefs);
                    }
                }

                parentAttributeStr = parentAttributeStr.append(parentVar +
                        pathEnd + "/text() " + opsOnItself);
            }

            // attribute doesn't contribute to middleStr for its parent
            // it only add to parent's attributeStr
            return "";

            // for leaf element, without any attribute
        } else if (!data.nodeType.equals("attribute")) {
            if (SCIA.debug_on) {
                System.err.println("not attribute");
            }

            //local multiple match
            if ((match.matchingUnits != null) &&
                    !match.matchingUnits.isEmpty()) {
                if (SCIA.debug_on) {
                    System.err.println("not attribute, local multiple");
                }

                // assume the k nodes are siblings
                // assume no condition involved
                // no FOR clause
                firstStr = "\n" + headSpace + "<" + tag + "> ";

                String endParenthesis = "";
                String openParenthesis = "";

                for (int i = 0; i < match.matchingUnits.size(); i++) {
                    MatchingUnit unit = (MatchingUnit) match.matchingUnits.get(i);

                    if (SCIA.debug_on) {
                        System.err.println("unit =" + unit);
                    }

                    String opWithNextNode = "";

                    if (unit.operations != null) {
                        opsOnItself = unit.operations;
                    }

                    if (unit.opConnectingNextUnit != null) {
                        opWithNextNode = unit.opConnectingNextUnit;

                        if (SCIA.debug_on) {
                            System.err.println("opWithNextNode =" +
                                opWithNextNode);
                        }

                        openParenthesis = " (";
                        endParenthesis = endParenthesis + ")";
                    } else {
                        openParenthesis = "";
                    }

                    if (unit.sourceNode.data.nodeType.equals("attribute")) {
                        pathEnd = "/@" + unit.sourceNode.data.getTag();
                    } else {
                        pathEnd = "/" + unit.sourceNode.data.getTag();
                    }

                    middleStr = middleStr + parentVar + pathEnd + "/text() " +
                        opsOnItself + opWithNextNode + openParenthesis;
                }

                middleStr = middleStr + endParenthesis;

                return firstStr + middleStr + lastStr;
            } else if ((match.groupAttrs == null) &&
                    (match.aggregateOp == null) && (realParent != null) &&
                    (realParent.fromComb != null) &&
                    (realParent.fromComb.groupAttrs != null)) {
                /* leaf element has only 1:1 non-aggregation local match
                   and its parent has grouping type match
                */
                /* if it is a group attribute, a var named as the source node tag
                   is bound already */
                String sourceVar = "$" + fromComb.tnode.getTag();

                if(SCIA.debug_on) 
                    System.err.println("In leafView,parent has grouping type match");
               
                if (realParent.fromComb.groupAttrs != null) {
                    // is a group attribute
                    if (realParent.fromComb.groupAttrs.contains(fromComb.tnode)) {
                        firstStr = "\n" + headSpace + "<" + tag + "> ";
                        middleStr = sourceVar + "/text()";
                    } else { // not a group attribute
                        firstStr = "\n" + headSpace + "FOR " + currentVar +
                            " IN " + parentVar + "/" + fromComb.tnode.getTag();

                        firstStr = firstStr + " RETURN \n";

                        // open tag
                        firstStr = firstStr + headSpace + "<" + tag + "> ";
			if (match.operations != null) {
			    String fName = match.operations;
			    String fImport = (String) SCIA.udfImportTable.get(fName);
			    String fDef = (String) SCIA.udfDefTable.get(fName);

			    if ((fImport != null) && (fDef != null)) {
				//get user defined function definitions
				queryHead = queryHead.append(fImport);
				udfDefs = udfDefs.append(fDef);
				if (SCIA.debug_on) {
				    System.err.println(tag + "'s op name =" + fName +
						       " queryhead = " + queryHead + " udfDefs = " +
						       udfDefs);
				}
				opsOnItself = fName;
				middleStr = opsOnItself + "(" + currentVar + "/text())";
			    } else {
				opsOnItself = match.operations;
				middleStr = currentVar + "/text() " + opsOnItself;
			    }

			} else {
			    middleStr = currentVar + "/text() ";
			}

                    }
                }

                return firstStr + middleStr + lastStr;
            } else if ((match.groupAttrs == null) &&
                    (match.aggregateOp == null) && (realParent != null) &&
                    (((realParent.fromComb != null) &&
                    (realParent.fromComb.groupAttrs == null) &&
                    (realParent.fromComb.aggregateOp == null)) ||
                    (realParent.fromComb == null))) {
                /* leaf element has only 1:1 non-aggregation local match
                   and its parent does not have grouping type match, or
                   parent has no match
                */
                if ((match.condition == null) && (match.conditionStr == null) &&
                        (match.tnode != null)) {
                    pathEnd = match.tnode.getPath(closestAncestorMatchNode);

                    firstStr = "\n" + headSpace + "FOR " + currentVar + " IN " +
                        parentVar + pathEnd;
                } else if ((match.conditionStr != null) &&
                        (match.condition == null)) { // only conditionStr exists

                    // for books_book --> books_author
                    if (SCIA.debug_on) {
                        System.err.println(
                            "conditionStr != null & condition = null");
                    }

                    if ((parent.fromComb != null) &&
                            !match.tnode.isDecendentOf(parent.fromComb.tnode)) {
                        // find the connecting variable
                        TNode matchParent = match.tnode.parent;

                        if (matchParent.parent.parent == null) {
                            // match parent is root 
                            firstStr = "\n" + headSpace + "FOR " + currentVar +
                                " IN " + " document(\"" + documentFile + "\")" +
                                "//" + tag + match.conditionStr;
                        }
                    }
                } else if (match.condition != null) {
                    // formated condition exists
                    // only allow condition to be involved with 1:1 match now
                    //match path is decendent of parent match path?
                    if ((parent.fromComb != null) &&
                            !match.tnode.isDecendentOf(parent.fromComb.tnode)) {
                        // find the connecting variable
                        TNode matchParent = match.tnode.parent;

                        if (matchParent == match.condition.matchNeighborNode.parent) {
                            if (data.nodeType.equals("attribute")) {
                                pathEnd = "/@" + match.tnode.data.getTag();
                            } else {
                                pathEnd = "/" + match.tnode.data.getTag();
                            }

                            String connectVar = "$" +
                                matchParent.data.getTag();
                            firstStr = "\n" + headSpace + "FOR " + connectVar +
                                " IN " + " document(\"" + documentFile + "\")" +
                                "//" + matchParent.data.getTag();
                            firstStr = firstStr + "\n" + headSpace + "    " +
                                "FOR " + currentVar + " IN " + connectVar +
                                pathEnd;

                            String matchNeighborPathEnd = "";
                            String workingNeighborPathEnd = "";

                            if (match.condition.matchNeighborNode.data.nodeType.equals(
                                        "attribute")) {
                                matchNeighborPathEnd = "@" +
                                    match.condition.matchNeighborNode.data.getTag();
                            } else {
                                matchNeighborPathEnd = match.condition.matchNeighborNode.data.getTag();
                            }

                            if (match.condition.workingNeighborNode.data.nodeType.equals(
                                        "attribute")) {
                                workingNeighborPathEnd = "@" +
                                    match.condition.workingNeighborNode.data.getTag();
                            } else {
                                workingNeighborPathEnd = match.condition.workingNeighborNode.data.getTag();
                            }

                            firstStr = firstStr + "\n" + headSpace + "    " +
                                "WHERE " + connectVar + "/" +
                                matchNeighborPathEnd + " = " + parentVar + "/" +
                                workingNeighborPathEnd;
                        }
                    }
                }

                firstStr = firstStr + " RETURN \n";

                // open tag
                firstStr = firstStr + headSpace + "<" + tag + "> ";

                if (match.operations != null) {
                    String fName = match.operations;
                    String fImport = (String) SCIA.udfImportTable.get(fName);
                    String fDef = (String) SCIA.udfDefTable.get(fName);

                    if ((fImport != null) && (fDef != null)) {
                        //get user defined function definitions
                        queryHead = queryHead.append(fImport);
                        udfDefs = udfDefs.append(fDef);
                        if (SCIA.debug_on) {
                            System.err.println(tag + "'s op name =" + fName +
                                     " queryhead = " + queryHead + " udfDefs = " +
                                     udfDefs);
                        }
                        opsOnItself = fName;
                        middleStr = opsOnItself + "(" + currentVar + "/text())";
                    } else {
                        opsOnItself = match.operations;
                        middleStr = currentVar + "/text() " + opsOnItself;
                    }

                } else {
                    middleStr = currentVar + "/text() ";
                }

                return firstStr + middleStr + lastStr;

            } else if (match.aggregateOp != null) {
                if (SCIA.debug_on) {
                    System.err.println("not attribute, aggregateOp != null");
                }

                // leaf element has 1:1 local aggregation match

                /* get grouping condition from its parent which has a grouping
                   type match, grouping condition is decided using the grouping
                   attributes specified by users or estimated from constraints
                */
                String cond = realParent.fromComb.groupCondForChild;

                // open tag
                firstStr = headSpace + "<" + tag + "> ";

                if ((match.aggregateOp != null) && (cond != null)) {
                    middleStr = match.aggregateOp + "(" + currentVar + cond +
                        "/" + match.tnode.getTag();

                    if (match.operations != null) {
                        middleStr = middleStr + " " + match.operations;
                    }

                    middleStr = middleStr + ")";

                    if (SCIA.debug_on) {
                        System.err.println("middleStr = " + middleStr);
                    }
                }

                return firstStr + middleStr + lastStr;
            } else if (match.groupAttrs != null) {
                //reasonable to group a leaf node???
            }
        } // end for leaf element

        return null;
    }

    public String generateView(String documentFile, String parentVar,
        String headSpace, Matching currentAncestorMatch, // for dependent 
                                                         // union matches 
        boolean isLastChildMatched, StringBuffer parentAttributeStr,
        StringBuffer queryHead, StringBuffer udfDefs) {
        String firstStr = headSpace;
        String middleStr = headSpace;
        String firstAndMiddleStr = "";
        StringBuffer attributeStr = new StringBuffer();

        String tag = getTag();

        // end tag, for the last child, "," is not needed 
        String lastStr = "\n" + headSpace + "</" + tag + ">\n";

        //if(SCIA.debug_on) System.err.println("handling " + tag);
        if (!isLastChildMatched) {
            // for non-last child, ",' is needed after end tag to speparate
            // different FOR clauses
            lastStr = "\n" + headSpace + "</" + tag + ">,\n";            

            //if(SCIA.debug_on) System.err.println("\n not last child matched\n");
        } else {
            //if(SCIA.debug_on) System.err.println("\n is last child matched\n");
        }

        // clean tag for avoiding illegal variable name in xquery, 
        // e.g. $seq-data is not accepted, a simple way is replace '-' with '_'
        String cleanTag = tag;
        cleanTag = cleanTag.replace('-', '_');

        String currentVar = "$" + cleanTag;

        //if(SCIA.debug_on) System.err.println("cleanTag is " + cleanTag);
        String str = "";

        // for non-root leaf
        if ((parent != null) && ((children == null) || children.isEmpty())) {
            //if(SCIA.debug_on) System.err.println("  it is a leaf element");
            // if a leaf element has no match, do nothing
            if (constantValue != null) { // for 0:1 match, has constant value
                firstStr = "\n" + headSpace + "<" + tag + "> ";
                middleStr = constantValue;
                str = firstStr + middleStr + lastStr;
            } else if ((independentUnionMatches != null) &&
                    !independentUnionMatches.isEmpty()) {
                //if(SCIA.debug_on) System.err.println("It has independentUnionMatches");
                for (int i = 0; i < independentUnionMatches.size(); i++) {
                    String lastStr1 = lastStr;
                    Matching match = (Matching) independentUnionMatches.get(i);

                    //if(SCIA.debug_on) System.err.println(i + "th is " + match.tnode.getPath());
                    //even isLastChildMatched, "," is needed except for the 
                    //last one in the unionMatches
                    if (i != (independentUnionMatches.size() - 1)) {
                        //if(SCIA.debug_on) System.err.println("lastStr = " + lastStr);
                        if (lastStr.endsWith(">\n")) {
                            int index = lastStr.indexOf(">");
                            lastStr1 = lastStr.substring(0, index) + ">,\n";

                            //if(SCIA.debug_on) System.err.println("new lastStr = " + lastStr1);
                        }

                        //System.exit(0);
                    }

                    str = str +
                        leafViewFromLocalMatch(documentFile, match, currentVar,
                            headSpace, parentVar, currentAncestorMatch.tnode,
                            lastStr1, parentAttributeStr, queryHead, udfDefs);
                }
            } else if ((dependentUnionMatches != null) &&
                    !dependentUnionMatches.isEmpty()) {
                //if(SCIA.debug_on) System.err.println("It has dependentUnionMatches");
                for (int i = 0; i < dependentUnionMatches.size(); i++) {
                    Matching match = (Matching) dependentUnionMatches.get(i);

                    if (SCIA.debug_on) {
                        System.err.println(i + "th match is " +
                            match.tnode.getPath());
                    }

                    // follow the ancestor match being handled if ancestor has 
                    // independent union matches 
                    //if(SCIA.debug_on) System.err.println("current ancestor match = " +
                    //                currentAncestorMatch.tnode.getPath());
                    if (match.tnode.isDecendentOf(currentAncestorMatch.tnode)) {
                        //if(SCIA.debug_on) System.err.println(" match is child of current ancestor match"); 
                    }

                    if ((currentAncestorMatch != null) && (match != null) &&
                            match.tnode.isDecendentOf(
                                currentAncestorMatch.tnode)) {
                        //if(SCIA.debug_on) System.err.println(" found the match following " +
                        //            "ancestor's current match");
                        str = leafViewFromLocalMatch(documentFile, match,
                                currentVar, headSpace, parentVar,
                                currentAncestorMatch.tnode, lastStr,
                                parentAttributeStr, queryHead, udfDefs);
                    }
                }
            } else if ((fromComb != null) && (fromComb.groupAttrs == null) &&
                    (fromComb.aggregateOp == null)) {
                Matching match = fromComb;
                str = leafViewFromLocalMatch(documentFile, match, currentVar,
                        headSpace, parentVar, currentAncestorMatch.tnode,
                        lastStr, parentAttributeStr, queryHead, udfDefs);

                //if(SCIA.debug_on) System.err.println("str = " + str);
            } else if ((fromComb != null) && (fromComb.aggregateOp != null)) {
                if (SCIA.debug_on) {
                    System.err.println("fromComb.aggregateOp != null");
                }

                //use parentVar for aggregation instead of currentVar
                Matching match = fromComb;
                str = leafViewFromLocalMatch(documentFile, match, parentVar,
                        headSpace, parentVar, currentAncestorMatch.tnode,
                        lastStr, parentAttributeStr, queryHead, udfDefs);

                //if(SCIA.debug_on) System.err.println("str = " + str);
            } else {
                if (SCIA.debug_on) {
                    System.err.println("     It has no match!");
                }
            }

            return str;
        }

        // for root
        if (parent == null) {
            //if(SCIA.debug_on) System.err.println("  it is root");
            firstStr = headSpace + "<" + tag + ">"; // open tag
            currentVar = " document(\"" + documentFile + "\")/";

            middleStr = middleStr +
                getChildrenView(documentFile, currentVar, headSpace, fromComb,
                    null, queryHead, udfDefs);

            // assume root has a single match,
            // for its children, no check for ancestor's match being handled, 
            // since only one exists
            firstAndMiddleStr = firstStr + middleStr;
        }

        // for non-root parent
        if ((parent != null) && (children != null) && !children.isEmpty()) {
            //if(SCIA.debug_on) System.err.println("  it is non-root parent");
            if ((independentUnionMatches != null) &&
                    !independentUnionMatches.isEmpty()) {
                //if(SCIA.debug_on) System.err.println("It has independentUnionMatches");
                for (int i = 0; i < independentUnionMatches.size(); i++) {
                    //if(SCIA.debug_on) System.err.println("size=" + independentUnionMatches.size());
                    Matching match = (Matching) independentUnionMatches.get(i);

                    /*if(SCIA.debug_on) System.err.println(i + "th is" + match.tnode.getPath());

                    if (currentAncestorMatch == null)
                        if(SCIA.debug_on) System.err.println("CAM = null");
                    */
                    String pathEnd = match.tnode.getPath(currentAncestorMatch.tnode);

                    /*if (pathEnd != null)
                        if(SCIA.debug_on) System.err.println("path between " +
                                           match.tnode.getPath() +
                                           " and " +
                                           currentAncestorMatch.tnode.getPath() +
                                           " = " + pathEnd);
                    else
                        if(SCIA.debug_on) System.err.println("pathEnd is null");
                    */
                    firstStr = "\n" + headSpace + "FOR " + currentVar + " IN " +
                        parentVar + pathEnd;
                    firstStr = firstStr + " RETURN \n";

                    // open tag
                    firstStr = firstStr + headSpace + "<" + tag;

                    middleStr = middleStr +
                        getChildrenView(documentFile, currentVar, headSpace,
                            match, attributeStr, queryHead, udfDefs);

                    // not the last one, add the end tag
                    String lastStr1 = lastStr;

                    if (i < (independentUnionMatches.size() - 1)) {
                        // add "," at end except for the last in the unionMatches
                        if (lastStr.endsWith(">\n")) {
                            int index = lastStr.indexOf(">");
                            lastStr1 = lastStr.substring(0, index) + ">,\n";

                            //if(SCIA.debug_on) System.err.println("new lastStr = " + lastStr1);
                        }

                        middleStr = middleStr + lastStr1;
                    }

                    firstAndMiddleStr = firstAndMiddleStr + firstStr +
                        attributeStr.toString() + "> \n" + middleStr;
                    middleStr = headSpace;

                    attributeStr = new StringBuffer(); // reset attributeStr!!!
                }
            } else if ((dependentUnionMatches != null) &&
                    !dependentUnionMatches.isEmpty()) {
                for (int i = 0; i < dependentUnionMatches.size(); i++) {
                    //if(SCIA.debug_on) System.err.println("size=" + dependentUnionMatches.size());
                    Matching match = (Matching) dependentUnionMatches.get(i);

                    //if(SCIA.debug_on) System.err.println(i + "th is" + match.tnode.getPath());

                    /*if (match.tnode.isDecendentOf(currentAncestorMatch.tnode)) {
                        if(SCIA.debug_on) System.err.println(" match is child of current " +
                                           " ancestor match");
                    }
                    if (currentAncestorMatch == null)
                        if(SCIA.debug_on) System.err.println("CAM = null");
                    */
                    if ((currentAncestorMatch != null) && (match != null) &&
                            match.tnode.isDecendentOf(
                                currentAncestorMatch.tnode)) {
                        if (SCIA.debug_on) {
                            System.err.println(" found the match following " +
                                "ancestor's current match");
                        }

                        String pathEnd = match.tnode.getPath(currentAncestorMatch.tnode);

                        /*if (pathEnd != null)
                            if(SCIA.debug_on) System.err.println("path between " +
                                               match.tnode.getPath() +
                                               " and " +
                                               currentAncestorMatch.tnode.
                                               getPath() +
                                               " = " + pathEnd);
                        else
                            if(SCIA.debug_on) System.err.println("pathEnd is null");
                        */
                        firstStr = "\n" + headSpace + "FOR " + currentVar +
                            " IN " + parentVar + pathEnd;
                        firstStr = firstStr + " RETURN \n";

                        // open tag
                        firstStr = firstStr + headSpace + "<" + tag;

                        middleStr = middleStr +
                            getChildrenView(documentFile, currentVar,
                                headSpace, match, attributeStr, queryHead,
                                udfDefs);

                        firstAndMiddleStr = firstAndMiddleStr + firstStr +
                            attributeStr.toString() + "> \n" + middleStr;
                        middleStr = headSpace;

                        // reset attributeStr!!!
                        attributeStr = new StringBuffer();
                    }
                }

                // has local 1:1 non-grouping match
            } else if ((fromComb != null) && (fromComb.groupAttrs == null) &&
                       (fromComb.aggregateOp == null)) { 
                /* to do: check hasDecendentMatchedFollow
                   (CurrentAncestorMatch.tnode)
                   this should be checked for when union matches exist too
                */
                if ((fromComb.conditionStr == null) &&
                        (fromComb.condition == null)) { // no condition

                    if (fromComb.tnode.isDecendentOf
                        (currentAncestorMatch.tnode)) {
                        if (fromComb.isCrossLevelGroupAttr == true) {
                        
                            firstStr = "\n" + headSpace + "FOR " + currentVar +
                                " IN DISTINCT ";
                        } else {
                            firstStr = "\n" + headSpace + "FOR " + currentVar +
                                " IN ";
                        }
                        firstStr = firstStr + parentVar + "/" + 
                            fromComb.tnode.getTag();
                    } else {
                        //check if hasDecendentMatched, if not return "", else
                        // check if hasDecendentMatchedFollow(CAM.tnode)
                        if (!hasDecendentMatched()) {
                            return "";
                        } else if (!hasDecendentMatchedFollow(
                                    currentAncestorMatch.tnode)) {
                            return "";
                        }
                    }
                } else if ((fromComb.conditionStr != null) &&
                        (fromComb.condition == null)) {
                    // only conditionStr exists

                    /* e.g:  books_book --> books_author:
                       bib/book --> books/author/book
                       get all books authored by this author
                    */

                    //if(SCIA.debug_on) System.err.println("conditionStr!=null & condition=null");
                    if ((parent.fromComb != null) &&
                            !fromComb.tnode.isDecendentOf(parent.fromComb.tnode)) {
                        // find the connecting variable
                        TNode fromCombParent = fromComb.tnode.parent;

                        if ((fromCombParent != null) &&
                                (fromCombParent.parent == null)) {
                            // fromComb parent is root 
                            firstStr = "\n" + headSpace + "FOR " + currentVar +
                                " IN " + " document(\"" + documentFile + "\")" +
                                "//" + tag + fromComb.conditionStr;
                        }
                    }
                }

                firstStr = firstStr + " RETURN \n";

                // open tag
                firstStr = firstStr + headSpace + "<" + tag;
                middleStr = middleStr +
                    getChildrenView(documentFile, currentVar, headSpace,
                        fromComb, attributeStr, queryHead, udfDefs);

                // for children, no check for ancestor's match being handled, 
                // since only one exists
                firstAndMiddleStr = firstStr + attributeStr + "> \n" +
                    middleStr;

                // has local 1:1 grouping type match
            } else if ((fromComb != null) && (fromComb.groupAttrs != null) &&
                       (fromComb.groupAttrs.size() > 0)) {

                // non-cross level grouping
                if (fromComb.crossLevelGrouping == false) {             
                    TNode attrNode1 = (TNode) fromComb.groupAttrs.get(0);
                    String var1 = "$" + attrNode1.getTag();
                    firstStr = "\n" + headSpace + "FOR " + var1 + 
                        " IN DISTINCT" +
                        " document(\"" + documentFile + "\")" + "//" +
                        attrNode1.getTag() + "\n" + headSpace + "LET " +
                        currentVar + ":= " + " document(\"" + documentFile + 
                        "\")" + "//" + fromComb.tnode.getTag() + "[" + 
                        attrNode1.getTag() + " = " + var1 + "]";

                    if (fromComb.groupAttrs.size() > 1) {
                        String grpCond = "[";

                        for (int i = 1; i < fromComb.groupAttrs.size(); i++) {
                            TNode attrNode2 = (TNode) fromComb.groupAttrs.
                                get(i);
                            String var2 = "$" + attrNode2.getTag();
                            grpCond = grpCond + attrNode2.getTag() + 
                                " = " + var2;
                            firstStr = firstStr + "\n" + headSpace + "FOR " + 
                                var2 + " IN DISTINCT " + currentVar + "/" +
                                attrNode2.getTag();

                            if (i < (fromComb.groupAttrs.size() - 1)) {
                                grpCond = grpCond + " and ";
                            }
                        }

                        fromComb.groupCondForChild = grpCond + "]";
                    }

                    firstStr = firstStr + " RETURN \n";

                    // open tag
                    firstStr = firstStr + headSpace + "<" + tag;
                    middleStr = middleStr +
                        getChildrenView(documentFile, currentVar, headSpace,
                                   fromComb, attributeStr, queryHead, udfDefs);

                    // for children, no check for ancestor's match being 
                    // handled, since only one exists
                    firstAndMiddleStr = firstStr + attributeStr + "> \n" +
                        middleStr;

                    //Handle cross level grouping
                } else {                  

                    /* Get grouping condition
                    */                    
                        String cond = "";
                    TNode grpAttrNode = (TNode) fromComb.groupAttrs.get(0);   
                    // if the 1st groupAttr is composite element, assume by 
                    // default all children are the real grouping attributes.
                    if (grpAttrNode.data.nodeType.equals("parentElem")) {

                        for (int i = 0; i < grpAttrNode.children.size(); i++) {
                            TNode child = (TNode)grpAttrNode.children.get(i);
                            cond = cond +
                                grpAttrNode.getTag() + "/" + 
                                child.getTag() + " = " + 
                                parentVar + "/" + child.getTag();
                            if (i != grpAttrNode.children.size() - 1){
                                cond = cond + " AND ";
                            } 
                        }
                    } else {
                        for (int i = 0; i < fromComb.groupAttrs.size(); i++) {
                            grpAttrNode = (TNode)fromComb.groupAttrs.get(i);
                            cond = cond + grpAttrNode.parent.getTag();
                            if (grpAttrNode.data.nodeType.equals("attribute")) {
                                cond = cond + "/@";
                            } else {
                                cond = cond + "/";
                            }
                            cond = cond +  grpAttrNode.getTag() + " = " + 
                                parentVar;
                            if (grpAttrNode.data.nodeType.equals("attribute")) {
                                cond = cond + "/@";
                            } else {
                                cond = cond + "/";
                            }                            
                            cond = cond + grpAttrNode.getTag();
                            if (i != fromComb.groupAttrs.size() - 1){
                                cond = cond + " AND ";
                            } 
                        }

                    }
                    fromComb.conditionStr = cond;
                    if (SCIA.debug_on) {
                        System.err.println(getTag() + " condition = " + cond);
                    }

                    
                    firstStr = "\n" + headSpace + "FOR " + currentVar + 
                        " IN DISTINCT" +
                        " document(\"" + documentFile + "\")" + "//" +
                        fromComb.tnode.getTag() + "[" + fromComb.conditionStr + 
                        "] RETURN \n";

                    // open tag
                    firstStr = firstStr + headSpace + "<" + tag;
                    middleStr = middleStr +
                        getChildrenView(documentFile, currentVar, headSpace,
                             fromComb, attributeStr, queryHead, udfDefs);
                    if (SCIA.debug_on) {
                        System.err.println(getPath() +
                            ": middleStr = " + middleStr);
                    }
                    firstAndMiddleStr = firstStr + attributeStr + "> \n" +
                        middleStr;
                }
            } else { // non-matched parent node

                // if it has no decendent nodes matched, then skip it
                if (!hasDecendentMatched()) {
                    if (SCIA.debug_on) {
                        System.err.println(getPath() +
                            " has no decendent matched");
                    }

                    return ""; // return an empty string
                } else { // has decendent nodes matched

                    if (SCIA.debug_on) {
                        System.err.println(getPath() +
                            " has decendent matched");
                    }

                    if (data.nodeType.equals("compositor")) {
                        return getChildrenView(documentFile, parentVar,
                            headSpace, currentAncestorMatch, attributeStr,
                            queryHead, udfDefs);
                    }

                    /* but has no decendent nodes matched to decendent of
                       currentAncestorMatch.tnode

                       when union matches exist, this is important to deal with
                       children

                       E.g.,
                       bsml/feature -> seq/segment,
                       bsml/segment ->seq/segment,
                       bsml/feature/.../start-pos ->seq/segment/seg-start
                       we don't want
                       "FOR $segment IN .../segment RETURN <Segment> </Segment>"
                    */
                    if (!hasDecendentMatchedFollow(currentAncestorMatch.tnode)) {
                        if (SCIA.debug_on) {
                            System.err.println(getPath() +
                                " has no decendent matched to " +
                                " decendent of " +
                                " currentAncestorMatch.tnode");
                        }

                        //System.exit(0);
                        return ""; // return an empty string
                    } else {
                        if (SCIA.debug_on) {
                            System.err.println("#######" + getPath() +
                                " has decendent matched to " +
                                " decendent of " +
                                " currentAncestorMatch.tnode.getPath()");
                        }
                    }

                    /* if it has decendent nodes matched,
                       and has decendent nodes matched to decendent of
                       currentAncestorMatch.tnode
                    */
                    // open tag             
                    firstStr = "\n" + headSpace + "<" + tag;

                    // skip this non-matched node in path, e.g. $author//f_name
                    currentVar = parentVar + "/";
                    middleStr = middleStr +
                        getChildrenView(documentFile, currentVar, headSpace,
                            currentAncestorMatch, attributeStr, queryHead,
                            udfDefs);
                    firstAndMiddleStr = firstStr + attributeStr.toString() +
                        "> \n" + middleStr;
                }
            }
        }

        str = firstAndMiddleStr + lastStr;

        /*
        if (tag.equalsIgnoreCase("author")) {
            if(SCIA.debug_on) System.err.println("***** author: lastStr = " + lastStr);
        }
        */
        return str;
    }

    public String getChildrenView(
        String documentFile, 
        String currentVar,
        String headSpace, 
         // the current match if matched, or the closest ancestor match
        Matching currentMatch, StringBuffer parentAttributeStr,
        StringBuffer queryHead, StringBuffer udfDefs) {

        String middleStr = headSpace;
        boolean isLastChildMatched;

        if (SCIA.debug_on) {
            System.err.println("handling " + getTag() + "'s children");
        }

        if ((children != null) && !children.isEmpty()) {
            headSpace = headSpace + "  ";

            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (SCIA.debug_on) {
                    System.err.println("current child is " +
                        child.data.getTag());
                }

                isLastChildMatched = true;

                if (i != (children.size() - 1)) {
                    for (int j = i + 1; j < children.size(); j++) {
                        TNode nextChild = (TNode) children.get(j);

                        if (SCIA.debug_on) {
                            System.err.println("next child is " +
                                nextChild.data.getTag());
                        }

                        if ((nextChild.fromComb != null) ||
                                (nextChild.independentUnionMatches != null) ||
                                (nextChild.dependentUnionMatches != null)) {
                            isLastChildMatched = false;
                        }
                    }
                }

                middleStr = middleStr +
                    child.generateView(documentFile, currentVar, headSpace,
                        currentMatch, isLastChildMatched, parentAttributeStr,
                        queryHead, udfDefs);
                if (SCIA.debug_on) {
                    System.err.println("Til " + child.getTag() + " view = " + 
                                       middleStr);
                }
            }
        }

        return middleStr;
    }

    boolean waitForUserConfirmation() throws InterruptedException {
        /* We're going to show the modal dialog on the event dispatching
         * thread with SwingUtilities.invokeLater(), so we create a
         * Thread class that shows the dialog and stores the user's
         * response.
         */
        class DoShowDialog implements Runnable {
            boolean proceedConfirmed;

            public void run() {
                Object[] options = { "Continue", "Cancel" };
                int n = JOptionPane.showOptionDialog(null,
                        "Example2: Continue?", "Example2",
                        JOptionPane.YES_NO_OPTION,
                        JOptionPane.QUESTION_MESSAGE, null, options, "Continue");
                proceedConfirmed = (n == JOptionPane.YES_OPTION);
            }
        }

        DoShowDialog doShowDialog = new DoShowDialog();

        /* The invokeAndWait utility can throw an InterruptedException,
         * which we don't bother with, and an InvocationException.
         * The latter occurs if our Thread throws - which would indicate
         * a bug in DoShowDialog.  The invokeAndWait() call will not return
         * until the user has dismissed the modal confirm dialog.
         */
        try {
            SwingUtilities.invokeAndWait(doShowDialog);
        } catch (java.lang.reflect.InvocationTargetException e) {
            e.printStackTrace();
        }

        return doShowDialog.proceedConfirmed;
    }

    public boolean hasDecendentMatched() {
        boolean res = false;

        if ((children != null) && !children.isEmpty()) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if ((child.fromComb != null) ||
                        (child.independentUnionMatches != null) ||
                        (child.dependentUnionMatches != null)) {
                    res = true;

                    return res;
                } else {
                    res = child.hasDecendentMatched();
                }
            }
        }

        return res;
    }

    /* determine whether this node has decendent nodes matched to decendent of
       the input TNode ancestorMatchNode
    */
    public boolean hasDecendentMatchedFollow(TNode ancestorMatchNode) {
        boolean res = false;

        if ((children != null) && !children.isEmpty()) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if ((child.fromComb != null) && (child.fromComb.tnode != null)) {
                    if (child.fromComb.tnode.isDecendentOf(ancestorMatchNode)) {
                        return true;
                    }
                } else if (child.independentUnionMatches != null) {
                    for (int j = 0; j < child.independentUnionMatches.size();
                            j++) {
                        Matching match = (Matching) child.independentUnionMatches.get(j);

                        if ((match != null) && (match.tnode != null)) {
                            if (match.tnode.isDecendentOf(ancestorMatchNode)) {
                                return true;
                            }
                        }
                    }
                } else if (child.dependentUnionMatches != null) {
                    for (int j = 0; j < child.dependentUnionMatches.size();
                            j++) {
                        Matching match = (Matching) child.dependentUnionMatches.get(j);

                        if ((match != null) && (match.tnode != null)) {
                            if (match.tnode.isDecendentOf(ancestorMatchNode)) {
                                return true;
                            }
                        }
                    }
                } else { // no match at all, then check child
                    res = child.hasDecendentMatchedFollow(ancestorMatchNode);
                }
            }
        }

        return res;
    }

    /* to avoid recursive reference, check whether there is a node with the
       same name or same type in the path to the root, if so, return that node as
       the reference node
    */
    public TNode getRefNode(String tag, String type) {
        TNode ref = null;
        TNode fromTag = getRefNode(tag);

        if (fromTag != null) {
            return fromTag;
        } else if ((type != null) && (parent != null)) {
            if (parent.getDataType() != null) {
                if (parent.getDataType().equals(type)) {
                    return parent;
                } else {
                    ref = parent.getRefNode(tag, type);
                }
            } else {
                ref = parent.getRefNode(tag, type);
            }
        }

        return ref;
    }

    public TNode getRefNode(String tag) {
        TNode ref = null;

        if (parent != null) {
            if (parent.getTag().equals(tag)) {
                return parent;
            } else {
                ref = parent.getRefNode(tag);
            }
        }

        return ref;
    }

    /* expand the reference one level and add the subtree from the refered
       TNode to this TNode as a subtree of this TNode
    */
    public void addRefSubTree() {
        if (reference != null) {
            TNode refNodeCopy = reference.getCopyOfSubTree();

            if (refNodeCopy != null) {
                // attach the reference subtree to this node
                if ((refNodeCopy.children != null) &&
                        (refNodeCopy.children.size() > 0)) {
                    children = refNodeCopy.children;

                    for (int i = 0; i < children.size(); i++) {
                        TNode child = (TNode) children.get(i);
                        child.parent = this;
                    }

                    if (data.nodeType.equals("leafElem")) {
                        data.nodeType = "parentElem";
                    }
                }
            }

            //update height, rowNum, tree occupancy
            setHeightForAll();
            tree.occupancy = tree.occupancy + getSubTreeSize();
            tree.setRowNum();
        }
    }

    public TNode getCopyOfRefSubTree() {
        TNode refNodeCopy = null;

        if (reference != null) {
            refNodeCopy = reference.getCopyOfSubTree();
        }

        return refNodeCopy;
    }

    public TNode getCopyOfSubTree() {
        TNode nodeCopy = getCopy();

        if (children != null) {
            nodeCopy.children = new Vector();

            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    TNode childCopy = child.getCopyOfSubTree();
                    childCopy.parent = nodeCopy;
                    nodeCopy.children.addElement(childCopy);
                }
            }
        }

        return nodeCopy;
    }

    public TNode getCopy() {
        SchemaNode sn = data.getCopy();
        TNode copy = new TNode(sn);
        copy.tree = tree;

        return copy;
    }

    public void setHeightForAll() {
        if (parent == null) {
            height = 0;
        } else if (!data.nodeType.equals("compositor")) {
            height = parent.height + 1;
        } else if (data.nodeType.equals("compositor")) {
            height = parent.height;
        }

        if (children != null) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);

                if (child != null) {
                    child.setHeightForAll();
                }
            }
        }
    }

    public void updateAncestorMatch() {
        //whether parent matchs to same subtree
        boolean consistentContext = false;

        // involved nodes in this match: target node + source nodes 
        int nodesInMatch = 2;

        // siblings + match siblings, >=2
        int sib = 2;

        if ((realParent != null) && (realParent.fromComb != null)) {
            if ((fromComb.tnode != null) &&
                    (realParent.fromComb.tnode != null) &&
                    realParent.fromComb.tnode.isAncestorOf(fromComb.tnode)) {
                consistentContext = true;
                nodesInMatch = 2;
                sib = realParent.children.size() +
                    realParent.fromComb.tnode.children.size();
            } else if (fromComb.matchingUnits != null) {
                nodesInMatch = fromComb.matchingUnits.size() + 1;

                Iterator it = fromComb.matchingUnits.iterator();

                while (!consistentContext && it.hasNext()) {
                    TNode matchNode = ((MatchingUnit) it.next()).sourceNode;

                    if ((matchNode != null) && (realParent.fromComb != null) &&
                            (realParent.fromComb.tnode != null) &&
                            realParent.fromComb.tnode.isAncestorOf(matchNode)) {
                        consistentContext = true;
                        sib = realParent.children.size() +
                            realParent.fromComb.tnode.children.size();
                    }
                }
            } // to do: consider parent may have global union matches
        }

        if (consistentContext) {
            /* propagate similarity to parent using formula:

            */
            double contribution = (double) nodesInMatch / (double) sib;
            double sim = (fromComb.sim * contribution * SCIA.STRUCT_WEIGHT_UPDATE_ANCESTOR) +
                (realParent.fromComb.sim * (1 -
                (contribution * SCIA.STRUCT_WEIGHT_UPDATE_ANCESTOR)));
            realParent.fromComb.sim = (fromComb.sim * contribution * SCIA.STRUCT_WEIGHT_UPDATE_ANCESTOR) +
                (realParent.fromComb.sim * (1 -
                (contribution * SCIA.STRUCT_WEIGHT_UPDATE_ANCESTOR)));

            //update upstairs recursively
            if (realParent.realParent != null) {
                realParent.updateAncestorMatch();
            }
        }
    }

    public void resolveMultiMatches(boolean interactive_on)
        throws Exception {
        if (interactive_on) {
            /* case 1: no match
               case 2: no match, has fixed value
               case 3: no match, but exist matches for its descendents
               case 4: single match: best
               case 5: single match: 2nd best
               case 6: global multiple matches: both
               case 7: global multiple matches: more
               case 8: local multiple matches: tell how to formulate them
            */

            //redraw mapping in the gui
            Thread.sleep(200);
            tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                tree.mainUI.treesSplitPane.getWidth(),
                tree.mainUI.treesSplitPane.getHeight());
            tree.mainUI.addMatchedLines();
            tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

            //highlight working target node and candidates in gui             
            Vector nodes = new Vector();
            nodes.add(this);
            nodes.add(fromComb.tnode);
            nodes.add(secFromComb.tnode);
            tree.mainUI.treesSplitPane.highlightNodes = nodes;
            tree.mainUI.treesSplitPane.drawHighlight(nodes,
                tree.mainUI.treesSplitPane.getGraphics());

            //user interaction
            String message = "For " + getPath() +
                ", the predicted best match is " + fromComb.tnode.getPath() +
                "  sim = " + format.format(fromComb.sim) +
                "  the second best match is " + secFromComb.tnode.getPath() +
                "  sim = " + format.format(secFromComb.sim) +
                " \nplease provide input and choose " + " one option:";

            Vector choices = new Vector();
            choices.add("No match for this path");
            choices.add("No match, I will input its " + "constant value");
            choices.add("No match for this path, " +
                "but there exist matches for its " + "decendents");
            choices.add("Take the best");
            choices.add("Take the best, I will add information");
            choices.add("Take the second best");
            choices.add("Union matches, " + "take both");
            choices.add("Union matches, take both and add more");
            choices.add("I will input the match manually");

            final String msg = message;
            final Vector c = choices;

            final Interactive2 inter2 = new Interactive2(msg, c);

            Thread.sleep(200);
            inter2.show();

            Thread.sleep(200);
            tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                tree.mainUI.treesSplitPane.getWidth(),
                tree.mainUI.treesSplitPane.getHeight());
            tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

            // wait for the option to close
            while (inter2.isVisible()) {
                Thread.sleep(100);
            }

            // unhighlight nodes
            int selectionRow = tree.mainUI.schemaJTree1.getMinSelectionRow();
            tree.mainUI.schemaJTree1.removeSelectionRow(selectionRow);
            selectionRow = tree.mainUI.schemaJTree2.getMinSelectionRow();
            tree.mainUI.schemaJTree2.removeSelectionRow(selectionRow);

            // set arrow to be un-selected
            tree.mainUI.arr.manualSelect("none");

            dataEntered = inter2.data;

            if (SCIA.debug_on) {
                System.err.println("option is " + dataEntered);
            }

            if (dataEntered.equals("1")) {
                fromComb = null;
                resetFromCombForAll();

                //redraw mapping in the gui
                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.addMatchedLines();
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());
            } else if (dataEntered.equals("2")) {
                // no matching path, but fixed value exists
                // reset the best matches, the fromComb field, of its subtree
                resetFromCombForAll();

                //redraw mapping in the gui
                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.addMatchedLines();
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                final String msg1 = "Please enter the value:";
                final Interactive1 inter3 = new Interactive1(msg1);

                Thread.sleep(200);
                inter3.show();

                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                // wait for the option to close
                while (inter3.isVisible()) {
                    Thread.sleep(100);
                }

                constantValue = inter3.data;
            } else if (dataEntered.equals("3")) {

            } else if (dataEntered.equals("5")) {

                // keep the best, set sim = 1 and userInputMatch = true
                fromComb.sim = 1.0;
                userInputMatch = true;

                //update ancetors' and subtree's match similarity values
                updateAncestorMatch();
                updateDecendentMatch();

                message = "Add information to this match";

                final Interactive4 inter = new Interactive4(message);

                Thread.sleep(200);
                inter.show();

                while (inter.isVisible()) {
                    Thread.sleep(100);
                }

                //redraw mapping in the gui
                //tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                //tree.mainUI.treesSplitPane.getWidth(), 
                //tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.schemaJTree1.setSelectionPath(null);
                tree.mainUI.schemaJTree2.setSelectionPath(null);
                tree.mainUI.treesSplitPane.highlightNodes = null;

                tree.mainUI.addMatchedLines();
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());
          
            } else if (dataEntered.equals("9")) {
                fromComb = null;

                //redraw mapping in the gui
                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.addMatchedLines();
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                tree.mainUI.schemaJTree1.setSelectionPath(null);
                tree.mainUI.schemaJTree2.setSelectionPath(null);
                tree.mainUI.treesSplitPane.highlightNodes = null;

                // input match manually
                message = "Please enter match(es) " + "for this path: " +
                    getPath();

                final Interactive4 inter = new Interactive4(message);

                Thread.sleep(200);
                inter.show();

                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                while (inter.isVisible()) {
                    Thread.sleep(100);
                }

                ;

                Thread.sleep(200);
                tree.mainUI.treesSplitPane.paintImmediately(0, 0,
                    tree.mainUI.treesSplitPane.getWidth(),
                    tree.mainUI.treesSplitPane.getHeight());
                tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.treesSplitPane.getGraphics());

                userInputMatch = true;
                updateAncestorMatch();

                if ((fromComb != null) && (fromComb.matchingUnits != null) &&
                        !fromComb.matchingUnits.isEmpty()) {
                    for (int i = 0; i < fromComb.matchingUnits.size(); i++) {
                        MatchingUnit unit = (MatchingUnit) fromComb.matchingUnits.get(i);

                        if (SCIA.debug_on) {
                            System.err.println(getPath() + " match unit" + i +
                                " = " + unit.sourceNode.getPath());
                        }
                    }
                }
            }
        }
    }

    /* if the best from context check is much less similar than w/o context
       fromComb that has very high sim value, it indicates condition may be
       needed, propose condition from 2 contexts connection point (foreign key),
       e.g., genomeref to genome@id, and ask users
    */
    public void resolveContextConflict() throws Exception {

       //check whether it is cross level grouping
        if (checkCrossLevelGrouping()) {
            return;
        }

        //highlight the possibly conditional match line and nodes
        Vector nodes = new Vector();
        nodes.add(this);
        nodes.add(fromComb.tnode);
        tree.mainUI.treesSplitPane.highlightNodes = nodes;
        tree.mainUI.treesSplitPane.drawHighlight(nodes,
            tree.mainUI.treesSplitPane.getGraphics());

        /* propose condition based on context match node's IDREF */

        // look for foreign key in parent match node's children
        TNode pmcRefNode = null; // parent match child reference node
        TNode mSibIDNode = null; // match node's sibling  which is an ID 
        Condition condition = null;

        if ((fromComb.tnode != null) && (fromComb.tnode.parent != null)) {
            TNode mp = fromComb.tnode.parent;

            for (int i = 0; i < mp.children.size(); i++) {
                TNode mSib = (TNode) mp.children.get(i);

                if (mSib.getTag().equalsIgnoreCase("ID")) {
                    mSibIDNode = mSib;
                }
            }

            //get the parent match child with most similar tag with match parent 
            if ((parent != null) && (parent.fromComb != null) &&
                    (parent.fromComb.tnode != null) &&
                    (parent.fromComb.tnode.children != null)) {
                double maxSim = 0;

                for (int i = 0; i < parent.fromComb.tnode.children.size();
                        i++) {
                    TNode n = (TNode) parent.fromComb.tnode.children.get(i);

                    if ((n != null) && (n.getDataType() != null) &&
                            n.getDataType().equalsIgnoreCase("IDREF")) {
                        double sim = LetterPairSimilarity.compareStrings
                            (n.getTag(), mp.getTag());

                        if (sim > maxSim) {
                            pmcRefNode = n;
                        }
                    }
                }
            }

            if ((pmcRefNode != null) && (mSibIDNode != null)) {
                condition = new Condition(mSibIDNode, "=", pmcRefNode);
                fromComb.condition = condition;
            }
        }

        String mes;

        if (condition != null) {
            mes = "Please check the predicted condition: " +
                condition.toString() + ". Please edit it if needed.";
        } else {
            mes = "Is any condition involved? If so " +
                " please enter the condition for the match, or edit it.";
        }

        String message = "The predicted best match is not " +
            " consistent with the context. " + mes;

        final Interactive4 inter = new Interactive4(message);

        Thread.sleep(200);
        inter.show();

        while (inter.isVisible()) {
            Thread.sleep(100);
        }
    }

    public void setRealParentChildrenForAll(TNode working) {
        TNode newWorking = new TNode();

        //skip compositor nodes
        if (data.nodeType.equals("compositor")) {
            newWorking = working;

            //add a non-compositor node into working node's realChildren
        } else if (!data.nodeType.equals("compositor")) {
            //set working node as this node's realParent
            realParent = working;

            if (working.realChildren == null) {
                working.realChildren = new Vector();
            }

            working.realChildren.add(this);

            //update working node
            newWorking = this;
        }

        // process children recursively
        if ((children != null) && !children.isEmpty()) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode) children.get(i);
                child.setRealParentChildrenForAll(newWorking);
            }
        }
    }

    /**
     * Return the first non-compositor anceptor TNode.
     */
    public TNode getRealParent() {
        if ((parent != null) && !parent.data.nodeType.equals("compositor")) {
            return parent;
        } else if (parent != null) {
            return parent.getRealParent();
        }

        return null;
    }

    /**
     * Return the subelement or attribute that serves as the identifier.
     */
    public TNode IDChild() {
        if (children != null && !children.isEmpty()) {
            for (int i = 0; i < children.size(); i++) {
                TNode child = (TNode)children.get(i);
                if (child.getTag().equalsIgnoreCase("id") ||
                    child.getNormalizedTag().equalsIgnoreCase("id")) {
                    return child;
                }
            }
        }
        return null;
    }


    public String getNormalizedTag() {
        return data.getNormalizedTag();
    }


    /**
     * Check cross level Grouping.
     */
    public boolean checkCrossLevelGrouping() throws Exception {
        boolean res = false;
        if (parent != null && parent.fromComb != null) {
            TNode parentMatchNode = parent.fromComb.tnode;
            if (!parent.fromComb.tnode.isAncestorOf(fromComb.tnode) &&
                fromComb.tnode.isAncestorOf(parent.fromComb.tnode)){

                //highlight the possibly conditional match line and nodes
                Vector nodes = new Vector();
                nodes.add(this);
                nodes.add(fromComb.tnode);
                tree.mainUI.treesSplitPane.highlightNodes = nodes;
                tree.mainUI.treesSplitPane.drawHighlight(nodes,
                    tree.mainUI.treesSplitPane.getGraphics());
                boolean byID = false; // grouping by id
                String message = "Context conflict exists! Are " + 
                    getTag() + "s to be regrouped by " + 
                    parentMatchNode.getTag();

                //check with user whether it is regrouping by the parent, 
                // or its id if exists.
                TNode idChild = parentMatchNode.IDChild();
                if (idChild != null) {
                    message = message + "'s " + idChild.getTag();
                    byID = true;
                } else {
                    message = message + 
                        ", by default all its attributes and subelements?";
                }

                final Interactive4 inter = new Interactive4(message);

                //Yes, No grouping, Edit the match manually
                Vector choices = new Vector();
                choices.add("Yes, grouping attributes are correct.");
                choices.add("No, I will edit the match.");
                choices.add("There is no grouping.");

                final String msg = message;
                final Vector c = choices;
                final Interactive2 inter2 = new Interactive2(msg, c);

                Thread.sleep(200);
                inter2.show();
                Thread.sleep(200);

                // wait for the option to close
                while (inter2.isVisible()) {
                    Thread.sleep(100);
                }

                dataEntered = inter2.data;

                if (SCIA.debug_on) {
                    System.err.println("option is " + dataEntered);
                }

                if (dataEntered.equals("1")) {
                    res = true;
                    // if it is regrouping by the parent
                    // keep the best, set sim = 1 and userInputMatch = true
                    //fromComb.sim = 1.0;
                    userInputMatch = true;
                    fromComb.crossLevelGrouping = true;
                    fromComb.groupAttrs = new Vector();
                    if (byID) {
                        fromComb.groupAttrs.add(idChild);
                    } else {
                        fromComb.groupAttrs.add(parentMatchNode);
                    }
                    // Determine use DISTINCT or not in view generation
                    parent.fromComb.isCrossLevelGroupAttr = true;
                } else if (dataEntered.equals("2")) {
                    res = true;
                    // wait for the user to enter grouping attributes

                    // keep the best, set sim = 1 and userInputMatch = true
                    fromComb.sim = 1.0;
                    userInputMatch = true;

                    //update ancetors' and subtree's match similarity values
                    updateAncestorMatch();
                    //updateDecendentMatch();
                    message = "Please edit this match";

                    final Interactive4 inter4 = new Interactive4(message);

                    Thread.sleep(200);
                    inter4.show();

                    while (inter4.isVisible()) {
                        Thread.sleep(100);
                    }
                    tree.mainUI.schemaJTree1.setSelectionPath(null);
                    tree.mainUI.schemaJTree2.setSelectionPath(null);
                    tree.mainUI.treesSplitPane.highlightNodes = null;
                    tree.mainUI.addMatchedLines();
                    tree.mainUI.treesSplitPane.drawMapping(tree.mainUI.
                        treesSplitPane.getGraphics());
                } else if (dataEntered.equals("3")) {
                    // keep the best, set sim = 1 and userInputMatch = true
                    fromComb.sim = 1.0;
                    userInputMatch = true;
                    // no grouping, just continue
                }
            }
        }
        return res;
    }

    public void updateDecendentMatch() {
    }
}
