/**
 * A collection of utility services (method calls) for the Semantic
 * Mediation System.
 * 
 *    '$RCSfile: SMSServices.java,v $'
 *
 *     '$Author: bowers $'
 *       '$Date: 2005/10/25 06:29:40 $'
 *   '$Revision: 1.6 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 * @ProposedRating red (sbowers@ucdavis.edu)
 * @AcceptedRating red (...)
 */

package org.kepler.sms;


import ptolemy.kernel.Entity;
import java.util.Vector;
import java.util.Iterator;
import ptolemy.actor.IOPort;


//////////////////////////////////////////////////////////////////////////
//// SMSServices
/**
 * This class provides a set of operations, or services for
 * interacting with the Semantic Mediation System. These services are
 * currently only partially defined.
 * 
 * @author Shawn Bowers
 * @version $Id: SMSServices.java,v 1.6 2005/10/25 06:29:40 bowers Exp $
 * @since Kepler alpha
 */


public class SMSServices { 
    
    public static int COMPATIBLE = 1;     
    public static int UNKOWN = 0;
    public static int INCOMPATIBLE = -1;


    /**
     * For use within an extended SCIA tool
     */
    // public String getSemanticCorrespondences(Actor source, Actor target)



    /**
     * @return The regular and virtual input ports of the given entity.
     */
    public static Vector getAllInputPorts(Entity entity) {
	Vector result = new Vector();
	for(Iterator iter = entity.portList().iterator(); iter.hasNext(); ) {
	    Object obj = iter.next();
	    if(obj instanceof IOPort) {
		IOPort p = (IOPort)obj;
		if(p.isInput())
		    result.add(p);
	    }
	}
	// get all the virtual ports
	Class filter = KeplerVirtualIOPort.class;
	for(Iterator iter = entity.attributeList(filter).iterator(); iter.hasNext(); ) {
	    KeplerVirtualIOPort p = (KeplerVirtualIOPort)iter.next();
	    if(p.isInput())
		result.add(p);
	}
	return result;
    }


    /**
     * @return The regular and virtual output ports of the given entity.
     */
    public static Vector getAllOutputPorts(Entity entity) {
	Vector result = new Vector();
	for(Iterator iter = entity.portList().iterator(); iter.hasNext(); ) {
	    Object obj = iter.next();
	    if(obj instanceof IOPort) {
		IOPort p = (IOPort)obj;
		if(p.isOutput())
		    result.add(p);
	    }
	}
	// get all the virtual ports
	Class filter = KeplerVirtualIOPort.class;
	for(Iterator iter = entity.attributeList(filter).iterator(); iter.hasNext(); ) {
	    KeplerVirtualIOPort p = (KeplerVirtualIOPort)iter.next();
	    if(p.isOutput())
		result.add(p);
	}
	return result;
    }



    /**
     * Compare the compatibility of the two sets of semantic types.
     * Both arguements represent conjoined sets of semantic types. The
     * method returns three possible values: compatible, unknown, or
     * incompatible.  If either type is empty, unknown is
     * returned. The types are considered compatible if they are
     * non-empty and if the conjunction of the first set implies the
     * conjunction of the second set. The types are considered
     * incompatible if they are not compatible and not unkown.
     * @param semSubtypes The semantic types that when conjoined form
     * a sub-class of the super type (semSupertypes).
     * @param semSupertypes The semantic types that when conjoined
     * form a super-class of the sub type (subSemTypes)
     * @return Answers {@link #COMPATIBLE} if the inputs are
     * compatible, {@link #UNKNOWN} if the inputs are unkown, and
     * {@link INCOMPATIBLE} if the types are incompatible. 
     *
     * FIXME: Need to somehow handle the case when the semtype is not
     * available locally: Can we assume they are known here? Do we
     * need to call the object manager? Do we throw an exception? For
     * now, we just ignore them!
     */
    public static int compare(Vector semSubtypes, Vector semSupertypes) {
	OntologyCatalog catalog = OntologyCatalog.instance();
	Vector subClasses = new Vector();
	Vector superClasses = new Vector();

	// first check if either is empty; and if so return unknown
	if(semSubtypes.size() == 0 || semSupertypes.size() == 0 )
	    return UNKOWN;

	// convert to ont classes; if we don't have knowledge of the
	// class, then don't add it ...
	for(Iterator iter = semSubtypes.iterator(); iter.hasNext();) {
	    NamedOntClass c = catalog.getNamedOntClass((SemanticType)iter.next());
	    if(c != null && !subClasses.contains(c)) 
		subClasses.add(c); // ignore unknown types
	}
	for(Iterator iter = semSupertypes.iterator(); iter.hasNext();) {
	    NamedOntClass c = catalog.getNamedOntClass((SemanticType)iter.next());
	    if(c != null && !superClasses.contains(c))
		superClasses.add(c); // ignore unkown types
	}

	// if the sem-subtypes contain a contradiction, then return
	// compatible (i.e., false implies anything)
	for(Iterator iter = subClasses.iterator(); iter.hasNext();) {
	    NamedOntClass cls = (NamedOntClass)iter.next();
	    for(Iterator iter2 = subClasses.iterator(); iter.hasNext();) {
		NamedOntClass tstCls = (NamedOntClass)iter.next();
		if(cls.isDisjointWith(tstCls))
		    return COMPATIBLE;
	    }
	}

	// if the sem-supertypes contain a contradiction, then return
	// incompatible, (i.e., we have true implies false)
	for(Iterator iter = superClasses.iterator(); iter.hasNext();) {
	    NamedOntClass cls = (NamedOntClass)iter.next();
	    for(Iterator iter2 = superClasses.iterator(); iter.hasNext();) {
		NamedOntClass tstCls = (NamedOntClass)iter.next();
		if(cls.isDisjointWith(tstCls))
		    return INCOMPATIBLE;
	    }	    
	}
	
	// check that every supertype has a corresponding subtype
	for(Iterator iter = superClasses.iterator(); iter.hasNext(); ) {
	    NamedOntClass superClass = (NamedOntClass)iter.next();
	    boolean found = false;
	    for(Iterator iter2 = subClasses.iterator(); iter2.hasNext(); ) {
		NamedOntClass subClass = (NamedOntClass)iter2.next();
		if(superClass.isEquivalent(subClass) || superClass.isSubClass(subClass))
		    found = true;
	    }
	    if(!found)
		return INCOMPATIBLE;
	}
	
	return COMPATIBLE;
    }


    /**
     *
     */
    public static boolean compatible(Vector semSubtypes, Vector semSupertypes) {
	return compare(semSubtypes, semSupertypes) == COMPATIBLE;
    }


    /**
     *
     */
    public static Vector getActorSemanticTypes(Entity entity) {
	Vector result = new Vector();
	for(Iterator iter = entity.attributeList(SemanticType.class).iterator(); iter.hasNext();) 
	    result.add(iter.next());
	return result;
    }


    /**
     * FIXME: Only returns the port types, not virtual ports ...
     */
    public static Vector getPortSemanticTypes(IOPort port) {
	Vector result = new Vector();
	for(Iterator sts = port.attributeList(SemanticType.class).iterator(); sts.hasNext();) {
	    SemanticType st = (SemanticType)sts.next();
	    result.add(st);
	}
	return result;
    }

    /**
     * FIXME: Only returns the ports, not virtual ports ...
     */
    public static Vector getInputPortSemanticTypes(IOPort port) {
	Vector result = new Vector();
	if(port.isInput()) {
	    for(Iterator sts = port.attributeList(SemanticType.class).iterator(); sts.hasNext();) {
		SemanticType st = (SemanticType)sts.next();
		result.add(st);
	    }
	}
	return result;
    }


    /**
     * FIXME: Only returns the ports, not virtual ports ...
     */
    public static Vector getOutputPortSemanticTypes(IOPort port) {
	Vector result = new Vector();
	if(port.isOutput()) {
	    for(Iterator sts = port.attributeList(SemanticType.class).iterator(); sts.hasNext();) {
		SemanticType st = (SemanticType)sts.next();
		result.add(st);
	    }
	}
	return result;
    }



    /**
     * 
     */
    public static String exportSemanticAnnotation(Entity entity) {
	//  We assume that annotation rules always have the same
	//  parents.  For example, given a port p1 :: {a = int, b =
	//  int} we have the following:
	//
	//    Annotating p1 with #m gives
	//       val: $1 p1 => inst: $1 #m
	//
	//    Annotating p1/a with #b gives
	//       val: $1 p1, val: $2 $1/a => inst: $2 #b
	//
	//    Annotation p1 linked to p1/2 via #p gives
	//       val: $1 p1, val: $2 $1/a => prop: $1 #p $2
	//
	//    Annotating generalization g1(p1/a, p1/b) with #d gives
	//       val: $1 p1, val: $2 $1/a, val: $3 $1/b => 
	//         inst: g1($2, $3) #d
	//
	//    where a and b are assumed to be values within the same
	//    port value
	//
	// Thus, we can't do "cross-product-style" annotation, e.g.,
	// given p1 with structural type {{a=int, b=int}} (a list of
	// records), we can't say things like:
	//
	//    val: $1 p1, val: $2 $1/elem, val: $3 $1/elem, 
	//    val: $4 $2/a, val: $5 $3/b => 
	//      inst: g1($4, $5) #d
	//
	// In particular, we would have instead:
	//
	//    val: $1 p1, val: $2 $1/elem, val: $3 $2/a, 
	//    val: $4 $2/b => 
	//      inst: g1($3, $4) #d
	//
	// That is, we assume that the components always have 
	// the same parent
	//
	return null;
    }
    

    /**
     *
     */
    public static void importSemanticAnnotation(String annotation, Entity entity) {
    }

    // searching ... 



} // SMSServices