/**
 *    '$RCSfile: ActorSemanticTypeEditorPane.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2005/11/01 20:39:13 $'
 *   '$Revision: 1.3 $'
 *
 *  For Details: http://kepler-project.org
 *
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.sms.gui;

import java.awt.Dimension;
import java.awt.Frame;
import java.util.Iterator;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SpringLayout;

import org.kepler.moml.NamedObjId;

import ptolemy.kernel.util.NamedObj;


/**
 * SemanticTypeEditor is a dialog for adding and removing semantic types to 
 * particular workflow components.
 * 
 * @author Shawn Bowers
 */
public class ActorSemanticTypeEditorPane extends JPanel
{

    /**
     * Constructor.
     */
    public ActorSemanticTypeEditorPane(Frame owner, NamedObj namedObj) {
 	super();
 	_owner = owner;
	_namedObj = namedObj;

	_table = new SemanticTypeTable(owner);
	_table.addAnnotationObject(namedObj);
	_table.setAnnotationObjectVisible(namedObj);

 	setLayout(new BoxLayout(this, BoxLayout.PAGE_AXIS));
 	add(_createHeader());
 	add(Box.createRigidArea(new Dimension(5,0)));	
 	add(_table);
 	setSize(450,285);
    }

 
    /**
     * Performs the commit button operation.
     */
    public void doCommit() {
	_table.commitAnnotationObjects();
    }
    

    /**
     * Performs the cancel button operation.
     */
    public void doClose() {
	_table.dispose();
    }


    /**
     * @return True if any semantic type annotations have been modified.
     */
    public boolean hasModifiedSemTypes() {
	return _table.hasModifiedAnnotationObjects();
    }
    

    /**
     * @return The first error if semantic type annotations are not
     * well formed, and null otherwise.
     */
    public String wellFormedSemTypes() {
	return _table.wellFormedSemTypes();
    }

    
    /** 
     * @return True if there are any semantic types that are not in
     * the catalog.
     */
    public boolean hasUnknownSemTypes() {
	return _table.hasUnknownSemTypes();
    }


    /**
     * Initializes the header
     */
    private JPanel _createHeader() {
 	JPanel headerPane = new JPanel();
 	SpringLayout layout = new SpringLayout();
 	headerPane.setLayout(layout);
 	// create labels
 	JLabel nameLabel = new JLabel("Object Name:");
 	JLabel nameField = new JLabel(" '" + _namedObj.getName() + "'");
 	JLabel typeLabel = new JLabel("Object Type:");
 	JLabel typeField = new JLabel(" " + _namedObj.getClass().getName());
 	JLabel idLabel = new JLabel("Object ID:");
 	JLabel idField = new JLabel(" " + _getNamedObjId());
 	// add components
 	headerPane.add(nameLabel);
 	headerPane.add(nameField);
 	headerPane.add(typeLabel);
 	headerPane.add(typeField);
 	headerPane.add(idLabel);
 	headerPane.add(idField);
 	// first row
 	layout.putConstraint(SpringLayout.WEST, nameLabel, 10, SpringLayout.WEST, headerPane);
 	layout.putConstraint(SpringLayout.SOUTH, nameLabel, 0, SpringLayout.SOUTH, nameField);
 	layout.putConstraint(SpringLayout.WEST, nameField, 10, SpringLayout.EAST, nameLabel);
 	layout.putConstraint(SpringLayout.NORTH, nameField, 10, SpringLayout.NORTH, headerPane);
 	// second row
 	layout.putConstraint(SpringLayout.WEST, typeLabel, 0, SpringLayout.WEST, nameLabel);
 	layout.putConstraint(SpringLayout.SOUTH, typeLabel, 0, SpringLayout.SOUTH, typeField);
 	layout.putConstraint(SpringLayout.WEST, typeField, 0, SpringLayout.WEST, nameField);
 	layout.putConstraint(SpringLayout.NORTH, typeField, 5, SpringLayout.SOUTH, nameField);
 	// third row
 	layout.putConstraint(SpringLayout.WEST, idLabel, 0, SpringLayout.WEST, nameLabel);
 	layout.putConstraint(SpringLayout.SOUTH, idLabel, 0, SpringLayout.SOUTH, idField);
 	layout.putConstraint(SpringLayout.WEST, idField, 0, SpringLayout.WEST, nameField);
 	layout.putConstraint(SpringLayout.NORTH, idField, 5, SpringLayout.SOUTH, typeField);
 	// 
 	//headerPane.setMaximumSize(new Dimension(Short.MAX_VALUE, 50));
 	JPanel pane = new JPanel();
 	pane.setLayout(new BoxLayout(pane, BoxLayout.LINE_AXIS));
 	pane.add(headerPane);
 	pane.setMaximumSize(new Dimension(Short.MAX_VALUE, 70));
 	pane.setMinimumSize(new Dimension(Short.MAX_VALUE, 70));
 	pane.setPreferredSize(new Dimension(Short.MAX_VALUE, 70));
 	return pane;
     }


    /**
     * @return The object id for the actor.
     */
    private String _getNamedObjId() {
	 if(_namedObj == null)
 	    return "<none>";
 	for(Iterator iter = _namedObj.attributeList().iterator(); iter.hasNext();) {
 	    Object att = iter.next();
 	    if(att instanceof NamedObjId) {
 		NamedObjId id = (NamedObjId)att;
 		return id.getExpression();
 	    }
 	}		
 	return "<none>";
     }


    /* Private members */

    private Frame _owner;
    private NamedObj _namedObj;
    private SemanticTypeTable _table;


} // ActorSemanticTypeEditorPane
