/* Abstract class for managing collections.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 package org.nddp;

import java.util.List;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.nddp.exceptions.CollectionException;
import org.nddp.tokens.DataToken;
import org.nddp.tokens.DomainObjectToken;
import org.nddp.tokens.ExceptionToken;
import org.nddp.tokens.MetadataToken;
import org.nddp.tokens.VariableToken;

import ptolemy.data.ObjectToken;
import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.kernel.util.IllegalActionException;

public abstract class CollectionManager {

	CollectionManager(Collection collection, CollectionManager 
        parentCollectionManager, CollectionPort outputPort) {
		
	    // check preconditions
		if ( collection == null ) {
		    throw new NullPointerException("Null collection");
		}
		
		// store reference to the collection to be managed
		_collection = collection;

		// store reference to the new parent CollectionManager
		_parentCollectionManager = parentCollectionManager;
		
		// add this CollectionManager to its parent's list of child 
		// CollectionManagers
		if  ( _parentCollectionManager != null ) {
	 	  	_parentCollectionManager._addChild(this);
	 	}

		_outputPort = outputPort;
	}

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
	
	public final void addDataToken(Token token) {
	    
	    assert ! (token instanceof ObjectToken) : "Instances of ObjectToken are" +
	    		"not allowed.";
	    
    	    // check preconditions
		if (_collectionClosed) {
		    throw new IllegalStateException("Collection is closed");
		}

		_newDataTokenList.add(new DataToken(_collection, token));
	}

	public final void addDomainObject(DomainObject object) {
	    
    	    // check preconditions
		if (_collectionClosed) {
		    throw new IllegalStateException("Collection is closed");
		}
		
        // lock the domain object so that it is immutable
        object.setWriteLock();
        
        // add the locked domain object to the list of new data tokens
        _newDataTokenList.add(new DomainObjectToken(_collection, object));
	}

	public void addException(Coactor source, CollectionException 
        exception) {

		// check preconditions
		if (_collectionClosed) {
		    throw new IllegalStateException("Collection is closed");
		}
		
		// create a new metadata token and store in the new metadata token list
		_newExceptionList.add(
		        new ExceptionToken(_collection, source, exception));
		
		_registerException();
	}
	
	public final void addMetadata(String key, Token value) {

		// check preconditions
		if (_collectionClosed) {
		    throw new IllegalStateException("Collection is closed");
		}
		
		// add it to the metadata table
		_metadataTable.put(key, value);
		
		// create a new metadata token and store in the new metadata 
		// token list
		_newMetadataTokenMap.put(key, 
		        new MetadataToken(_collection, key, value));
	}
	
	public final void addVariable(String key, Token value) {
		
	    // check preconditions
		if (_collectionClosed) {
		    throw new IllegalStateException("Collection is closed");
		}
		
		// add it to the metadata table
		_metadataTable.put(key, value);

		// create a new metadata token and store in the new metadata token list
		_newMetadataTokenMap.put(key, 
		        new VariableToken(_collection, key, value));
	}
	
	final void _clearNewMetadataList() {
		_newMetadataTokenMap.clear();
	}
		
	public abstract void closeCollection() throws IllegalActionException;
	
	public final Collection collection() {
		return _collection;
	}
	
	public boolean containsException() {
	    return _containsException;
	}
	
	public void discard(
	        boolean dummy) {
    };
	
	public boolean discarded() {
	    return false;
	}
	
	public String expandMetadataSymbols(String string) {
	    
	    	Matcher matcher = METADATA_SYMBOL_PATTERN.matcher(string);
		StringBuffer stringBuffer = new StringBuffer();
		
	    	while (matcher.find()) {
		    String metadataName = matcher.group(1);
		    Token metadataValue = metadataValue(metadataName);
		    
		    if (metadataValue instanceof StringToken) {
		        
		    		// eliminates quotes around value of string
		        matcher.appendReplacement(stringBuffer,
		                ((StringToken)metadataValue).stringValue());
		        
		    } else {
			    matcher.appendReplacement(stringBuffer, 
			            metadataValue.toString());
		    }
		 }
	    	
		matcher.appendTail(stringBuffer);

	    return stringBuffer.toString();
	}
	
	public final Object getAttribute(String key) {
	    
		return _attributeTable.get(key);
	}

	public void ignore(boolean dummy) {
    };	
	
	public  boolean ignored() {
	    return false;
    }
	
	public CollectionPort outputPort() {
	    return _outputPort;
	}
	
	public final Token metadataValue(String key) {
	    
	    	if (_metadataTable.containsKey(key)) {    	    

	        	return (Token)_metadataTable.get(key);
	        	
	    	} else if (_parentCollectionManager != null) {
	    	    
			return _parentCollectionManager.metadataValue(key);
			
	    	} else {
	    	    
	    	    	return MetadataToken.NO_MATCHING_METADATA;
	    	}
	}

	public final CollectionManager parentCollectionManager() {
	    return _parentCollectionManager;
	}
	
	public final void setAttribute(String key, Object value) {
	    
		_attributeTable.put(key, value);
	}

	public final void setOutputPort(CollectionPort outputPort) {
	    _outputPort = outputPort;
	}

	///////////////////////////////////////////////////////////////////
	////                 package-protected methods                 ////
	
	final void _registerException() {
	    
	    if (!_containsException) {
	    
	        _containsException = true;
	        
	        if (_parentCollectionManager != null) {
	            _parentCollectionManager._registerException();
	        }
	    }
	}

	final void _sendChildCollections() throws IllegalActionException {
		
		for (Iterator i = _childCollectionManagers.iterator(); i.hasNext(); ) {
		    
		    CollectionManager childCollectionManager = 
		        (CollectionManager)i.next();
		    
		    if (! childCollectionManager.discarded()) {
		        childCollectionManager._sendCollection();
		    }
		}
	}
	
	final void _sendClosingDelimiterToken() throws IllegalActionException {
	    
		_sendToken(_collection.closingDelimiter());						
		_collectionClosed = true;
	}

	abstract void _sendCollection() throws IllegalActionException;
	
	final void _sendNewDataTokens() throws IllegalActionException {
		
		for (Iterator i = _newDataTokenList.iterator(); i.hasNext(); ) {
			_sendToken((Token)(i.next()));
		}
	}
	
	final void _sendNewExceptions() throws IllegalActionException {
		    
	    for (Iterator i = _newExceptionList.iterator(); i.hasNext(); ) {
			_sendToken((Token)(i.next()));		        
	    }
	}
	
	final boolean _sendNewMetadataTokens() throws IllegalActionException {
	
		if ( _newMetadataTokenMap.isEmpty() ) {
			return false;
		} else {
		    	for (Iterator i = _newMetadataTokenMap.keySet().iterator(); 
		    			i.hasNext(); ) {
		    	    
				String key = (String)i.next();
				_sendToken((Token)(_newMetadataTokenMap.get(key)));
		    	}
		return true;
		}
	}
	
	final void _sendOpeningDelimiterToken() throws IllegalActionException {
	    
		_sendToken(_collection.openingDelimiter());		
	}

	final void _sendToken(Token token) throws IllegalActionException {
	    
	    if ( _outputPort != null ) {
			_outputPort.send(token);
	    }
	}

	///////////////////////////////////////////////////////////////////
	////                 package-protected variables               ////
	
	boolean _collectionClosed = false;

	final List _inputExceptionList = new LinkedList();
	
	final HashMap _metadataTable = new HashMap();
	
	final Map _newMetadataTokenMap = new HashMap();

	///////////////////////////////////////////////////////////////////
	////                         private methods                   ////

	private final void _addChild(CollectionManager manager) {
		
		// add the passed CollectionManager to this CollectionManager's
	    // list of child CollectionManagers
		_childCollectionManagers.add(manager);
	}

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
	
	private final Map _attributeTable = new HashMap();

	private final List _childCollectionManagers = new LinkedList();

	private final Collection _collection;

	private boolean _containsException = false;

	private final List _newDataTokenList = new LinkedList();

	protected List _newExceptionList = new LinkedList();

	private CollectionPort _outputPort;

	private final CollectionManager _parentCollectionManager;	

   	private static final Pattern METADATA_SYMBOL_PATTERN = 
   	    Pattern.compile("M\\{(.*?)\\}");
}