/* TODO One line description of class.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp;

import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


import ptolemy.actor.IOPort;
import ptolemy.actor.NoRoomException;
import ptolemy.data.ArrayToken;
import ptolemy.data.Token;
import ptolemy.kernel.util.IllegalActionException;

////////////////////////////////////////////////////////////////////////// 
//// InputPortBuffer 
 
/** 
  @author Timothy M. McPhillips
*/ 

public class InputPortBuffer {
    
    public InputPortBuffer(IOPort port) throws IllegalActionException {

        _port = port;
        
        // makd sure port name indicates it is bound to collection contents
        Matcher matcher = PORT_NAME_PATTERN.matcher(port.getName());
        if (! matcher.find()) {
            throw new IllegalActionException(
                "Port name does not match expected pattern.");
        }
            
        // extract the name of the data type bound to port
        String typeName = matcher.group(1);

        // make sure type name matches a registered data type
        if (! DataTypes.isDefined(typeName)) {
            throw new IllegalActionException(
                "No collection or data type " + typeName + ".");
        }
        
        // look up the data type
        _dataType = DataTypes.valueOfString(typeName);

        String group2 = matcher.group(2);
        if (group2 == null) {
            _minimum = 1;
            _maximum = 1;
            _bundlesTokens = false;
            _singleton = true;
        } else if (group2.equals("?")) {
            _minimum = 0;
            _maximum = 1;
            _bundlesTokens = false;
        } else if (group2.equals("+")) {
            _minimum = 1;
            _maximum = 0; 
            _bundlesTokens = true;
        } else if (group2.equals("*")) {
            _minimum = 0;
            _maximum = 0;
            _bundlesTokens = true;
        }

        String group4 = matcher.group(4);
        if (group4 != null) {
            int tokenCount = Integer.parseInt(group4);
            _minimum = tokenCount;
            _maximum = tokenCount;
            _bundlesTokens = true;
        }
        
        // determine if matching data tokens should be discarded
        String options = matcher.group(6);
        if (options != null) {
            
            _discardsInputTokens = (options.indexOf("-") != -1);
            
            if (options.indexOf("array") != -1) {
                _bundlesTokens = true;
            }

            if (options.indexOf("singleton") != -1) {
                _bundlesTokens = false;
            }

            // look for metadata key
            Matcher keyMatcher = METADATA_KEY_PATTERN.matcher(options);
            if (keyMatcher.find()) {
                _metadataKey = keyMatcher.group(1);
            }
        }
    }
    
    public Class dataType() {
        return _dataType;
    }
    
    public boolean put(Token token) {
        
        // TODO need to avoid storing tokens inside internally disconnected ports
        _tokenList.add(token);
        return _discardsInputTokens;
    }
    
    public void clear() {

        _tokenList.clear();
    }
    
    public boolean isReady() {
        
        int size = _tokenList.size();
        return (size >= _minimum && (_maximum == 0 || size <= _maximum));
    }
    
    public boolean bundlesTokens() {
        return _bundlesTokens;
    }
    
    public boolean singleton() {
        return _singleton;
    }
    
    public String metadataKey() {
        return _metadataKey;
    }
    
    public void flush() throws NoRoomException, IllegalActionException {
        
        Token tokenToSend = null;
        
        int tokenCount = _tokenList.size();
        
        if (tokenCount > 0) {

            if (_bundlesTokens) {
                
                Token[] tokenArray = new Token[tokenCount];
                
                for(int i = 0; i < tokenCount; i++) {
                    tokenArray[i] = (Token) _tokenList.get(i);
                }
                
                tokenToSend = new ArrayToken(tokenArray);
                
            } else {
                
                tokenToSend = (Token) _tokenList.get(0);
            }
        }
        
        if (tokenToSend != null) {
            _port.sendInside(0, tokenToSend);
        }
        
        _tokenList.clear();
    }
    
    private IOPort _port;
    private Class _dataType;
    private int _minimum;
    private int _maximum;
    private boolean _discardsInputTokens;
    private boolean _bundlesTokens = false;
    private boolean _singleton = false;
    private List _tokenList = new LinkedList();
    private String _metadataKey = null;

    private static final Pattern PORT_NAME_PATTERN = 
        Pattern.compile("(\\p{Alnum}+)([\\?\\+\\*])?(\\{(\\p{Digit}+)\\})?\\s*(\\[([^\\]]+)\\])?"); 

    private static final Pattern METADATA_KEY_PATTERN = 
        Pattern.compile("key\\s*=\\s*(\\p{Alnum}+)"); 
}