/* TODO One line description of class.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.actors.phylip;

import java.io.IOException;

import org.nddp.actors.EnvironmentActor;
import org.nddp.exceptions.ExternalApplicationException;
import org.nddp.util.Parameters;
import org.nddp.util.ProcessEnvironment;
import org.nddp.util.ProcessRunnerConcrete;

import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.actor type="phylogenetics"
 *
 *  @author Timothy M. McPhillips
 */

public class RunConsense extends EnvironmentActor {

    public RunConsense(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException  {
        
        super(container, name);

        consensusThreshold = 
            Parameters.doubleParameter(this, "consensusThreshold", 1.0);
        
        extendedConsensus = 
            Parameters.booleanParameter(this, "extendedConsensus", false);
    
        rootedTrees = 
            Parameters.booleanParameter(this, "rootedTrees", false);
        }

    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////
    public Parameter consensusThreshold;
    public Parameter extendedConsensus;
    public Parameter rootedTrees;
    
    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////
    public final void attributeChanged(Attribute attribute) 
        throws IllegalActionException {
            
        if (attribute == consensusThreshold) {
            _consensusThreshold = Parameters.doubleValue(consensusThreshold);
            if (_consensusThreshold < 0.5 || _consensusThreshold > 1.0) {
                throw new IllegalActionException(
                        "consensusThreshold must be between 0.5 and 1.0");
            }
    
        } else if (attribute == extendedConsensus ) {
                _extendedConsensus = Parameters.booleanValue(extendedConsensus);

        } else if (attribute == rootedTrees ) {
                _rootedTrees = Parameters.booleanValue(rootedTrees);
        
        } else {
            super.attributeChanged(attribute);
        }
    }    

    public void fire() throws IllegalActionException {
 
        // call superclass method
        super.fire();
        
        // get a reference to the process environment
        ProcessEnvironment processEnvironment = receiveEnvironment();

        // run consense
        try {
            
            // create a new process runner and start consense
            ProcessRunnerConcrete runner = new ProcessRunnerConcrete(
                processEnvironment, System.getProperty("PHYLIP_EXE_DIR") + "/consense");
        
            // write to the standard input of consense
            _writeStandardInput(runner);
            
            // wait for consense to complete
            runner.waitForProgramCompletion();

        } catch (ExternalApplicationException e) {
            e.printStackTrace();
        }

        // forward the process environment
        sendEnvironment();
    }
 
    private void _writeStandardInput(ProcessRunnerConcrete runner) 
        throws ExternalApplicationException {
    
        try {
            boolean strictConsensus;
            
            if (! _extendedConsensus && 
                    Math.abs(_consensusThreshold - 1.0) < 0.01 ) {
                
                strictConsensus = true;
                
            } else {
                
                strictConsensus = false;
            }
                    
            if (! _extendedConsensus) {
                
                if (strictConsensus) {
                        runner.writeToProcess("C");
                } else {
                        runner.writeToProcess("C");
                        runner.writeToProcess("C");
                        runner.writeToProcess("C");
                }
            }
                
            if (_rootedTrees) {
                runner.writeToProcess("R");
            }
            
            // Accept configuration and begin execution
            runner.writeToProcess("Y");
        
            if (! _extendedConsensus && ! strictConsensus) {
                runner.writeToProcess(_consensusThreshold + "");
            }
        
                
        } catch (IOException e) {
            throw new ExternalApplicationException(
                "Error writing to CONSENSE process");
            }           
    }    
    
    ///////////////////////////////////////////////////////////////////
    ////                    private variables                      ////
    
    private double _consensusThreshold;
    private boolean _extendedConsensus;
    private boolean _rootedTrees;
 
    private static final long serialVersionUID = 1L;
}
