/* Actor that parses Characters block of a Nexus-formatted string.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.actors.phylogeny;

import java.util.Iterator;

import org.nddp.exceptions.IndexOutOfBoundsException;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.TaxonomicCharacter;
import org.nddp.util.Port;

import ptolemy.actor.TypedIOPort;
import ptolemy.data.ObjectToken;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.actor type="phylogenetics"
 *
 *  @author Timothy M. McPhillips
 */

public class ParseNexusCharactersBlock extends NexusParsingActor {

    public ParseNexusCharactersBlock(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException  {
        
        super(container, name);

        // create an output port for sending the character matrix
        characterMatrixPort = new TypedIOPort(this, "characterMatrix", false, true);
        characterMatrixPort.setTypeEquals(BaseType.OBJECT);
        
        // precompile the pattern for finding the named Nexus block
        setBlockName("CHARACTERS");
        }

    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////

    public TypedIOPort characterMatrixPort;
    
    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////
        
    public void fire() throws IllegalActionException {
        
        super.fire();
        
        if (! actorEnabled()) {
            return;
        }
        
        // read the nexus file content string from the input port       
        String nexusFileString = Port.getString(nexusFilePort, 0, true);

        // extract the characters block
        String charactersBlockString = extractNexusBlock(nexusFileString);
        
        // parse the characters block if it was found
        if (charactersBlockString != null) {
            
            // parse the dimensions command
            NexusCommand dimensionsCommand = 
                NexusCommand.extractNexusCommand("DIMENSIONS", charactersBlockString);
            
            // parse the format command
            NexusCommand formatCommand = 
                NexusCommand.extractNexusCommand("FORMAT", charactersBlockString);
            
            // parse the matrix command
            NexusCommand matrixCommand = 
                NexusCommand.extractNexusCommand("MATRIX", charactersBlockString);
            
            // parse the charstatelables command
            NexusCommand characterDescriptions = 
                NexusCommand.extractNexusCommand("CHARSTATELABELS", charactersBlockString);
            
            // get the height of the character matrix
            int taxonCount = matrixCommand.argumentCount()/2;
            
            // get the width of the character matrix
            int characterCount = 
                Integer.parseInt(dimensionsCommand.getProperty("NCHAR"));
            
            // create a character matrix object of the right size
            CharacterMatrix matrix = 
                new CharacterMatrix(taxonCount, characterCount);

            // set the type of the matrix to morphological
            matrix.setDataType(TaxonomicCharacter.MORPHOLOGICAL_DATA);
            
            // set missing symbol for characters
            matrix.setMissingSymbol(
                formatCommand.getProperty("MISSING").charAt(0));
            
            // set gap symbol for characters
            matrix.setGapSymbol(
                formatCommand.getProperty("GAP").charAt(0));
            
            try {
            
                int taxonIndex = 0;
                for (Iterator i = matrixCommand.argumentIterator(); 
                            i.hasNext(); i.next(), taxonIndex++) {
                    
                    String taxonName = (String) i.next();
                    matrix.setTaxonName(taxonIndex, taxonName);
                }
    
                taxonIndex = 0;
                for (Iterator i = matrixCommand.argumentIterator(); 
                            i.hasNext(); taxonIndex++) {
                    
                    i.next();
                    String characters = (String) i.next();
                    matrix.addCharacterValues(taxonIndex, characters);
                }
                
                for (Iterator i = characterDescriptions.argumentIterator(); 
                            i.hasNext(); ) {
                    
                    int characterIndex = Integer.parseInt((String)i.next()) - 1;
                    
                    String characterDescription = (String)i.next();
                    
                    characterDescription = 
                        NEXUS_UNDERSCORE_PATTERN.matcher(
                                characterDescription).replaceAll(" ");
                    
                    matrix.setCharacterDescription(characterIndex,
                            characterDescription);
                }
                
            } catch (IndexOutOfBoundsException e) {
                throw new IllegalActionException("Index out of bounds.");
            }

            // send the new character matrix
            characterMatrixPort.broadcast(new ObjectToken(matrix));            
        }
    }
 
    private static final long serialVersionUID = 1L;
}
