/* Actor that parses Data block of a Nexus-formatted string.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.actors.phylogeny;

import java.util.Iterator;

import org.nddp.exceptions.IndexOutOfBoundsException;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.TaxonomicCharacter;
import org.nddp.util.Port;

import ptolemy.actor.TypedIOPort;
import ptolemy.data.ObjectToken;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.actor type="phylogenetics"
 *
 *  @author Timothy M. McPhillips
 */

public class ParseNexusDataBlock extends NexusParsingActor {

    public ParseNexusDataBlock(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException  {
        
        super(container, name);

        // create an output port for sending the character matrix
        characterMatrixPort = new TypedIOPort(this, "characterMatrix", false, true);
        characterMatrixPort.setTypeEquals(BaseType.OBJECT);
        
        // precompile the pattern for finding the named Nexus block
        setBlockName("DATA");
        }

    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////

    public TypedIOPort characterMatrixPort;
    
    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////
    
    public void fire() throws IllegalActionException {
        
        super.fire();
        
        if (! actorEnabled()) {
            return;
        }

        // read the nexus file content string from the input port       
        String nexusFileString = Port.getString(nexusFilePort, 0, true);

        // extract the characters block
        String dataBlockString = extractNexusBlock(nexusFileString);
        
        // parse the characters block if it was found
        if (dataBlockString != null) {
            
            NexusCommand dimensionsCommand = NexusCommand.extractNexusCommand("DIMENSIONS", dataBlockString);
            
            NexusCommand formatCommand = NexusCommand.extractNexusCommand("FORMAT", dataBlockString);
            
            NexusCommand matrixCommand = NexusCommand.extractNexusCommand("MATRIX", dataBlockString);
            
            int taxonCount = matrixCommand.argumentCount()/2;
            
            int characterCount = 
                Integer.parseInt(dimensionsCommand.getProperty("NCHAR"));
            
            CharacterMatrix matrix = 
                new CharacterMatrix(taxonCount, characterCount);
            
            String dataTypeString = formatCommand.getProperty("DATATYPE");
            
            if (dataTypeString.equals("PROTEIN")) {
                matrix.setDataType(TaxonomicCharacter.PROTEIN_DATA);
            } else if (dataTypeString.equals("DNA")) {
                matrix.setDataType(TaxonomicCharacter.DNA_DATA);
            } else if (dataTypeString.equals("RNA")) {
                matrix.setDataType(TaxonomicCharacter.RNA_DATA);
            } else {
                throw new IllegalActionException("Unsupported Nexus data type:" + dataTypeString);
            }

            matrix.setMissingSymbol(
                    formatCommand.getProperty("MISSING").charAt(0));
            
            matrix.setGapSymbol(
                    formatCommand.getProperty("GAP").charAt(0));
            
            try {
                
                int taxonIndex = 0;
                for (Iterator i = matrixCommand.argumentIterator(); 
                            i.hasNext(); i.next(), taxonIndex++) {
                    
                    String taxonName = (String) i.next();
                    matrix.setTaxonName(taxonIndex, taxonName);
                }
    
                taxonIndex = 0;
                for (Iterator i = matrixCommand.argumentIterator(); 
                            i.hasNext(); taxonIndex++) {
                    
                    i.next();
                    String characters = (String) i.next();
                    matrix.addCharacterValues(taxonIndex, characters);
                }

            } catch (IndexOutOfBoundsException e) {
                throw new IllegalActionException("Index out of bounds.");
            }
                        
            // send the new character matrix
            characterMatrixPort.broadcast(new ObjectToken(matrix));            
        }
    }
 
    private static final long serialVersionUID = 1L;
}
