/* Base class for actors that buffer whole collections before forwarding them.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

package org.nddp.coactors;

import org.nddp.CollectionPortBuffer;
import org.nddp.Collection;
import org.nddp.CollectionManager;
import org.nddp.CollectionIOPort;
import org.nddp.CollectionTypes;
import org.nddp.exceptions.CollectionException;
import org.nddp.util.Parameters;
import org.nddp.CollectionHandler;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.coactor type="general"
 */
public class CollectionBuffer extends CollectionTransformer {

    public CollectionBuffer(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException  {

    	    // call the superclass constructor
        	super(container, name);

        	bufferedCollectionType = Parameters.stringParameter(this, 
        		"bufferedCollectionType", "");
    }

	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

    	public Parameter bufferedCollectionType;
    	
	public static final BufferDisposition DISCARD_BUFFER = 
	    new BufferDisposition();
	
	public static final BufferDisposition FORWARD_BUFFER = 
	    new BufferDisposition();

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
  
	public void initialize() throws IllegalActionException {
 
	    	// call superclass constructor
    		super.initialize();

		_setInputPort(input);
        	_bufferedCollection = null;
        	_bufferedPort.clear();
        	_flushOutstanding = false;
        	_discardRequested = false;
    }

   	public boolean postfire() throws IllegalActionException {
  	    
  	    if (_flushOutstanding) {
  	        _bufferedPort.flushToPort(output);
  	        _flushOutstanding = false;
  	    } else if (_discardRequested) {
  	        _bufferedPort.clear();
  	        _discardRequested = false;
  	    }
  	    
  	    return true;
  	}
  	
   	///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////
   	
    public CollectionIOPort _getDefaultOutputPort() {	
		return output;
    }

    protected BufferDisposition _handleBufferingEnd(CollectionManager 
        collectionManager) throws IllegalActionException, CollectionException {

		return FORWARD_BUFFER;
	}

    protected void  _handleBufferingStart() {
   	}   	
    
   	public void _handleCollectionEnd(CollectionManager collectionManager)
   		throws IllegalActionException, CollectionException {
	    
	    if (collectionManager.collection() == _bufferedCollection) {
	        
	        _bufferedCollection = null;
	        
	        	if (_handleBufferingEnd(collectionManager) == FORWARD_BUFFER) { 
				_flushOutstanding = true;
    			} else { 
  	        		_discardRequested = true;
	        	}
	    }
	}
   	
   	public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager 
   	        collectionManager) throws CollectionException {
	    
	    	Collection collection = collectionManager.collection();
   	    
	    if (_bufferedCollection == null && 
            _bufferedCollectionType.isInstance(collection)) {
	        
	        _bufferedCollection = collection;
	    }

    		if (_bufferedCollection != null) {
	        	collectionManager.setOutputPort(_bufferedPort);
    		}
    		
    		_handleBufferingStart();
	    
        return CollectionHandler.PROCESS_AND_FORWARD_COLLECTION;
   	}

  	public void _handleParameterChange(Parameter parameter, Token value) 
  		throws IllegalActionException {
  	    
		if (parameter == bufferedCollectionType) {
        		_bufferedCollectionType = CollectionTypes.valueOfToken(value);
        	} else {
			super._handleParameterChange(parameter, value);
        	}
    }
   	
//  protected boolean _bufferingInProgress() {
//   	    return _bufferedCollection != null;
//   	}

   	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private Collection _bufferedCollection;
	private Class _bufferedCollectionType;
	private CollectionPortBuffer _bufferedPort = new CollectionPortBuffer();
	private boolean _discardRequested;
	private boolean _flushOutstanding;
	
   	///////////////////////////////////////////////////////////////////
	////                       public inner classes                ////

	public static class BufferDisposition {
	    	private BufferDisposition() {}
	}
}
