/* TODO One line description of class.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 

package org.nddp.coactors;

import java.io.IOException;
import java.util.Map;

import org.nddp.CollectionManager;
import org.nddp.CollectionIOPort;
import org.nddp.CollectionTypes;
import org.nddp.DataTypes;
import org.nddp.DomainObject;
import org.nddp.exceptions.ExternalApplicationException;
import org.nddp.graphics.DcsGraph;
import org.nddp.util.Parameters;
import org.nddp.CollectionHandler;

import ptolemy.data.RecordToken;
import ptolemy.data.Token;
import ptolemy.data.ScalarToken;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import ptolemy.data.expr.Parameter;

/**
 * @nddp.actor actorType="general"
 */
public class CollectionGraph extends CollectionSink {

	public CollectionGraph(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException  {

        super(container, name);

        /* the string displayed at the top of the graph */
        title = Parameters.stringParameter(this, "title", "");
    
        xElementLabel = 
            Parameters.stringParameter(this, "xRecordElement", "");
        
        yElementLabels = 
            Parameters.stringArrayParameter(this, "yElementLabels");
        
        /* the x axis label */
        xLabel = Parameters.stringParameter(this, "xLabel", "x");        

        /* the y axis label and default associated with new subtraces */
        yLabel = Parameters.stringParameter(this, "yLabel","y");
    
        /* a string of configuration parameters for the DCSGraph widget */
        configuration = 
            Parameters.stringParameter(this, "configuration", "-gridOn true");
    
        /* the root of the unique name given to each trace */
        traceRoot = Parameters.stringParameter(this, "traceRoot", "xy");
    
        /*  the minimum and maximum values along the x axis to display */			
        defaultLimitsX = 
            Parameters.stringParameter(this, "defaultLimitsX", "{} {}");
    
        /* the minimum and maximum values along the x axis to display */
        defaultLimitsY = 
            Parameters.stringParameter( this, "defaultLimitsY","{} {}");    

        /* should successive traces should be overlayed or discarded at each 
         * execution ? */
        overlayTraces = 
            Parameters.booleanParameter(this, "overlayTraces", true);
    
        /* should the DCS graph widget be updated after each data point? */
        drawSynchronously = 
            Parameters.booleanParameter(this, "drawSynchronously", true);
    
        /* sets the colors of subtraces */
        subtraceColors = 
            Parameters.stringArrayParameter(this, "subtraceColors");

        /*  sets the y axis labels associated with subtraces */
        subtraceLabels = 
            Parameters.stringArrayParameter(this, "subtraceLabels");
    
         /* the width of the window containing the graph widget */
         widgetWidth = Parameters.intParameter(this, "widgetWidth", 600);

         /* the height of the window containing the graph widget */
         widgetHeight = Parameters.intParameter(this, "widgetHeight", 400);
         
         /* the graph will be cleared for each delimiter received with 
          * the indicated content type */
         clearGraphOn = Parameters.stringParameter(this, "clearGraphOn", "");
         
         /* the existing traces will be hidden whenever a delimiter is 
          * received with the indicated content type */
        	hideTracesOn = 
        	    Parameters.stringParameter(this, "hideTracesOn", "");        

		/* specify the shape of the icon representing this actor using svg */
        _attachText("_iconDescription", "<svg>\n" +
            "<rect x=\"0\" y=\"0\" "
            + "width=\"60\" height=\"90\" "
            + "style=\"fill:white\"/>\n" +
            "</svg>\n");
    }

	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

    public Parameter clearGraphOn;
    public Parameter configuration;
    public Parameter defaultLimitsX;
    public Parameter defaultLimitsY;
    public Parameter drawSynchronously;
    public Parameter hideTracesOn;
    public Parameter overlayTraces;
    public Parameter subtraceColors;
    public Parameter subtraceLabels;
    public Parameter title;
    public Parameter traceRoot;
    public Parameter widgetHeight;
    public Parameter widgetWidth;
    public CollectionIOPort x;
    public Parameter xElementLabel;
    public Parameter xLabel;
    public Parameter yElementLabels;
    public Parameter yLabel;
    
	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

    public void initialize() throws IllegalActionException {
        
        // call the superclass's initialize method
        super.initialize();
 		
        /* start up the graph application if it has never run or if it is 
         * no longer running */
        if ( _dcsGraph.isRunning() == false ) {
            
            _dcsGraph.start();

            // load in the serialized graph data if overlaying graphs
	        if (_overlayTraces == true && _dcsGraph.traceIndex() > 0 ) {
	            _dcsGraph.restoreFromTemporaryFile();
	        } else {
	            _dcsGraph.purgeTemporaryFile();
	        }

	        _customizeGraph();
        }
        
        // clear the graph and reset the trace index counter if not 
        // overlaying graphs
        if ( _overlayTraces == false && _dcsGraph.traceIndex() > 0 ) {
	        _dcsGraph.clear();
        }

        // get the root of the trace name
        _dcsGraph.setTraceNameRoot(Parameters.stringValue(traceRoot));
        
        _newTraceName = null;
        _newTraceRequested = true;
        _clearGraphRequested = false;
        _hideTracesRequested = false;
    }

    
    public void wrapup() throws IllegalActionException {
      
        // serialize the graph contents if overlaying traces
        if ( _dcsGraph.runningAsOfLastCheck() && _overlayTraces == true ) {
            try {
                _dcsGraph.saveToTemporaryFile();
            } catch (IOException e) {
                	throw new IllegalActionException(
                		"Error saving temporary file");
            }
        } else {
            _dcsGraph.purgeTemporaryFile();
        }

        _dcsGraph.flushCommands();

        // call the superclass method
        super.wrapup();
    }

   	///////////////////////////////////////////////////////////////////
	////                       protected methods                   ////

    public void _handleCollectionEnd(CollectionManager collectionManager) 
		throws ExternalApplicationException {
	    
		_dcsGraph.flushCommands();
	}

    public CollectionHandler.CollectionDisposition _handleCollectionStart(
   	        CollectionManager collectionManager) {
	
		if (_clearGraphOn.isInstance(collectionManager.collection())) {
		    _clearGraphRequested = true;
		}
		
		if (_hideTracesOn.isInstance(collectionManager.collection())) {
		    _hideTracesRequested = true;
	    	}
		
		_newTraceName = collectionManager.collection().name();
		_newTraceRequested = true;
		
   		return CollectionHandler.PROCESS_AND_DISCARD_COLLECTION;
	}
	
    public CollectionHandler.TokenDisposition _handleData(CollectionManager collectionManager, 
        Token token) throws IllegalActionException {
  	    
		if (token instanceof RecordToken) {
		    
		 	RecordToken recordToken = (RecordToken)token;

		    if (_newTraceRequested) {
			    _startNewTraces();
			}
	
	        	// write the x value to the graph application
	  	    _trace.add(((ScalarToken)recordToken.get(
	  	            _xElementLabel)).doubleValue());
				
			// iterate over the list of y record elements
	        for (int i = 0; i < _yElementLabels.length; i++ ) {
	           
	            _subtraces[i].add(((ScalarToken)recordToken.get(
	                _yElementLabels[i])).doubleValue());
	    		}
        }
		
		return CollectionHandler.DISCARD_TOKEN;
  	}
	    
    public CollectionHandler.TokenDisposition _handleDomainObject(CollectionManager 
        collectionManager, DomainObject object) throws IllegalActionException {
   	    
		if (_newTraceRequested) {
		    _startNewTraces();
		}

		_trace.add(DataTypes.getDoubleProperty(object, _xElementLabel));
		
		// iterate over the list of y record elements
        for (int i = 0; i < _yElementLabels.length; i++ ) {
           
            _subtraces[i].add(DataTypes.getDoubleProperty(object, 
                _yElementLabels[i]));
    		}

        return CollectionHandler.DISCARD_TOKEN;
    }

    public void _handleParameterChange(Parameter parameter, Token value) 
   		throws IllegalActionException {
        
        /* handle changed attribute based on its identity */
        if (parameter == drawSynchronously) {
        		_dcsGraph.setDrawSynchronously(Parameters.booleanValue(value));
        
        } else if (parameter == overlayTraces) {
        		_overlayTraces = Parameters.booleanValue(value);
        
        } else if (parameter == clearGraphOn) {
     		_clearGraphOn = CollectionTypes.valueOfToken(value);
     		
        } else if (parameter == hideTracesOn) {
        		_hideTracesOn = CollectionTypes.valueOfToken(value);
        		
        } else if (parameter == subtraceColors) {
            _subtraceColorsMap = Parameters.stringMapValue(value);

        } else if (parameter == subtraceLabels) {
            _subtraceLabelsMap = Parameters.stringMapValue(value);
            
        } else if (parameter == xElementLabel) {
            _xElementLabel = Parameters.stringValue(value);
            
        } else if (parameter == yElementLabels) {
            	_yElementLabels = Parameters.stringArrayValue(value);
			_subtraces = new DcsGraph.Trace.Subtrace[_yElementLabels.length];

        } else if (parameter == title || 
                		parameter == xLabel || 
                		parameter == yLabel || 
                		parameter == configuration || 
                		parameter == defaultLimitsX || 
                		parameter == defaultLimitsY || 
                		parameter == widgetWidth || 
                		parameter == widgetHeight ) {
            
            _customizeGraph();
            
        } else {
            super._handleParameterChange(parameter, value);
        }
    }

   	///////////////////////////////////////////////////////////////////
	////                         private methods                   ////
	
  	// TODO adapt to handleParameterChanged mode
    private void _customizeGraph() throws IllegalActionException {
		    
		if ( _dcsGraph != null  && _dcsGraph.isRunning() ) {
		    
		    _dcsGraph.setTitle(Parameters.stringValue(title) );
			_dcsGraph.setXLabel(_xLabel = Parameters.stringValue(xLabel));
			_dcsGraph.setYLabel(_yLabel = Parameters.stringValue(yLabel));
			_dcsGraph.setConfiguration(Parameters.stringValue(configuration));
			_dcsGraph.setZoomDefaultX(Parameters.stringValue(defaultLimitsX));
			_dcsGraph.setZoomDefaultY(Parameters.stringValue(defaultLimitsY));
			_dcsGraph.setWindowSize(Parameters.intValue(widgetWidth), 
				Parameters.intValue(widgetHeight));
		}
    }
	
    private void _startNewTraces()  throws IllegalActionException {
  
    		// clear new trace request flag
        _newTraceRequested = false;
       
    		if ( _clearGraphRequested ) {
	        _dcsGraph.clear();
    			_clearGraphRequested = false;
    		}
    	
    		if ( _hideTracesRequested ) {		        
	      	_dcsGraph.hideAllTraces();
    			_hideTracesRequested = false;
    		}
    		
        /* create the new trace */
        _trace = _dcsGraph.createTrace(_newTraceName, _xLabel);
                
        for (int i = 0; i < _yElementLabels.length; i++ ) {
        
			String subtraceName = _yElementLabels[i];
			String subtraceAxisLabels;
			
			// construct the label string associated with the port
			if (_subtraceLabelsMap.containsKey(subtraceName)) {
			
			    // the label string is that set for the port in the 
			    // subtraceLabels parameter if so specified
			    subtraceAxisLabels = 
			        (String)_subtraceLabelsMap.get(subtraceName);
			} else {
			    // otherwise it is the current label of the y axis
			    subtraceAxisLabels = "{" + _yLabel + "}";
			}
            
			_subtraces[i] = _trace.createSubtrace(subtraceName, 
			        subtraceAxisLabels);
                
			if ( _subtraceColorsMap.containsKey(subtraceName) ) {
        			_subtraces[i].setColor(
        				(String)_subtraceColorsMap.get(subtraceName));
            	}
        }
    }
    
	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private Class _clearGraphOn;
	private boolean _clearGraphRequested;
	private DcsGraph _dcsGraph = new DcsGraph();	    
	private Class _hideTracesOn;
	private boolean _hideTracesRequested;
	private String _newTraceName;
	private boolean _newTraceRequested;
	private boolean _overlayTraces;
	private Map _subtraceColorsMap;
	private Map _subtraceLabelsMap;
	private DcsGraph.Trace.Subtrace[] _subtraces;
	private DcsGraph.Trace _trace;
    private String _xElementLabel;
	private String _xLabel;
	private String[] _yElementLabels;
	private String _yLabel;
}
