/* Actor that computes statistics about record elements in collections.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.coactors;

import java.security.InvalidParameterException;

import org.nddp.AtomicCoactor;
import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.util.Parameters;

import ptolemy.data.DoubleToken;
import ptolemy.data.IntToken;
import ptolemy.data.RecordToken;
import ptolemy.data.ScalarToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.coactor type="general"
 */
public class RecordStatistics extends CollectionTransformer {
	
	public RecordStatistics(CompositeEntity container, String name)
		throws NameDuplicationException, IllegalActionException  {
		
		super(container, name);
	
        label = Parameters.stringParameter(this, "label", "");
	}
	
	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

	public Parameter label;

	///////////////////////////////////////////////////////////////////
	////                         public methods                    ////     

	public void initialize() throws IllegalActionException {
		super.initialize();
		
		_count = 0;
		_sum = 0;
	}
   	
	///////////////////////////////////////////////////////////////////
	////                      protected methods                    ////     

    public void _handleCollectionEnd(CollectionManager collectionManager) 
		throws IllegalActionException {
    			
    		collectionManager.addMetadata("count", new IntToken(_count));

    		if (_count > 0) {		
			collectionManager.addMetadata("mean", 
		        		new DoubleToken(_sum/_count));
		}

		_count += 
		    ((Integer)(collectionManager.getAttribute("count"))).intValue();
		
		_sum += 
		    ((Double)(collectionManager.getAttribute("sum"))).doubleValue();
	}   
    
    public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager 
        collectionManager) throws IllegalActionException {

		collectionManager.setAttribute("count", new Integer(_count));
		_count = 0;

		collectionManager.setAttribute("sum", new Double(_sum));
		_sum = 0;
		
		return  CollectionHandler.PROCESS_AND_FORWARD_COLLECTION;
  	} 

    public CollectionHandler.TokenDisposition _handleData(CollectionManager 
        collectionManager, Token token) throws IllegalActionException {
       
      	if (! (token instanceof RecordToken)) {
      	    return  CollectionHandler.FORWARD_TOKEN;
      	}
      	
      	Token elementToken = ((RecordToken)token).get(_label);
      	
      	if (elementToken == null) {
      	    return  CollectionHandler.FORWARD_TOKEN;
      	}
      	    
      	if ( ! (elementToken instanceof ScalarToken)) {
      	    
  	        throw new InvalidParameterException(
  	                "Token is not a scalar type: " + elementToken);
  	    }

      	_count++;
		_sum += ((ScalarToken)elementToken).doubleValue();

		return  CollectionHandler.FORWARD_TOKEN;
    }	
    
    public void _handleParameterChange(Parameter parameter, Token value) 
		throws IllegalActionException {
            
        if (parameter == label ) {
            _label = Parameters.stringValue(value);               
        } else {
            super._handleParameterChange(parameter, value);
        }
    }

  	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
		
    private String _label;
	private int _count;
	private double _sum;
}
