/* Actor that writes TextFileCollections to disk files.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

 package org.nddp.coactors;

import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import org.nddp.AtomicCoactor;
import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.exceptions.FileWriteException;
import org.nddp.util.FileOrUrl;
import org.nddp.util.Parameters;

/**
 * @nddp.coactor type="io"
 */
public class TextFileWriter extends CollectionSink {
 
    public TextFileWriter(CompositeEntity container, String name)
        throws IllegalActionException, NameDuplicationException {
        
        super(container, name);

        directory = Parameters.stringParameter(this, "directory", "$HOME");
        fileName = Parameters.stringParameter(this, "fileName", "output.txt");
        overwriteFiles = 
            Parameters.booleanParameter(this, "overwriteFiles", false);
        
        _enableParameterOverride(directory);
        _enableParameterOverride(fileName);

        Parameters.fix(collectionPath, "TextFile/String");
        
        _attachText("_iconDescription", "<svg>\n"
                + "<rect x=\"-25\" y=\"-20\" "
                + "width=\"50\" height=\"40\" "
                + "style=\"fill:white\"/>\n"
                + "<polygon points=\"-15,-10 -12,-10 -8,-14 -1,-14 3,-10"
                + " 15,-10 15,10, -15,10\" "
                + "style=\"fill:red\"/>\n"
                + "</svg>\n");
    }
    
	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

    	public Parameter directory;
    	public Parameter fileName;
    	public Parameter overwriteFiles;

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
	    
	public void initialize() throws IllegalActionException {
 
	    // call superclass constructor
        	super.initialize();
        	
        	_fileWriter = null;
	}    
	
	///////////////////////////////////////////////////////////////////
	////                      protected methods                    //// 
	    
    public void _handleCollectionEnd (CollectionManager collectionManager) 
		throws FileWriteException  {
	        
	    try {
			_fileWriter.close();
			
	    } catch(IOException ex) {
	        
	        throw new FileWriteException(
	                "Error closing file " + _fileName + ".");
	        
	    } finally {
	        
			_fileWriter = null;
	    }
	}
	
    public  CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager 
        collectionManager) throws FileWriteException {
   		 
   	    //TODO prevent file overwrites if parameter so requests
    		try {
   	    	    
   	   	    	File file = new File(_directory, _fileName);
		    	file.createNewFile();
		    _fileWriter = new FileWriter(file);
		    
    		} catch(IOException ex) {
   	        	throw new FileWriteException("Error creating file " + 
   	        	        _fileName + " in directory " + _directory + ".");
   	    	}
   	    	
    		return  CollectionHandler.PROCESS_AND_DISCARD_COLLECTION;
	}
 
    public CollectionHandler.TokenDisposition _handleData(CollectionManager
        collectionManager, Token token) throws FileWriteException {
		
        String text = ((StringToken)token).stringValue();
        
        try {
    			_fileWriter.write(text);
    			
        	} catch(IOException ex) {
        	    
        	    	throw new FileWriteException(
        	    	        "Error writing to file " + _fileName + ".");
        	}
    	    
        	return  CollectionHandler.DISCARD_TOKEN;
    	}
    	
    public void _handleParameterChange(Parameter parameter, Token value) 
		throws IllegalActionException {
            
  	    // handle changes directory parameter
        if (parameter == directory ) {
            
        		_directory = Parameters.stringValue(value);
        		_directory = FileOrUrl.substituteSpecialStrings(_directory);
        	
        	// handle changes in fileName parameter
        	} else if (parameter == fileName ) {
        		_fileName = Parameters.stringValue(value);
        	
        	// handle all other attribute changes
        	} else {
            	super._handleParameterChange(parameter,value);
        	}
    }
	
	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private String _directory;
	private String _fileName;
	private FileWriter _fileWriter;
    	private boolean _overwriteFiles;
}
