/* Actor that transforms Nexus collections to text file collections.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

package org.nddp.coactors.phylogeny;

import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;

import org.nddp.Collection;
import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.DomainObject;
import org.nddp.TextFileCollection;
import org.nddp.coactors.CollectionTransformer;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.NexusCollection;
import org.nddp.phylogeny.Tree;
import org.nddp.phylogeny.WeightVector;
import org.nddp.util.Parameters;

import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.coactor type="phylogenetics"
 */
public final class NexusFileComposer extends CollectionTransformer {

    public NexusFileComposer(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException  {

	    super(container, name);
	    
	    abbreviateNodeNames = Parameters.booleanParameter(
	            this, "abbreviateNodeNames", true);
	    hideBranchLengths = Parameters.booleanParameter(
	            this, "hideBranchLengths", true);

	    Parameters.fix(collectionPath, "Nexus//");
	}

	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

	public Parameter abbreviateNodeNames;
	public Parameter hideBranchLengths;

	///////////////////////////////////////////////////////////////////
	////                      protected methods                    //// 

    public void _handleCollectionEnd(CollectionManager collectionManager)
		throws IllegalActionException {
	
	    if (collectionManager == _nexusCollectionManager) {
	        
	        Collection nexusFileCollection = new TextFileCollection(
                	_nexusCollectionManager.collection().name());
	        
	        CollectionManager nexusFileCollectionManager = 
	            manageNewCollection(nexusFileCollection, 
	                _nexusCollectionManager.parentCollectionManager());
	        
	        _textBuffer = new StringBuffer();
	        _indentLevel = 0;
	        
	        _appendLine("#NEXUS");
	        _appendLine("[Written by Beagle NexusComposer]");
	        _appendBlankLine();
	    
	        	_appendTaxaBlock();
	    	    _appendBlankLine();
    	
	        	if (_characterMatrix != null) {
	        	    
	        	    if (_characterMatrix.dataType().isMolecularData()) {
	        	       	_appendDataBlock();
	        	    } else {
		        		_appendCharactersBlock();
	        	    }

	        	    _appendBlankLine();
	        	}
	        
	        	_appendAssumptionsBlock();
	        	_appendBlankLine();
	        	
	        	if (! _treeList.isEmpty()) {
	        	    _appendTreesBlock();
	        	}
	        	
        		nexusFileCollectionManager.addDataToken(
        		        new StringToken(_textBuffer.toString()));
    			
	        	_nexusCollectionManager = null;
	    }    
	}
	
    public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager 
        collectionManager) throws IllegalActionException {
    
    		if (collectionManager.collection() instanceof NexusCollection) {
    		    
    		    _nexusCollectionManager = collectionManager;
    		    _treeList = new LinkedList();
    		    _characterMatrix = null;
    		    _weightSet = null;
        }
            
        return  CollectionHandler.PROCESS_AND_DISCARD_COLLECTION;
    }
	    
    public CollectionHandler.TokenDisposition _handleDomainObject(CollectionManager 
        collectionManager, DomainObject object) throws IllegalActionException {
 	        
        if (object instanceof CharacterMatrix) {
	    
            _characterMatrix = (CharacterMatrix) object;
        
        } else if (object instanceof WeightVector) {
            
            _weightSet = (WeightVector)object;
            
        } else if (object instanceof Tree) {
            
            _treeList.add((Tree)object);
        }

        return  CollectionHandler.DISCARD_TOKEN;
    }

    public void _handleParameterChange(Parameter parameter, Token value) 
		throws IllegalActionException {
	    
	    if (parameter == hideBranchLengths) {
	        _hideBranchLengths = Parameters.booleanValue(value);
	    } else if (parameter == abbreviateNodeNames) {
	        _abbreviateNodeNames = Parameters.booleanValue(value);
	    } else {
	        super._handleParameterChange(parameter, value);
	    }
	}

	///////////////////////////////////////////////////////////////////
	////                         private methods                   ////
	
	private void _append(String text) {

	    _textBuffer.append(text);
	}

	private void _appendAssumptionsBlock() {

	    	_appendLine("BEGIN ASSUMPTIONS;");
	    	_increaseIndent();
		_appendLine("OPTIONS DEFTYPE=unord PolyTcount=MINSTEPS;");
		
		if (_weightSet != null ) {
		    
		    _appendLineStart("WTSET * UNTITLED =");
		    _appendPropertySet(_weightSet);
		    _appendLineEnd(";");
		}
		
		_decreaseIndent();
		_appendLine("END;");
	}

	private void _appendBlankLine() {
	    _textBuffer.append("\n");
	}
	
	private void _appendCharactersBlock() {
	    
	    	_appendLine("BEGIN CHARACTERS;");
	    	_increaseIndent();
	    	_appendLine("DIMENSIONS  NCHAR=" + 
	    	        _characterMatrix.characterCount() + ";");

	    	_appendLineStart("FORMAT SYMBOLS=\"");
	    	
	    	for (Iterator i = _characterMatrix.allowedSymbolsSet().iterator();
	    			i.hasNext(); ) {
	    	    
	    	    _append(" " + (Character)i.next());	    	    
	    	}
	    
	    	_appendLineEnd("\"  MISSING=" + 
	    	        _characterMatrix.missingStateCharacter() +
            		"  GAP=" + _characterMatrix.gapStateCharacter() + ";");
	    
	    	_appendLine("CHARSTATELABELS");
	    
	    	for (int i = 0, characterCount = _characterMatrix.characterCount();
	    			i < characterCount; i++) {
	    	    
	    	    String underscoreReplacedDescription = 
	    	        NEXUS_SPACE_PATTERN.matcher(
    	                _characterMatrix.characterDescription(i)).
    	                	replaceAll("_");
	    	    
	    	    	_appendLineStart((i+1) + "  " + 
	    	    	        underscoreReplacedDescription);
	    	    	
	    	    	if (i + 1 < characterCount) {
	    	    	    _appendLineEnd(",");
	    	    	} else {
	    	    	    _appendLineEnd("");
	    	    	}
	    	}
	    	
    		_appendLine(";");
    		_appendLine("MATRIX");
    		_increaseIndent();
    		
    		int maxNameLength = 0;
	    	for (int i = 0, taxonCount = _characterMatrix.taxonCount(); 
	    			i < taxonCount; i++) {
	    	    
	    	    int length = _characterMatrix.taxonName(i).length();
	    	    if (maxNameLength < length) {
	    	        maxNameLength = length;
	    	    }
	    	}

	    	for (int i = 0, taxonCount = _characterMatrix.taxonCount(); 
    	    			i < taxonCount; i++) {
	    	    
			StringBuffer paddedTaxonName = 
			    new StringBuffer("                                          ");
    	        	paddedTaxonName.insert(0, _characterMatrix.taxonName(i));
    	        	paddedTaxonName.setLength(maxNameLength + 1);
    	        
	    	    	_appendLine(paddedTaxonName.toString() +	
	    	    	        _characterMatrix.characterVector(i).toString());	    	
	    	}
	    	
    	    _decreaseIndent();
	    _appendLine(";");
	    _decreaseIndent();
	    _appendLine("END;");
	}
	
	private void _appendDataBlock() {
	    
	    	_appendLine("BEGIN DATA;");
	    	_increaseIndent();
	    	_appendLine("DIMENSIONS" +
    	        	"  NTAX=" + _characterMatrix.taxonCount() + 
    	        	"  NCHAR=" + _characterMatrix.characterCount() + ";");

	    	_appendLine("FORMAT  DATATYPE=" + _characterMatrix.dataType().toString() +
    	        	"  MISSING=" + _characterMatrix.missingStateCharacter() +
        		"  GAP=" + _characterMatrix.gapStateCharacter() + ";");
	    	    	
    		_appendLine("MATRIX");
    		_increaseIndent();
    		
    		int maxNameLength = 0;
	    	for (int i = 0, taxonCount = _characterMatrix.taxonCount(); 
	    			i < taxonCount; i++) {
	    	    
	    	    int length = _characterMatrix.taxonName(i).length();
	    	    if (maxNameLength < length) {
	    	        maxNameLength = length;
	    	    }
	    	}

	    	for (int i = 0, taxonCount = _characterMatrix.taxonCount(); 
    	    			i < taxonCount; i++) {
	    	    
			StringBuffer paddedTaxonName = 
			    new StringBuffer("                                          ");
    	        	paddedTaxonName.insert(0, _characterMatrix.taxonName(i));
    	        	paddedTaxonName.setLength(maxNameLength + 1);
    	        
	    	    	_appendLine(paddedTaxonName.toString() +	
	    	    	        _characterMatrix.characterVector(i).toString());	    	
	    	}
	    	
    	    _decreaseIndent();
	    _appendLine(";");
	    _decreaseIndent();
	    _appendLine("END;");
	}
	
	private void _appendLine(String text) {
	    
	    _appendLineStart(text);
	    _appendLineEnd("");
	}
	
	private void _appendLineEnd(String text) {
	    
	    _textBuffer.append(text + "\n");
	}
	
	private void _appendLineStart(String text) {
	    
	    for (int i = 0; i < _indentLevel; i++ ) {
	        _textBuffer.append("    ");
	    }
	    
	    _textBuffer.append(text);
	}
	
	private void _appendPropertySet(WeightVector propertySet) {
	    
	    Set valueSet = new HashSet();
	    
	    for (int i = 0; i < propertySet.size(); i++) {
	        valueSet.add(new Integer(propertySet.get(i))); 
	    }
	    
	    boolean firstValue = true;
	    
	    for (Iterator i = valueSet.iterator(); i.hasNext(); ) {
	        
	        int value = ((Integer)(i.next())).intValue();
	        
	        if ( ! firstValue) {
	            _append(",");
	        } else {
	            firstValue = false;
	        }
	        
	        _append(" " + value + ":");

	        int lastMatchingIndex = Integer.MAX_VALUE;

	        boolean inSeries = false;
	        
	        for (int j = 0; j < propertySet.size(); j++) {
	                
				if ( propertySet.get(j) == value) {
	                
					if (lastMatchingIndex == j - 1) {
						inSeries = true;
					} else {
	             		_append(" " + (j + 1));
	             		inSeries = false;
	                	}
	                	
	                	lastMatchingIndex = j;
	            
	            	} else if (inSeries && lastMatchingIndex == j - 1) {
	                
	                	_append("-" + j + " ");
	            	}
	            	
	        }
	    }
	}

	private void _appendTaxaBlock() {
        
	    _appendLine("BEGIN TAXA;");
	    _increaseIndent();
	    _appendLine("DIMENSIONS  NTAX=" + _characterMatrix.taxonCount() + ";");
	    
	    _appendLine("TAXLABELS");
	  	_increaseIndent();
	    
	    for (int i = 0; i < _characterMatrix.taxonCount(); i++ ) {
            _appendLine(_characterMatrix.taxonName(i));
	    }
	    
	    _decreaseIndent();
	    
	    	_appendLine(";");
	    _decreaseIndent();
	    	_appendLine("END;");
	}

	private void _appendTreesBlock() {
	    
	    	_appendLine("BEGIN TREES;");
	    	_increaseIndent();
	    	
	    	if (_abbreviateNodeNames) {
		    	_appendLine("TRANSLATE");
		    	_increaseIndent();
		    	
		    	for (int i = 0; i < _characterMatrix.taxonCount(); i++) {
		    	    
				StringBuffer indexBuffer = new StringBuffer("    ");
				String indexString = Integer.toString(i + 1);
				indexBuffer.replace(5 - indexString.length(), 4, indexString);

				_appendLineStart(indexBuffer.toString() + "  " +
	    	            _characterMatrix.taxonName(i));

		    	    if (i < _characterMatrix.taxonCount() - 1) {
		    	        _appendLineEnd(","); 
		    	    } else {
		    	        _appendLineEnd("");
		    	    }
		    	}
	
		    	_decreaseIndent();
		    	_appendLine(";");
	    	}
	    	
	    	for (Iterator i = _treeList.iterator(); i.hasNext(); ) {
	    	    
	    	    Tree tree = (Tree)i.next();
	    	    _appendLine("TREE " + tree.id() + " = " +
    	            tree.newickString(_abbreviateNodeNames, 
    	                    _hideBranchLengths, _characterMatrix) + ";");
	    	}
	    	
	    	_decreaseIndent();
		_appendLine("END;");
	}

	private void _decreaseIndent() {
	    assert _indentLevel > 0;
	    _indentLevel--;
	}
		
	private void _increaseIndent() {
	    _indentLevel++;
	}

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
		
	private boolean _abbreviateNodeNames;
	private CharacterMatrix _characterMatrix;
	private boolean _hideBranchLengths;
	private int _indentLevel;
	private CollectionManager _nexusCollectionManager;
	private StringBuffer _textBuffer;
	private List _treeList;
	private WeightVector _weightSet;
    private static final Pattern NEXUS_SPACE_PATTERN = Pattern.compile(" +");
    
    private static final long serialVersionUID = 1L;
}