/* Wrapper class for DCS graph command-line application.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.graphics;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.nddp.util.ExternalProcess;
import org.nddp.util.ProcessEnvironment;

import ptolemy.kernel.util.IllegalActionException;


final public class DcsGraph {
    
    public DcsGraph() {
    	
        _environment = new ProcessEnvironment(_ENVIRONMENT_STRING_ARRAY);
    }
        
	///////////////////////////////////////////////////////////////////
	////                         public methods                    ////
    
    	public void clear() { 
    	    _traceIndex = 0;
    	    _nextColorIndex = 0;
    	    _write("clear"); 
	}

    	public Trace createTrace(String traceName, String xAxisLabel) {
    	    
    	    return new Trace( traceName, xAxisLabel );
    	}
    	
    	public void flushCommands() {

	    if ( _process != null ) {
	        try {    
	            _process.flushStandardInput();        
	        } catch (java.io.IOException ex) {
	        }
    		}
	}

	public void hideAllTraces() { 
	    _write("hideAllTraces"); 
    }
	
    public boolean isRunning() throws IllegalActionException {

        if ( _process == null ) return false;
        
        try {  
            // write and flush a null command to the DCSGraph wrapper    
            _process.write("noop\n");
            _process.flushStandardInput();
            
        } catch (java.io.IOException ex) {
        
            // record the absence of a graph application if exception is thrown
            _process = null;
        }

        return _process != null;
    }

    public void purgeTemporaryFile() {
        
    	    if (_temporaryFile != null ) {
	  		_deleteGraphFile(_temporaryFile.getAbsolutePath());
	  		_temporaryFile = null;
    	    }
    	}
       	
    	public void restoreFromTemporaryFile() {
    	    
    	    if (_temporaryFile != null ) {
	    	    _restoreGraphFromFile(_temporaryFile.getAbsolutePath());
    		}
    	}

	public boolean runningAsOfLastCheck() { 
	    
	    return _process != null; 
    }

    	public void saveToTemporaryFile() throws IOException {
		
    	    File temporaryFile = File.createTempFile("beagle", ".tmp");
		_saveGraphToFile(temporaryFile.getAbsolutePath());
		temporaryFile.deleteOnExit();
    	}
    	
	public void setConfiguration(String configuration) { 
	    
	    _write("configure " + configuration); 
    }

	public void setDrawSynchronously(boolean drawSynchronously) { 
	    
	    _drawSynchronously = drawSynchronously;
    }	

	public void setTitle(String title) { 
	    
	    _write("configure -title {" + title + "}" ); 
    }

	public void setTraceNameRoot(String root) {
	    
    		_traceRoot = root;
    }    

	public void setWindowSize(int width, int height) { 
	    
	    _write("size " + width + " " + height); 
    }		

	public void setXLabel(String xLabel) { 
	    
	    _write("configure -xLabel {" + xLabel + "}" ); 
    }

    public void setYLabel(String yLabel) { 
        
        _write("configure -yLabel {" + yLabel + "}" ); 
    }

	public void setZoomDefaultX(String xLimits) { 
	    
	    _write("setZoomDefaultX " + xLimits); 
    }
     
	public void setZoomDefaultY(String yLimits) { 
	    
	    _write("setZoomDefaultX " + yLimits); 
    }

	public void start() {

		try {
    			_process = 
    			    _environment.startProcess(_COMMAND, null, null);
    			
		} catch(IOException ex) {		    	
            _process = null;
		}
	}	

    	public int traceIndex() {
    	    return _traceIndex;
    	}
    	
	///////////////////////////////////////////////////////////////////
	////                         private methods                   ////
    
    private void _appendToArray(String arrayName, double value) {
        
        _write ("append " + arrayName + " " + value); 
    }

    private void _deleteGraphFile(String filename) { 
        
        _write("deleteFile " + filename); 
    }

    	private void _restoreGraphFromFile(String filename) { 
    	    
    	    _write("openFile " + filename); 
    }

    private void _saveGraphToFile(String filename) { 
        
        _write("saveFile " + filename); 
    }

    private void _setSubtraceColor(String traceName, String subtraceName, 
        String color) {
        
    		_write("configureSubTrace " + traceName + " " + subtraceName +
	        " -color " + color );
    }
    
	private void _write(String command) {
    
	    if ( _process != null ) {
		    
	        try {    
	            	// write the command to the standard input of the graph 
	            // application
	            	_process.write(command + "\n");
				if (_drawSynchronously) _process.flushStandardInput();
	
	        } catch (java.io.IOException ex) {
	        
	            // record the absence of a graph application if an io 
	            // exception is thrown
	            _process = null;
	        }
    		}      
	}	
   	
   	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
	
    	private static final String _COMMAND = 
    	    "/sw/bin/wish /Volumes/Workspace/EclipseWorkspace/kepres/lib/DcsWidgets/DCSGraphRunner.tcl";
    	
    	private boolean _drawSynchronously = true;	
	private final ProcessEnvironment _environment;

    	// TODO Make this work for any Tcl environment
  	private static final String[] _ENVIRONMENT_STRING_ARRAY = {
		"DISPLAY=" + System.getProperty("DISPLAY"), 
		"TCLLIBPATH=/Volumes/Workspace/EclipseWorkspace/kepres/lib/DcsWidgets",
		"HOME=" + System.getProperty("HOME")};
  	
	private static String[] _graphColor = 
		{"black", "red","green", "blue", "purple", "brown", "orange", "pink"};
	
   	private int _nextColorIndex = 0;
	private ExternalProcess _process = null;
   	private File _temporaryFile;	
	private int _traceIndex = 0;
    	private String _traceRoot = "trace";
	
	///////////////////////////////////////////////////////////////////
	////              public inner classes for DCSGraph            ////
    	
    public class Trace {
        
        private Trace(String traceName, String xAxisLabel) {

            	_traceIndex++;
         
			if (traceName != null) {
            		_traceName = traceName;
        		} else {
            		_traceName = _traceRoot + _traceIndex;
        		}
        
	        	_traceArrayName = "x_" + _traceIndex + "_tracearray";

	        	_createDcsArray(_traceArrayName);
	        	_createDcsTrace(_traceName, xAxisLabel, _traceArrayName);
		}
	
		///////////////////////////////////////////////////////////////////
		////           public methods for DCSGaph.Trace                //// 
        
        	public void add(double value) {
        	    
			_appendToArray(_traceArrayName, value);
        	}  
        	
        	public void add(String subtraceName, double value) {
        	    
        	    Subtrace subtrace = 
        	        (Subtrace)_subtraceNameMap.get(subtraceName);
        	    subtrace.add(value);
        	}	

        	public Subtrace createSubtrace(String subtraceName, 
    	        String subtraceAxisLabels) {
        	    
        	    return new Subtrace(subtraceName, subtraceAxisLabels);
        	}
        	
        	public String name() { 
        	    return _traceName; 
    	    }
    	        	
        	public void setColor(String subtraceName, String color) {
        	    
        	    Subtrace subtrace = 
        	        (Subtrace)_subtraceNameMap.get(subtraceName);
        	    _setSubtraceColor(_traceName, subtraceName, color);
        	}
	
		///////////////////////////////////////////////////////////////////
		////            private methods for DCSGaph.Trace              ////
        	
        private void _createDcsArray(String arrayName) {
            
            _write("array " + arrayName); 
        }
       
	    	private void _createDcsTrace(String traceName, String label, 
    	        	String arrayName) {
	    	    
			_write("createTrace " + traceName + 
		        " {{" + label + "}} " + arrayName );
		}

	    	
		///////////////////////////////////////////////////////////////////
		////            private variables for DCSGaph.Trace            ////
		
		private Map _subtraceNameMap = new HashMap();
		private final String _traceArrayName;
        	private final String _traceName;
		
		///////////////////////////////////////////////////////////////////
		////           public inner classes for DCSGaph.Trace          ////

		public class Subtrace {

			private Subtrace(String subtraceName, String subtraceAxisLabels) {
    		        
    		        _subtraceName = subtraceName;
    		        _subtraceNameMap.put(_subtraceName, this);
    		        
    		        _subtraceArrayName = _subtraceName + "_" + 
    		        		_traceIndex + "_subarray";
    		        
    		        _createDcsArray(_subtraceArrayName);
    		        
    		        _createDcsSubtrace(_traceName, _subtraceName, 
	                subtraceAxisLabels, _subtraceArrayName);
    		        
                // get the color of the subtrace from the next element of the
    		        // graph color array
                String color = _graphColor[_nextColorIndex++];
                
                if ( _nextColorIndex == _graphColor.length ) {
                    _nextColorIndex = 0;
                }
            
                // configure the subtrace accordingly in the graph application
                _setSubtraceColor( _traceName, _subtraceName, color );
    		    	}
    		    	
    		    	
			///////////////////////////////////////////////////////////////////
			////       public methods for DCSGraph.Trace.SubTrace          ////
		
	        	public void add(double value) {
	        	    
				_appendToArray(_subtraceArrayName, value);
	        	} 
        			
            	public void setColor(String color) {
            	    
                _setSubtraceColor(_traceName, _subtraceName, color);
            	}

			///////////////////////////////////////////////////////////////////
			////      private methods for DCSGraph.Trace.SubTrace          ////
		
        		private void _createDcsSubtrace(String traceName, String subtraceName, 
    		        String subtraceAxisLabels, String subtraceArrayName ) {
        		    
	     		_write("createSubTrace " + traceName + " " + subtraceName + 
	     		        " {" + subtraceAxisLabels + "} "  + subtraceArrayName);
    			}
    		    		
			///////////////////////////////////////////////////////////////////
			////       public variables for DCSGraph.Trace.SubTrace        ////
						
			private final String _subtraceName;
    		    private final String _subtraceArrayName;
    		}
    }   
}
  
