/* Class representing a taxonomic character matrix.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.phylogeny;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeSet;

import org.nddp.DomainObjectBase;
import org.nddp.exceptions.InconsistentDataException;
import org.nddp.exceptions.IndexOutOfBoundsException;
import org.nddp.util.Xml;

public class CharacterMatrix extends DomainObjectBase {
    
    public CharacterMatrix(CharacterMatrix matrix) {
        
        super();
        
        _taxonCount = matrix._taxonCount;
        
        _characterCount = matrix._characterCount;   
        
        _taxonName = (String[])matrix._taxonName.clone();
        
        _characterVector = (CharacterVector[])matrix._characterVector.clone();
        
        _characterVectorNameMap = 
            (HashMap)matrix._characterVectorNameMap.clone();
        
        _taxonIndexMap = (HashMap)matrix._taxonIndexMap.clone();
        
        _characterDescription = (String[])matrix._characterDescription.clone();
        
        _character = (TaxonomicCharacter[])matrix._character.clone();
        
        _mostCommonStateCount = (int[]) matrix._mostCommonStateCount.clone();
        
        _missingStateCount = (int[])matrix._missingStateCount.clone();
        
        _localGapSymbol = matrix._localGapSymbol;
        
        _localMissingSymbol = matrix._localMissingSymbol;

        _weightVector = new WeightVector(matrix._weightVector);
        
        _dataType = matrix._dataType;
        
        _assertMatrixIsComplete();
    }

    public CharacterMatrix(int taxonCount, int characterCount) {
        
        super();
    
        if (taxonCount < 1) {
            throw new IllegalArgumentException("taxonCount < 1");
        }
        
        if (characterCount < 1) {
            throw new IllegalArgumentException("characterCount < 1");
        }
        
        _taxonCount = taxonCount;
        _characterCount = characterCount;       
        _taxonName = new String[_taxonCount];
        _characterDescription = new String[_characterCount];
        _characterVector = new CharacterVector[_taxonCount];
        _character = new TaxonomicCharacter[_characterCount];
        
        _mostCommonStateCount = new int[_characterCount];
        Arrays.fill(_mostCommonStateCount, 0);
        
        _missingStateCount = new int[_characterCount];
        Arrays.fill(_missingStateCount, 0);
    
        _weightVector =new WeightVector(_characterCount);
    
        for (int i = 0; i < _taxonCount; i++ ) {
            _characterVector[i] = 
                new CharacterVector(_characterCount, this);
        }
        
        _taxonIndexMap = new HashMap();
    }
        
    ///////////////////////////////////////////////////////////////////
	////                       public variables                    //// 

	public static final char STANDARD_GAP_SYMBOL = '-';
    public static final char STANDARD_MISSING_SYMBOL = '?';
        
	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

    public void addCharacterValues(int taxonIndex, String characters) 
        throws IndexOutOfBoundsException {
    
        assertNotWriteLocked();
        
        if (taxonIndex < 0 || taxonIndex >= _taxonCount) 
            throw new IndexOutOfBoundsException("taxonIndex", taxonIndex, 
                0, _taxonCount - 1);
        
        _characterVector[taxonIndex].setNextStates(characters);         
    }

	public Set allowedSymbolsSet() {
	    
	    Set set = new TreeSet();
	    
	    for (int i = 0; i < _characterCount; i++) {
	        set.addAll(character(i).allowedStateSet());
	    }
	    
	    return set;
	}
	
    public void applyWeightSet(WeightVector newWeightVector) 
        throws InconsistentDataException {

        assertNotWriteLocked();
        
        if (newWeightVector.size() != _characterCount) {
            throw new InconsistentDataException(
                "Weight set size does not match character matrix.");
        }
        
        _weightVector = new WeightVector(newWeightVector);
    }

	public TaxonomicCharacter character(int characterIndex) {	    
	    
	    if (_character[characterIndex] == null) {
	        _character[characterIndex] = 
	            new TaxonomicCharacter(_characterDescription[characterIndex],
                    this, characterIndex);
	    }
	    
        return _character[characterIndex];
	}
	
    public final int characterCount() { 
        return _characterCount; 
    }
    
    public String characterDescription(int characterIndex) {
        return _characterDescription[characterIndex];
    }
    
    public CharacterVector characterVector(int characterIndex) {
        return _characterVector[characterIndex];
    }
    
    public CharacterVector characterVector(String name) {
		return (CharacterVector)_characterVectorNameMap.get(name);
    }

    public TaxonomicCharacter.NexusDataType dataType() {
	    return _dataType;
	}

	public final char element(int taxonIndex, int characterIndex) {
	    return _characterVector[taxonIndex].state(characterIndex);
	}

	public char gapStateCharacter() {
	    return _localGapSymbol;
	}

	public boolean isAutapomorphy(int characterIndex) {
	    
        return _mostCommonStateCount(characterIndex) + 
        		missingStateCount(characterIndex) == _taxonCount - 1 ;
	}
	
	public double maximumPossibleChanges(int characterIndex) {
	    
	    return _taxonCount - _mostCommonStateCount(characterIndex) - 
	    		missingStateCount(characterIndex);
	}

	public char missingStateCharacter() {
	    return _localMissingSymbol;
	}
	
	public int missingStateCount(int characterIndex) {
	    
	    if (_missingStateCount[characterIndex] == 0) {

	        	_missingStateCount[characterIndex] = 
	        	    taxaWithState(characterIndex, _localMissingSymbol);
	    }
	    
	    return _missingStateCount[characterIndex];
	}

    public RowIterator rowIterator() { 
        return new RowIterator(); 
    }

    public void setCharacterDescription(int characterIndex, 
            String characterDescription) throws IndexOutOfBoundsException {

        assertNotWriteLocked();

        if (characterIndex < 0 || characterIndex >= _characterCount) 
            throw new IndexOutOfBoundsException("characterIndex", 
                characterIndex, 0, _characterCount - 1);
        
        _characterDescription[characterIndex] = characterDescription;
    }

    public void setDataType(TaxonomicCharacter.NexusDataType type) {
        assertNotWriteLocked();
        _dataType = type;
    }
    
    public void setGapSymbol(char symbol) {
        assertNotWriteLocked();
        _localGapSymbol = symbol;
    }
    
    public void setMissingSymbol(char symbol) {
        assertNotWriteLocked();
        _localMissingSymbol = symbol;
    }

    public void setTaxonName(int taxonIndex, String name) 
        throws IndexOutOfBoundsException {
        
        assertNotWriteLocked();
    
        if (taxonIndex < 0 || taxonIndex >= _taxonCount) 
            throw new IndexOutOfBoundsException("taxonIndex", taxonIndex, 
                0, _taxonCount - 1);
        
        _taxonName[taxonIndex] = name;
        _characterVectorNameMap.put(name, _characterVector[taxonIndex]);
        _taxonIndexMap.put(name, new Integer(taxonIndex));
    }

    public void setWeight(int characterIndex, int weight) {
        assertNotWriteLocked();
        _weightVector.set(characterIndex, weight);
    }

	public int taxaWithState(int characterIndex, char state) {
	    
        _assertMatrixIsComplete();

        int count = 0;
	    
	    for (int taxonIndex = 0; taxonIndex < _taxonCount; taxonIndex++) {
	        if (_characterVector[taxonIndex].state(characterIndex) == state) {
	            count++;
	        }
	    }
	    
	    return count;
	}

    public final int taxonCount() { 
        return _taxonCount; 
    }

	public int taxonIndex(String taxonName) {
	    Integer index = (Integer)_taxonIndexMap.get(taxonName);
	    if (index == null) {
	        return 0;
	    } else {
		    return index.intValue() + 1;
	    }
	}

	public TaxonIterator taxonIterator() { 
	    return new TaxonIterator(); 
    }

	public String taxonName(int taxonIndex) {
	    return _taxonName[taxonIndex];
	}
	
	public String toString() {
        
        	StringBuffer buffer = new StringBuffer("");

		for (int i = 0; i < _taxonCount; i++ ) {
            buffer.append("[" + _characterVector[i] + "]");
        	}
		
		return buffer.toString();
    }

	public String type() {
	    return "CharacterMatrix";
	}
	
	public int weight(int characterIndex) {
	    return _weightVector.get(characterIndex);
	}

	public WeightVector weightVector() {
	    return _weightVector;
	}

	public String xmlAttributeString() {
		return null;
    }
	
	public String xmlContentString(Xml.Indentation indentation) {

	    StringBuffer buffer = new StringBuffer();

	    buffer.append(indentation.toString() + "<Matrix" +
            Xml.attribute("datatype", _dataType.toString()) + 
            Xml.attribute("taxa", _taxonCount) +
            Xml.attribute("characters", _characterCount) +
            Xml.attribute("gap", _localGapSymbol) +
            Xml.attribute("missing", _localMissingSymbol) +
     		">\n");
	    
	    indentation.increase();
	    
		for (int i = 0; i < _taxonCount; i++ ) {
            buffer.append(indentation.toString()+ "<MatrixRow" + 
                Xml.attribute("taxon", _taxonName[i]) +
                Xml.attribute("states", _characterVector[i].toString()) +
                " />\n"); 
        	}
		
		indentation.decrease();
		
		buffer.append(indentation.toString() + "</Matrix>\n");
		
		if (! _dataType.isMolecularData()) {
		    
			buffer.append(indentation.toString() + "<Characters>\n");
		    indentation.increase();
			for (int i = 0; i < _characterCount; i++) {
			    buffer.append(indentation.toString() + "<Character" + 
			    		Xml.attribute("description", _characterDescription[i]) +
//TODO			    		Xml.attribute("states", _character[i].allowedStateSet().toString()) +
			    		" />\n");
			}
			
		    indentation.decrease();
		    
			buffer.append(indentation.toString() + "</Characters>\n");
		}
		
		return buffer.toString();
	}

    public void zeroWeightAutapomorphies() {
        for (int i = 0; i < _characterCount; i++) {
            if (isAutapomorphy(i)) {
                setWeight(i, 0);
                System.out.println("Character " + i + " is an autapormophy");
            }
        }
    }

    ///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////

    protected void _assertMatrixIsComplete() {
        
        if (_matrixIsComplete) return;
        
        for (int i = 0; i < _taxonCount; i++ ) {
            if (! _characterVector[i].isComplete() ) {
                throw new IllegalStateException(
                		"character matrix is incomplete");
            }
        }
        
        _matrixIsComplete = true;
    }

	///////////////////////////////////////////////////////////////////
	////                         protected variables               ////

    	protected TaxonomicCharacter[] _character;
    	protected int _characterCount;    	
	protected String[] _characterDescription;
    	protected CharacterVector[] _characterVector;
	protected HashMap _characterVectorNameMap = new HashMap();
	protected char _localGapSymbol = STANDARD_GAP_SYMBOL;
	protected char _localMissingSymbol = STANDARD_MISSING_SYMBOL;
    	protected int[] _missingStateCount;
    	protected int[] _mostCommonStateCount;
	protected int _taxonCount;
	protected HashMap _taxonIndexMap;
    	protected String[] _taxonName;
    	protected WeightVector _weightVector;
    	protected TaxonomicCharacter.NexusDataType _dataType = null;

	///////////////////////////////////////////////////////////////////
	////                         private methods                   ////
	
	private int _mostCommonStateCount(int characterIndex) {

	    if (_mostCommonStateCount[characterIndex] == 0) {
	        
		    int stateCount = character(characterIndex).allowedStatesCount();
		    int[] count = new int[stateCount];
		    int mostCommonStateIndex = -1;
		    int mostCommonStateCount = -1;
		    
		    // TODO ignore missing data
		    for (int stateIndex = 0 ; stateIndex < stateCount; stateIndex++) {
		        
		        count[stateIndex] = taxaWithState(characterIndex, 
                    _character[characterIndex].allowedState(stateIndex));
		        
		        if (count[stateIndex] > mostCommonStateCount) {
		            mostCommonStateIndex = stateIndex;
		            mostCommonStateCount = count[stateIndex];
		        }
		    }
		    
		    assert mostCommonStateIndex != -1;
		    
		    _mostCommonStateCount[characterIndex] = mostCommonStateCount;
	    }
	    
	    return _mostCommonStateCount[characterIndex];
	}
	
	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
	
	private boolean _matrixIsComplete = false;

	///////////////////////////////////////////////////////////////////
	////                       public inner classes                ////
    	
    	public class RowIterator implements Iterator {
        
		private RowIterator() {}
		
		///////////////////////////////////////////////////////////////////
		////     public methods for CharacterMatrix.RowIterator        ////

		public boolean hasNext() { return _rowIndex < _taxonCount;}
       	public Object next() { return _characterVector[_rowIndex++]; }
		public void remove() { throw new UnsupportedOperationException(); }

		///////////////////////////////////////////////////////////////////
		////     private variables for CharacterMatrix.RowIterator     ////

		private int _rowIndex = 0;
    }
    
    public class TaxonIterator implements Iterator {
        
        private TaxonIterator() {}
        
		///////////////////////////////////////////////////////////////////
		////     public moethods for CharacterMatrix.TaxonIterator     ////

        public boolean hasNext() { return _rowIndex < _taxonCount;}
        public Object next() { return _taxonName[_rowIndex++]; } 
        public void remove() { throw new UnsupportedOperationException(); }
        
		///////////////////////////////////////////////////////////////////
		////     private variables for CharacterMatrix.TaxonIterator   ////
        
        private int _rowIndex = 0;
	}
}
