/* Class representing a Nexus file command.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.phylogeny;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

class NexusCommand {

	public NexusCommand(String originalCommandString) {
	    
	    _originalCommandString = originalCommandString;
        _orderedArguments = new TreeMap();
        _orderedPropertyKeys = new TreeMap();
        _propertyValues = new HashMap();
        
        // remove terminal semicolon
	    String commandString = 
	        _originalCommandString.substring(0, 
	                _originalCommandString.length()-1);
	    
	    commandString = _parseArguments(commandString, 
	            NEXUS_QUOTED_ARGUMENT_PATTERN, false);
	    
	    commandString = _parseProperties(commandString, 
	            NEXUS_QUOTED_PROPERTY_PATTERN);
	    
	    commandString = _parseProperties(commandString, 
	            NEXUS_UNQUOTED_PROPERTY_PATTERN);
	    
    	    commandString = _parseArguments(commandString, 
    	            NEXUS_UNQUOTED_ARGUMENT_PATTERN, true);
    }

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
    
    public int argumentCount() {
        return _orderedArguments.size();
    }
    
    public Iterator argumentIterator() { 
        return new ArgumentIterator(); 
    }

    public String getProperty(String key) {
        
        return (String)_propertyValues.get(key);
    }
	
    public String name() { 
        return _commandName; 
    }
    
    public Iterator propertyIterator() { 
        return new PropertyIterator(); 
    }
		
	public String toString() {
		return _originalCommandString;
	}

	///////////////////////////////////////////////////////////////////
	////                         private methods                   ////

	private String _parseArguments(String commandString, Pattern matchPattern, 
        boolean firstArgumentIsCommandName) {
		
	    Matcher argumentMatcher = matchPattern.matcher(commandString);	    
	    
		if (firstArgumentIsCommandName) {
		    argumentMatcher.find();
	    	    _commandName = argumentMatcher.group().toUpperCase();
		}
		
	    int matchPosition = 0;
	    while (argumentMatcher.find()) {
	        String matchString = argumentMatcher.group(1);
	        matchPosition = 
	            _originalCommandString.indexOf(matchString, matchPosition);
	        _orderedArguments.put(new Integer(matchPosition), matchString);
	        matchPosition += matchString.length();
	    }
	    
	    return  matchPattern.matcher(commandString).replaceAll("");
	}
	
	private String _parseProperties(String commandString, 
        Pattern matchPattern) {
	    
	    Matcher propertyMatcher = matchPattern.matcher(commandString);
	    
	    int matchPosition = 0;
	    while (propertyMatcher.find()) {
	        String matchString = propertyMatcher.group(0);
	        //TODO Consider storing an unmodified value of the key as well.
	        String key = propertyMatcher.group(1).toUpperCase();
	        String value = propertyMatcher.group(2);
	        matchPosition = 
	            _originalCommandString.indexOf(matchString, matchPosition);
	        _orderedPropertyKeys.put(new Integer(matchPosition), key);
	        _propertyValues.put(key,value);
	        matchPosition += matchString.length();
	    }
	    
	    return matchPattern.matcher(commandString).replaceAll("");	
	}

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
	
    	private String _commandName;
    	private final Map _orderedArguments;
    	private final Map _orderedPropertyKeys;
  	private final String _originalCommandString;
	private final Map _propertyValues;
	private static final Pattern NEXUS_QUOTED_ARGUMENT_PATTERN = 
	    Pattern.compile("\'(.*?)\'");
	private static final Pattern NEXUS_QUOTED_PROPERTY_PATTERN = 
	    Pattern.compile("(\\S+)=\"(.*?)\"");
	private static final Pattern NEXUS_UNQUOTED_ARGUMENT_PATTERN = 
	    Pattern.compile("([^,\\s]+)");	
	private static final Pattern NEXUS_UNQUOTED_PROPERTY_PATTERN = 
	    Pattern.compile("(\\S+)=(\\S+)");

	///////////////////////////////////////////////////////////////////
	////                       private inner classes               ////
	
	private class ArgumentIterator implements Iterator {
	    
		///////////////////////////////////////////////////////////////////
		////   public methods for NexusCommand.ArgumentIterator        ////

	    public boolean hasNext() { 
		    return _argumentKeyIterator.hasNext(); 
	    }
		
        	public Object next() { 
        	    return _orderedArguments.get(_argumentKeyIterator.next()); 
    	    }
        	
        	public void remove() {
        	    throw new UnsupportedOperationException(); 
    	    }

		///////////////////////////////////////////////////////////////////
		////   private variables for NexusCommand.ArgumentIterator     ////

        	private Iterator _argumentKeyIterator = 
        	    _orderedArguments.keySet().iterator();
	}
	
	private class PropertyIterator implements Iterator {
	    
		///////////////////////////////////////////////////////////////////
		////   public methods for NexusCommand.PropertyIterator        ////

	    public boolean hasNext() { 
		    return _propertyMetaKeyIterator.hasNext(); 
	    }
		
        	public Object next() { 
        	    return _propertyValues.get(_orderedPropertyKeys.get(
        	            _propertyMetaKeyIterator.next()));
    	    }
        	
        	public void remove() { 
        	    throw new UnsupportedOperationException(); 
    	    }

		///////////////////////////////////////////////////////////////////
		////   private variables for NexusCommand.PropertyIterator     ////

        	private Iterator _propertyMetaKeyIterator = 
        	    _orderedPropertyKeys.keySet().iterator();
	}
}
