/* Class representing a phylogenetic tree.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

package org.nddp.phylogeny;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.nddp.DomainObjectBase;
import org.nddp.exceptions.InconsistentDataException;
import org.nddp.exceptions.ParseException;
import org.nddp.util.Xml;

public class Tree extends DomainObjectBase {

	protected Tree() {
		super();
	}

    public Object clone() {

        Tree clone = null;
        
        try {
            clone = new Tree(newickString(false, false, null), _expandedNameMap);
        } catch (ParseException e) {
            System.out.println(e.getMessage());
        }
                
        return clone;
    }   
    
    public Tree(String newickTreeString, Map expandedNameMap) 
        throws ParseException{

        if (newickTreeString == null) {
            throw new NullPointerException("null newickTreeString");
        }
        
        _nameToNodeMap = new HashMap();
        
        // OK if null
        _expandedNameMap = expandedNameMap;

            // remove comment blocks and white space from tree file string
        newickTreeString = NEWICK_COMMENT_PATTERN.
            matcher(newickTreeString).replaceAll("");
            
        newickTreeString = NEWICK_WHITE_SPACE_PATTERN.
            matcher(newickTreeString).replaceAll("");
        
        newickTreeString = 
            newickTreeString.substring(1, newickTreeString.length() - 1);
        
        if (newickTreeString.indexOf(':') != -1) {
            _distancesDefined = true;
        }
        
        _root = new TreeNode(this, null, 0.0, newickTreeString);
    }

    ///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
	
 	public int equivalenceHashCode() {
	    
	    if (_equivalenceHashCode == 0 || ! isWriteLocked()) {
	        _equivalenceHashCode = _root.equivalenceHashCode();
	    }
	    
	    return _equivalenceHashCode;
	}
 	
	public boolean equivalentTo(Tree tree) {
	    return equivalenceHashCode() == tree.equivalenceHashCode();
	}
    
    public String id() {
	    return _id;
	}
    
	public String newickString(boolean abbreviated, boolean hideLength, CharacterMatrix matrix) {
		return _root.toString(abbreviated, hideLength, matrix);
	}
	
	public int nodeCount() { 
	    return _nodeCount; 
	}
	
	public static void removeEquivalentTrees(List treeList) {

	    for (int i = 0; i < treeList.size(); i++) {
	        
	        Tree treeOne = (Tree)treeList.get(i);
	        
	        for (int j = i + 1; j < treeList.size(); ) {
	        
	            Tree treeTwo = (Tree)treeList.get(j);
	            
	            if (treeOne.equivalentTo(treeTwo)) {
	                treeList.remove(j);
	            } else {
	                j++;
	            }
	        }
	    }   
	}
    public void reroot(String nodeName) throws InconsistentDataException {
        
        assertNotWriteLocked();
        
        TreeNode node = (TreeNode)_nameToNodeMap.get(nodeName);
            
        if (node == null) {
            throw new InconsistentDataException(
                "No node with name: " + nodeName);
        }
        
        if (node != _root) {
            _oldRoot = _root;
            node.reroot(null);
            _root = node;
        }       
    }
	
	public TreeNode root() {
	    return _root;
	}
        
    public void setName(String name) {

        assertNotWriteLocked();

        _id = name;
    }
    
    public void setParentNames(String[] childNameArray, 
            String[] parentNameArray) throws InconsistentDataException {

        assertNotWriteLocked();

        // make a local copy of the parent name array so that it 
        // can be modified
        parentNameArray = (String[]) parentNameArray.clone();
        
        // repeat scan of arrays of names until every node has a name
        while ( _nameToNodeMap.size() < _nodeCount) {
            
            // reset flag indicating whether a name was added on this
            // pass through the name arrays
            boolean nameAdded = false;
            
            // loop over indices of name arrays
            for (int i = 0; i < parentNameArray.length; i++ ) {
                
                // extract the parent and child names corresponding to 
                // the array index
                String parentName = _expandName(parentNameArray[i]);
                String childName = _expandName(childNameArray[i]);
                
                // check if parentName
                if ( parentName != null) {
                    if (! _nameToNodeMap.containsKey(parentName)) {
                        
                        TreeNode child = 
                            (TreeNode)_nameToNodeMap.get(childName);
                        
                        if ( child != null) {
                        
                            TreeNode parent = (TreeNode)child.parent();
                            
                            if (parent != null) {
                                parent._setName(parentName);
                            }
                            
                            parentNameArray[i] = null;
                            nameAdded = true;
                        }
                    }
                }
            }
                
            if (! nameAdded) {
                System.out.println();
                throw new InconsistentDataException("Missing name");
            }
        }
    }   
    
    public String toString() { 
	    return newickString(true, true, null); 
    }
	
	public String type() {
	    return "Tree";
	}

    public String xmlAttributeString() {
	    return Xml.attribute("tree", newickString(true, true, null)); 
    }
	
	public String xmlContentString(Xml.Indentation indentation) {
	    return null;
	}
	
	///////////////////////////////////////////////////////////////////
	////                         protected variables               ////

	protected boolean _distancesDefined = false;
	protected String _id;
	protected int _nodeCount = 0;
	protected TreeNode _root;

    ///////////////////////////////////////////////////////////////////
    ////                         protected methods                 ////

    protected String _expandName(String name) {
        
        String expandedName;
        
        if ( _expandedNameMap != null && (expandedName = 
            (String)_expandedNameMap.get(name)) != null) {
            
            return expandedName;
        
        } else {
        
            return name;
        }
    }
    
	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////	
    
	private int _equivalenceHashCode = 0;
    private Map _expandedNameMap;
    HashMap _nameToNodeMap;
    private static final Pattern NEWICK_COMMENT_PATTERN = 
        Pattern.compile("\\[.*?\\]", Pattern.DOTALL);    
    private static final Pattern NEWICK_WHITE_SPACE_PATTERN = 
        Pattern.compile("\\s*");
    TreeNode _oldRoot;
}
