/* Base class for Phylip tree inference runners.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.phylogeny.phylip;

import java.io.IOException;
import java.util.List;

import org.nddp.exceptions.CollectionException;
import org.nddp.exceptions.ExternalApplicationException;
import org.nddp.exceptions.ExternalEnvironmentException;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.Tree;
import org.nddp.util.ProcessEnvironment;
import org.nddp.util.ProcessRunner;

public class PhylipConsenseRunner extends ProcessRunner {

    public PhylipConsenseRunner() {
        
        super();
   	}

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

	public void rootAtNode(String name) {
	    _rootAtNode = name;
	}
	
	public Object run() throws CollectionException {
		
	    ProcessEnvironment environment = _createProcessEnvironment("CONSENSE");
	    
	    _writeInputFiles(environment);
	    
	    _startProgram(environment, 
	    		System.getProperty("PHYLIP_EXE_DIR") + "/consense");
	    
	    _writeStandardInput();

	    _waitForProgramCompletion();
	    
	    Tree consensusTree = _parseOutputFiles(environment);
	    
	    _destroyEnvironment(environment);
		
	    return consensusTree;
	}
		
	public void setConsensusThreshold(double threshold) {
	    _consensusThreshold = threshold;
	}
	
	public void setExtendendConsensus(boolean extended) {
	    _extendedConsensus = extended;
	}
	
	public void setInputTreeList(List trees) {
	    _inputTreeList = trees;
	}
	
	public void setRootedTrees(boolean rooted) {
	    _rootedTrees = rooted;
	}
	
	///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////
        
    protected void _writeInputFiles(ProcessEnvironment environment) 
    		throws ExternalEnvironmentException {
        
	    _characterMatrix = PhylipFiles.writeTreeFile(
	            _inputTreeList, environment.createInputFile("intree"));
    }
    
    protected void _writeStandardInput() 
    		throws ExternalApplicationException {
        
		try {
			boolean strictConsensus;
			
		    if (! _extendedConsensus && 
		            Math.abs(_consensusThreshold - 1.0) < 0.01 ) {
		        
				strictConsensus = true;
				
		    } else {
		        
		        strictConsensus = false;
		    }
			        
		    if (! _extendedConsensus) {
			    
				if (strictConsensus) {
		        		_writeToProcess("C");
			    } else {
			    		_writeToProcess("C");
			    		_writeToProcess("C");
			    		_writeToProcess("C");
			    }
			}
			    
		    if (_rootedTrees) {
		        _writeToProcess("R");
		    }
		    
		    	// Accept configuration and begin execution
			_writeToProcess("Y");

			if (! _extendedConsensus && ! strictConsensus) {
		    		_writeToProcess(_consensusThreshold + "");
			}

		    	
		} catch (IOException e) {
			throw new ExternalApplicationException(
		        "Error writing to CONSENSE process");
        	}		    
    }
    
    private Tree _parseOutputFiles(ProcessEnvironment environment)
    		throws CollectionException {

	    // Read the output tree file
	    String outtree = environment.readFile("outtree");
        
		// Parse the tree file
		List treeList = PhylipFiles.parseTreeFileString(outtree, null);

	    Tree tree = (Tree)treeList.get(0);
	    
	    if (! _rootAtNode.equals("")) {
		    tree.reroot(_rootAtNode);
	    }
	    
	    tree.setName("CONSENSE_Tree");
	    
	    return tree;
    }

    ///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private String _rootAtNode;	
	private CharacterMatrix _characterMatrix;
	private double _consensusThreshold;
	private boolean _extendedConsensus;
	private boolean _rootedTrees;
	private List _inputTreeList;
}
