/*  Base class for Phylip tree inference actors.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 package org.nddp.phylogeny.phylip;

import java.util.Iterator;
import java.util.List;

import org.nddp.AtomicCoactor;
import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.DomainObject;
import org.nddp.exceptions.CollectionException;
import org.nddp.exceptions.CollectionStructureException;
import org.nddp.exceptions.InconsistentDataException;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.Tree;
import org.nddp.phylogeny.WeightVector;
import org.nddp.util.Parameters;
import org.nddp.util.ProcessRunningActor;

import ptolemy.data.IntToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

public abstract class PhylipTreeInferenceActor extends ProcessRunningActor {

    public PhylipTreeInferenceActor(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException {
        
        	super(container, name);
        
        	ignoreAutopomorphies = Parameters.booleanParameter(
        	        this, "ignoreAutopomorphies", false);
        	
        	outgroupIndex = Parameters.intParameter(this, "outgroupIndex", 1);
        	
		rootAtNode = Parameters.stringParameter(this, "rootAtNode", "");

		Parameters.fix(collectionPath,"Nexus/");

		_enableParameterOverride(new Parameter[] {ignoreAutopomorphies, 
	       	outgroupIndex, rootAtNode} );
    }

	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

	public Parameter ignoreAutopomorphies;
	public Parameter outgroupIndex;
	public Parameter rootAtNode;
      	    
	///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////
	
	protected void _configureRunner() throws InconsistentDataException {

	    PhylipTreeInferenceRunner runner = (PhylipTreeInferenceRunner)_runner;
	    
	    runner.setCharacterMatrix(_characterMatrix);
	    runner.setWeights(_weightSet);
	    runner.ignoreAutopomorphies(_ignoreAutopomorphies);
	    runner.setOutgroup(_outgroupIndex);
	    runner.rootAtNode(_rootAtNode);
	}
	
    public void _handleCollectionEnd(CollectionManager collectionManager) 
		throws IllegalActionException, CollectionException {
    
		if (collectionManager == _nexusCollectionManager) {
		    
		    if (_characterMatrix != null) {
				_runProgram();
			    _addTreesToCollection((List)_runResult);
		    }
		}
    }
  	
// 	protected void _handleCollectionException(CollectionManager 
//        collectionManager, CollectionException exception) {
//        
//		_resetActorState();
//		
//	    super._handleCollectionException(collectionManager, exception);
//    }
    
    public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager
        collectionManager) throws IllegalActionException {
	
  	    _resetActorState();
  	    
  		_nexusCollectionManager = collectionManager;

	    return  CollectionHandler.PROCESS_AND_FORWARD_COLLECTION;
    }
	
    public CollectionHandler.TokenDisposition _handleDomainObject(CollectionManager 
        	fileCollectionManager, DomainObject object) 
		throws CollectionException {

        	if (object instanceof CharacterMatrix) {
        	    
            if (_characterMatrix == null) {
		        _characterMatrix = (CharacterMatrix)object;
            } else {
                throw new CollectionStructureException(
                    "Multiple CharacterMatrix objects in one " +
                    "NexusFileCollection not allowed.");
            }
            
        } else if (object instanceof WeightVector) {
            if (_weightSet == null) {
	            _weightSet = (WeightVector) object;
            } else {
                throw new CollectionStructureException(
                    "Multiple WeightSet objects in one " +
                    "NexusFileCollection not allowed.");
            }
            
        } else if (object instanceof Tree) {
            _treeCount++;
        }

        return  CollectionHandler.FORWARD_TOKEN;
    }
	
    public void _handleParameterChange(Parameter parameter, Token newValue) 
		throws IllegalActionException {
            
        if (parameter == ignoreAutopomorphies) {
        		_ignoreAutopomorphies = Parameters.booleanValue(newValue);
        } else if (parameter == outgroupIndex) {
        		_outgroupIndex = Parameters.intValue(newValue);
        } else if (parameter == rootAtNode) {
        		_rootAtNode = Parameters.stringValue(newValue);
        } else {
            super._handleParameterChange(parameter, newValue);
        }
    } 	

	///////////////////////////////////////////////////////////////////
	////                         private methods                   ////
	
    private void _addTreesToCollection(List treeList) {
	    
	    for (Iterator i = treeList.iterator(); i.hasNext(); ) {
    		    
		    	Tree tree = (Tree)i.next();
    	    		_nexusCollectionManager.addDomainObject(tree);
		}

	    _treeCount += treeList.size();
	    
		_nexusCollectionManager.addMetadata(
	        "treeCount", new IntToken(_treeCount));
	}
    
    private void _resetActorState() {
        
        	_characterMatrix = null;
        	_nexusCollectionManager = null;
        	_treeCount = 0;
        	_weightSet = null;    	    
    	}
 
	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
  	  	
	private CharacterMatrix _characterMatrix;
	private boolean _ignoreAutopomorphies;
	private CollectionManager _nexusCollectionManager;
	private int _outgroupIndex;	
	private String _rootAtNode;
	private int _treeCount;
	private WeightVector _weightSet;
}
