/* Atom class for protein structures.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

package org.nddp.proteins;

import java.text.NumberFormat;

import org.nddp.DomainObjectBase;
import org.nddp.util.Xml;

public final class ProteinAtom extends DomainObjectBase {

	private ProteinAtom() {}

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
	
	public String atomName() {
	    return _atomName;
	}

    public int atomNumber() {
	    return _atomNumber;
	}
	
	public String atomType() {
	    return _atomType;
	}
	
    	public double b() {
    		return _b;
    	}
    	
    	public String chainID() {
		return _chainID;
	}
	
	public String id() {
	    return null;
	}
    
	public String label() {
	    return _label;
	}

    	public double occupancy() {
    		return _occupancy;
    	}

    	public int residueNumber() {
	    return _residueNumber;
	}

	public String residueType() {
	    return _residueType;
	}
	
	public static ProteinAtom valueOfPDBAtomLine(String PDBAtomLine) {
		
	    ProteinAtom atom = new ProteinAtom();
	    
	    // parse the atom label ("ATOM" or "HETA")
        atom._label = PDBAtomLine.substring(0,6).trim();
            
        // parse the atom number (unique for each atom in the PDB file)
        atom._atomNumber = Integer.parseInt(PDBAtomLine.substring(6,11).trim());
           
        // parse the atom name (e.g., "CA" for the C-alpha)
        atom._atomName = PDBAtomLine.substring(11,16).trim();
            
        // parse the residue type (e.g., "TYR" for tyrosine)
        atom._residueType = PDBAtomLine.substring(16,20).trim();
        
        // parse the chain ID (blank for PDB files with only one chain, 
        // otherwise a single letter)
        atom._chainID = PDBAtomLine.substring(20,22).trim();
        if (atom._chainID.equals("") ) {
        		atom._chainID = "null";
        }
        
        // parse the residue number
        atom._residueNumber = Integer.parseInt(PDBAtomLine.substring(22,26).trim());
            
        // parse the x coordinate of the atom
        atom._x = Double.parseDouble(PDBAtomLine.substring(26,38));
            
        // parse the y coordinate of the atom
        atom._y = Double.parseDouble(PDBAtomLine.substring(38,46));
            
        // parse the z coordinate of the atom
        atom._z = Double.parseDouble(PDBAtomLine.substring(46,54));

        // parse the occupancy of the atom (0.0 - 1.0)
        atom._occupancy = Double.parseDouble(PDBAtomLine.substring(54,60));
            
        // parse the temperature factor for the atom
        atom._b = Double.parseDouble(PDBAtomLine.substring(60,66));
            
        // parse the atom type (e.g., "C" for carbon atoms)
        atom._atomType = PDBAtomLine.substring(66,78).trim();
        
        return atom;
	}
    
    public String toString() {
        
        return xmlAttributeString();
    }
    
    public String xmlAttributeString() {
        
        return Xml.attribute("label", _label) + 
        		Xml.attribute("atomNumber", _atomNumber) + 
        		Xml.attribute("atomName", _atomName) + 
        		Xml.attribute("residueType", _residueType) + 
        		Xml.attribute("chainID", _chainID) +
        		Xml.attribute("residueNumber", _residueNumber) + 
        		Xml.attribute("x",_x, _xyzFormat) + 
        		Xml.attribute("y",_y, _xyzFormat) + 
        		Xml.attribute("z",_z, _xyzFormat) +
        		Xml.attribute("occupancy", _occupancy, _bAndOccFormat) +
        		Xml.attribute("b", _b, _bAndOccFormat) + 
        		Xml.attribute("atomType", _atomType);
    }
    
    public String xmlContentString(Xml.Indentation indentation) {
        return null;
    }

    public double x() {
    	    return _x;
    	}
    	
    	public double y() {
    	    return _y;
    	}
    	
    	public double z() {
    	    return _z;
    	}
    	
    	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private String _atomName;
	private int _atomNumber;
	private String _atomType;
	private double _b;
	private static NumberFormat _bAndOccFormat;
	private String _chainID;
	private String _label;
	private double _occupancy;
	private int _residueNumber;
	private String _residueType;
	private static NumberFormat _xyzFormat;
	private double _x;
	private double _y;
	private double _z;
	
    	///////////////////////////////////////////////////////////////////
	////                   static initializers                     ////

	static {
        	_xyzFormat = NumberFormat.getNumberInstance();
        	_xyzFormat.setMaximumFractionDigits(3);
        	_xyzFormat.setMinimumFractionDigits(3);
        
        	_bAndOccFormat = NumberFormat.getNumberInstance();
        	_bAndOccFormat.setMaximumFractionDigits(2);
       	_bAndOccFormat.setMinimumFractionDigits(2);
	}
}
