/* Class representing a protein sequence.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.proteins;

import java.util.HashMap;
import java.util.List;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.StringTokenizer;

import org.nddp.DomainObjectBase;
import org.nddp.util.Xml;

public class ProteinSequence extends DomainObjectBase {
    
    public ProteinSequence() {
    }
    
    public ProteinSequence(String sequenceString) {

        // TODO add support for one amino acid sequence
        addResidues(sequenceString);
    }

	public ProteinSequence(ProteinSequence sequence) {
	    
	    _id = sequence._id;
	    _sequenceList.addAll(sequence._sequenceList);
	}

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
    
    public void addResidue(String residueString) {
        
        assertNotWriteLocked();
        
        String trimmedResidueString = residueString.trim();
        String residue = null;
        
        int length = trimmedResidueString.length();
        if ( length == 1 ) {
            residue = trimmedResidueString; 
        } else if ( length == 3 ) {
            residue = toOneLetterCode(trimmedResidueString);
        } else {
        }
        
        if ( residue != null ) {
            _sequenceList.add(residue);
        }
    }
    
    public void addResidues(String sequenceString) {
        
        assertNotWriteLocked();

        String trimmedSequenceString = sequenceString.trim();
        if (trimmedSequenceString.charAt(3) == ' ') {
            _parseThreeLetterCodedSequence(trimmedSequenceString);
        } else {
            _parseOneLetterCodedSequence(trimmedSequenceString);
        }
    }

    public String id() {
        return _id;
    }
    
    public int length() {
        return _sequenceList.size();
    }

    public void setId(String id) {
        assertNotWriteLocked();
        _id = id;
    }
    
    public static String toOneLetterCode(String threeLetterSymbol) {

    		return (String)_threeLetterToOneLetterMap.	get(
	        threeLetterSymbol.toUpperCase()); 
    }

	public String toString() {
	
		StringBuffer sequenceString = new StringBuffer();

		for (Iterator i = _sequenceList.iterator(); i.hasNext(); ) {
			sequenceString.append(i.next());	
		}
		
		return sequenceString.toString();
	}

    public String xmlAttributeString() {
        
		return Xml.attribute("sequence", toString());
    }
    
    public String xmlContentString(Xml.Indentation indentation) {
        return null;
    }

    public static String toThreeLetterCode(String oneLetterSymbol) {
    	
        return (String)_oneLetterToThreeLetterMap.get(
            oneLetterSymbol.toUpperCase());
    }

    public String type() {
	    return "ProteinSequence";
	}

	///////////////////////////////////////////////////////////////////
	////                       protected variables                 ////

    protected String _id;
    protected static Map _oneLetterToThreeLetterMap;
    protected List _sequenceList = new LinkedList();
    protected static Map _threeLetterToOneLetterMap;

    
    ///////////////////////////////////////////////////////////////////
    ////                         private methods                   ////
    
    private void _parseOneLetterCodedSequence(String trimmedSequenceString) {
        
            for (int i = 0, length = trimmedSequenceString.length(); 
                i < length; i++) {
                
                String residue = trimmedSequenceString.substring(i,i+1);
             _sequenceList.add(residue);
        }
        }
    
    private void _parseThreeLetterCodedSequence(String trimmedSequenceString) {
    
        for (StringTokenizer st = new StringTokenizer(trimmedSequenceString); 
                st.hasMoreTokens(); ) {
        
                String residue = toOneLetterCode(st.nextToken());
             _sequenceList.add(residue);
        }
    }
    
    ///////////////////////////////////////////////////////////////////
	////                 static initializers                       ////

    static {

        _threeLetterToOneLetterMap = new HashMap();
        
        _oneLetterToThreeLetterMap = new HashMap();
        
        String[] aaTranslationArray = { 
            "ALA", "A", "ASP", "D", "ASN", "N", "ARG", "R", 
            "CYS", "C", "HIS", "H", "GLU", "E", "GLN", "Q",
            "GLY", "G", "ILE", "I", "LEU", "L", "LYS", "K", 
            "MET", "M", "PHE", "Y", "PRO", "P", "SER", "S", 
            "THR", "T", "TRP", "W", "TYR", "Y", "VAL", "V" };

        for (int i =0; i < aaTranslationArray.length; i+=2) {
            
	        _threeLetterToOneLetterMap.put(
	                aaTranslationArray[i], aaTranslationArray[i+1] ); 
	        
	        _oneLetterToThreeLetterMap.put(
	                aaTranslationArray[i+1], aaTranslationArray[i] );
        }
    }
}
