/** Timestamp returns the current date and time.
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.sdm.spa;

import java.text.SimpleDateFormat;
import java.util.Date;

import ptolemy.actor.lib.Source;
import ptolemy.data.StringToken;
import ptolemy.data.expr.StringParameter;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

//////////////////////////////////////////////////////////////////////////
//// Timestamp
/**
 * Returns the current date and time in "yyyy-MM-dd z HH:mm:ss" format.
 *
 * @author Ilkay Altintas
 * @version $Id: Timestamp.java,v 1.5 2006/04/04 22:01:13 altintas Exp $
 *
 * Trigger: Any token received along this port will cause the actor to fire.
 *
 * Output: The current timestamp is output through this port.
 */

public class Timestamp extends Source {

	/** Construct a constant source with the given container and name.
	 *  Create the <i>value</i> parameter, initialize its value to
	 *  the default value of an IntToken with value 1.
	 *  @param container The container.
	 *  @param name The name of this actor.
	 *  @exception IllegalActionException If the entity cannot be contained
	 *   by the proposed container.
	 *  @exception NameDuplicationException If the container already has an
	 *   actor with this name.
	 */
	public Timestamp(CompositeEntity container, String name)
		throws NameDuplicationException, IllegalActionException {
		super(container, name);

		// Set the type constraint.
		output.setTypeEquals(BaseType.STRING);

		format = new StringParameter(this, "format");
		format.setExpression("yyyy-MM-dd z HH:mm:ss");
		format.addChoice("yyyy-MM-dd z HH:mm:ss");
		format.addChoice("EEE, d MMM yyyy HH:mm:ss Z");
		format.addChoice("yyyyMMddHHmmssS");

		_attachText(
			"_iconDescription",
			"<svg>\n"
				+ "<rect x=\"0\" y=\"0\" "
				+ "width=\"60\" height=\"20\" "
				+ "style=\"fill:white\"/>\n"
				+ "</svg>\n");
	}

	public StringParameter format;
	///////////////////////////////////////////////////////////////////
	////                         public methods                    ////
        /** Send the token in the <i>value</i> parameter to the output.
         *  @exception IllegalActionException If it is thrown by the
         *   send() method sending out the token.
         */
        public void fire() throws IllegalActionException {
                super.fire();
                Date currentDatetime = new Date(System.currentTimeMillis());
                String dateFormat = format.getExpression();
                SimpleDateFormat formatter = new SimpleDateFormat(dateFormat);
                String myDate = formatter.format(currentDatetime);
                output.send(0, new StringToken(myDate));
        }

	/** Post fire the actor. Return false to indicated that the
	 *  process has finished. If it returns true, the process will
	 *  continue indefinitely.
	 */
	public boolean postfire() {
		return true;
	}
}
