/*
 Copyright (c) 1997-2005 The Regents of the University of California.
 All rights reserved.
 Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.

 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.

 PT_COPYRIGHT_VERSION_2
 COPYRIGHTENDKEY

 */
package ptolemy.kernel.util;


import java.io.IOException;
import java.util.ResourceBundle;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.kepler.moml.NamedObjId;
import java.util.MissingResourceException;


//////////////////////////////////////////////////////////////////////////
//// XMLIconConfig

/**
 Configuration Utilities for ComponentEntity class, which is a base class for
 all actors.

 @author Matthew Brooke
 @version $Id: ComponentEntityConfig.java,v 1.17 2006/02/27 08:25:59 brooke Exp $
 @since Ptolemy II 0.2
 @Pt.ProposedRating
 @Pt.AcceptedRating
 */
public class ComponentEntityConfig {

  /**
   * attribute name for attribute added to actors
   * to describe (batik-renderable) SVG image file location
   */
  public static final String SVG_ICON_ATTRIB_NAME = "_svgIcon";

  /**
   * attribute name for attribute added to actors
   * to describe raster thumbnail image file location
   */
  public static final String RASTER_THUMB_ATTRIB_NAME = "_thumbnailRasterIcon";

  /**
   * attribute name for LSID added to actors
   */
  public static final String ACTOR_LSID_ATTRIB_NAME = "entityId";


  ///////////////////////////////////////////////////////////////////


  /**
   * Key for finding default actor-icon name in resource bundle
   * (example value associated with this key: "basic-actor")
   */
  private static final String DEFAULT_ACTOR_ICON_KEY = "DEFAULT_ACTOR_ICON";

  /**
   * Key for finding default director-icon name in resource bundle
   * (example value associated with this key: "director")
   */
  private static final String DEFAULT_DIRECTOR_ICON_KEY
    = "DEFAULT_DIRECTOR_ICON";

  private static String DEFAULT_ACTOR_ICON_BASENAME;
  private static String DEFAULT_DIRECTOR_ICON_BASENAME;

  private static Log log;
  private static boolean isDebugging;

  static {
    // logging...
    log = LogFactory.getLog("SVG." + ComponentEntityConfig.class.getName());
    isDebugging = log.isDebugEnabled();

    try {
      DEFAULT_ACTOR_ICON_BASENAME
        = StaticResources.getUISettingsBundle().getString(DEFAULT_ACTOR_ICON_KEY);
    } catch (Exception ex) {
      if (isDebugging) {
        log.debug("EXCEPTION GETTING DEFAULT_ACTOR_ICON_BASENAME: "
                  + ex.getMessage());
      }
    }
    try {
      DEFAULT_DIRECTOR_ICON_BASENAME
        = StaticResources.getUISettingsBundle().getString(DEFAULT_DIRECTOR_ICON_KEY);
    } catch (Exception ex) {
      if (isDebugging) {
        log.debug("EXCEPTION GETTING DEFAULT_DIRECTOR_ICON_BASENAME: "
                  + ex.getMessage());
      }
    }
  }


  private static final String PERIOD = ".";

  ///////////////////////////////////////////////////////////////////
  ////                         public methods                    ////
  ///////////////////////////////////////////////////////////////////


  public static synchronized void addSVGIconTo(NamedObj namedObj) throws
    IOException, IllegalArgumentException,
    NameDuplicationException, IllegalActionException {

    if (StaticResources.getUISettingsBundle() == null) {
      throw new IOException("Could not access actor icon mappings: "
                            + StaticResources.UI_SETTINGS_BUNDLE);
    }
    if (namedObj == null) {
      throw new IllegalArgumentException("addSVGIconTo() received NULL arg");
    }

    // LSID attribute has not been set by the time this is called, so try to get
    // a generic icon definition from byClass resourcebundle:
    String iconBaseName = tryToAssignIconByClass(namedObj);

    // if iconBaseName is null or is empty string, there was no entry in the
    // byClass resourcebundle so try to get a default icon
    if (iconBaseName == null || iconBaseName.trim().length() < 1) {
      iconBaseName = getDefaultIconBaseName(namedObj);
    }

    if (iconBaseName == null) {
      if (isDebugging) {
        log.warn("Couldn't assign batik icon for: " + namedObj.getClassName());
      }
      return;
    }
    addRasterAndSVGAttributes(namedObj, iconBaseName);
  }



  /**
   * Look for an LSID in an Attribute, and if it exists, try to find an icon
   * mapping in the appropriate resource bundle
   *
   * @param container NamedObj
   * @throws IOException
   * @throws NameDuplicationException
   * @throws IllegalActionException
   */
  public static void tryToAssignIconByLSID(NamedObj container) throws
    IOException, NameDuplicationException, IllegalActionException {

    if (container == null) {
      return;
    }

    NamedObjId lsid
      = (NamedObjId)container.getAttribute(ACTOR_LSID_ATTRIB_NAME);
    String iconBaseName = null;
    if (lsid != null) {

      if (isDebugging) {
        log.debug("\n\n*** FOUND LSID (" + lsid.getExpression()
                 + ") for: " + container.getClassName());
      }
      try {
        iconBaseName
          = getByLSIDResBundle().getString(lsid.getExpression());
        if (isDebugging) {
          log.debug("*** icon (" + iconBaseName + ") selected by LSID ("
                    + lsid.getExpression() + ") for: "
                    + container.getClassName());
        }
      } catch (Exception ex) {
        log.debug(ex.getMessage());
        iconBaseName = null;
      }
    }

    if (iconBaseName == null) {
      return;
    }

    addRasterAndSVGAttributes(container, iconBaseName);
  }



  ///////////////////////////////////////////////////////////////////
  ////                         private methods                   ////
  ///////////////////////////////////////////////////////////////////


  private static String tryToAssignIconByClass(NamedObj namedObj) throws
    IOException {

    String iconBaseName = null;
    if (getByClassResBundle() != null) {
      try {
        iconBaseName = getByClassResBundle().getString(namedObj.getClassName());
        if (isDebugging) {
          log.debug("icon (" + iconBaseName + ") selected by ClassName for: "
                    + namedObj.getClassName());
        }
      } catch (Exception ex1) {
        log.debug(ex1.getMessage());
        iconBaseName = null;
      }
    }
    return iconBaseName;
  }


  private static void addRasterAndSVGAttributes(NamedObj namedObj,
                                                String iconBaseName) throws
    IllegalActionException, NameDuplicationException, IOException {

    // If iconBaseName is null or empty, just abort; either a previously-
    // assigned icon, or default icon from Ptolemy base classes will be used.
    if (iconBaseName == null || iconBaseName.trim().length() < 1) {
      if (isDebugging) {
        log.debug("Using base class default icon for: "
                  + namedObj.getClassName());
      }
      return;
    } else {
      if (isDebugging) {
        log.debug("*** FOUND ICON MAPPING (" + iconBaseName
                  + ") FOR: " + namedObj.getClassName());
      }
    }

    // gets here only if iconBaseName is not null or empty...

    // add base SVG icon
    _addIconAttributeToNamedObj(namedObj, SVG_ICON_ATTRIB_NAME,
                                "SVG_ICON_BASE_PATH", iconBaseName,
                                "SVG_BASE_ICON_EXT");

    //now add raster thumbnail icon
    _addIconAttributeToNamedObj(namedObj, RASTER_THUMB_ATTRIB_NAME,
                                "RASTER_THUMBNAIL_BASE_PATH",
                                iconBaseName, "RASTER_THUMBNAIL_EXT");
  }


  private static String getDefaultIconBaseName(NamedObj namedObj) {

    String iconBaseName = null;

    if (namedObj instanceof ptolemy.actor.Director) {
      if (isDebugging) {
        log.debug("getDefaultIconBaseName() found a DIRECTOR: "
                  + namedObj.getClassName());
      }
      iconBaseName = DEFAULT_DIRECTOR_ICON_BASENAME;
    } else {
      if (isDebugging) {
        log.debug("getDefaultIconBaseName() found an ACTOR: "
                  + namedObj.getClassName());
      }
      iconBaseName = DEFAULT_ACTOR_ICON_BASENAME;
    }
    return iconBaseName;
  }


  private static void _addIconAttributeToNamedObj(NamedObj namedObj,
                                                  String attribName,
                                                  String iconBasePathKey,
                                                  String iconBaseName,
                                                  String iconExtensionKey) throws
    IOException, NameDuplicationException, IllegalActionException {

    if (isDebugging) {
      log.debug("_addIconAttributeToNamedObj() received: "
                + "\n namedObj:" + namedObj.getClassName()
                + "\n attribName:" + attribName
                + "\n iconBasePathKey:" + iconBasePathKey
                + "\n iconBaseName:" + iconBaseName
                + "\n iconExtensionKey:" + iconExtensionKey);
    }

    String iconBasePath = StaticResources.getUISettingsBundle().getString(iconBasePathKey);
    if (iconBasePath == null) {
      throw new IOException("Could not get icon base path (key = "
                            + iconBasePathKey + ") from file: "
                            + StaticResources.SVG_ICON_MAPPINGS_BYCLASS_BUNDLE);
    }
    String iconExtension = StaticResources.getUISettingsBundle().getString(iconExtensionKey);
    if (iconExtension == null) {
      throw new IOException("Could not get icon extension (key = "
                            + iconExtensionKey + ") from file: "
                            + StaticResources.SVG_ICON_MAPPINGS_BYCLASS_BUNDLE);
    }

    //make sure we haven't already got an attribute with this name; if we
    //have, get it instead of trying to create a new one
    ConfigurableAttribute svgIconAttrib = getExistingOrNewAttribute(namedObj,
                                                                    attribName);

    StringBuffer buff = new StringBuffer(iconBasePath);

    //ensure last char of iconBasePath is a slash:
    if (!iconBasePath.endsWith("/")) {
      buff.append("/");
    }

    buff.append(iconBaseName.trim());

    //ensure iconExtension contains a ".":
    if (iconExtension.indexOf(PERIOD) < 0) {
      buff.append(PERIOD);
    }

    buff.append(iconExtension);
    if (isDebugging) {
      log.debug("_addIconAttributeToNamedObj(): "
                + "\n actor is: " + namedObj.getClassName()
                + "\n attribute name: " + attribName
                + "\n setExpression:" + buff.toString());
    }
    svgIconAttrib.setExpression(buff.toString());

    //set non-persistent so it doesn't get saved in MOML
    svgIconAttrib.setPersistent(false);

    if (isDebugging) {
      log.debug("from actual svgIconAttrib: "
                + "\n svgIconAttrib.getExpression() = "
                + svgIconAttrib.getExpression()
                + "\n svgIconAttrib.getContainer() = "
                + svgIconAttrib.getContainer().getClass().getName());
    }
  }


  private static ConfigurableAttribute getExistingOrNewAttribute(
    NamedObj namedObj, String attribName) {

    ConfigurableAttribute attrib = null;
    try {
      attrib = (ConfigurableAttribute)namedObj.getAttribute(attribName);
    } catch (Exception ex) {
      ex.printStackTrace();
      if (isDebugging) {
        log.warn(namedObj.getClass().getName()
                 + ") : exception getting svgIcon attribute: " + attribName
                 + "; exception was: " + ex.getMessage() + "\n\n");
      }
      attrib = null;
    }
    if (attrib != null) {
      return attrib;
    }

    try {
      // the Attribute is automatically added to namedObj as an attribute,
      // simply by passing namedObj to the constructor
      attrib = new ConfigurableAttribute(namedObj, attribName);
    } catch (Exception ex) {
      ex.printStackTrace();
      if (isDebugging) {
        log.warn(namedObj.getClass().getName()
                 + ") : exception getting svgIcon attribute: " + attribName
                 + "; exception was: " + ex.getMessage() + "\n\n");
      }
      attrib = null;
    }
    return attrib;
  }




  private static ResourceBundle getByClassResBundle() throws IOException {

    if (byClassResBundle == null) {
      byClassResBundle = ResourceBundle.getBundle(
        StaticResources.SVG_ICON_MAPPINGS_BYCLASS_BUNDLE);
    }
    return byClassResBundle;
  }


  private static ResourceBundle getByLSIDResBundle() throws IOException {

    if (byLSIDResBundle == null) {
      byLSIDResBundle = ResourceBundle.getBundle(
        StaticResources.SVG_ICON_MAPPINGS_BYLSID_BUNDLE);
    }
    return byLSIDResBundle;
  }


  ///////////////////////////////////////////////////////////////////
  ////                      private variables                    ////
  ///////////////////////////////////////////////////////////////////

  private static ResourceBundle byClassResBundle;
  private static ResourceBundle byLSIDResBundle;
}
