/*
 Copyright (c) 1998-2004 The Regents of the University of California.
 All rights reserved.
 Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.

 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.

 PT_COPYRIGHT_VERSION_2
 COPYRIGHTENDKEY
 2
 */

package ptolemy.vergil.basic;


import java.io.IOException;
import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import diva.gui.GUIUtilities;
import ptolemy.actor.gui.TableauFrame;
import ptolemy.util.OrderedResourceBundle;
import javax.swing.JComponent;
import diva.gui.toolbox.JContextMenu;
import ptolemy.kernel.util.StaticResources;


/**
 * MenuMapper
 *
 * A menu creation Runnable that creates and adds a new JMenuBar that provides a
 * 'view' (in a loose sense) of the ptii menus. It enumerates all the existing
 * ptii menu items at runtime, and re-uses them, rearranged (and optionally
 * renamed) - all set via mappings in a localizable resourcebundle properties
 * (text) file.
 *
 * If/when new menu items get added to ptii, they are immediately available for
 * use here, just by adding the relevant text mapping to the properties file.
 * (currently configs/ptolemy/configs/kepler/uiMenuMappings_en_US
 *
 * @author  Matthew Brooke
 * @version $Id: MenuMapper.java,v 1.11 2006/04/04 16:38:02 brooke Exp $
 * @since
 * @Pt.ProposedRating
 * @Pt.AcceptedRating
 */
public class MenuMapper implements Runnable {

  //attempt to load menu mapping prefs at startup,
  //so it doesn't need to be done during a pack()
  static {
    try {
      getMenuMappingsResBundle();
    } catch (Exception ex) {
      //no worries - just try again when we actually need it
    }
  }


  ///////////////////////////////////////////////////////////////////
  ////                         public variables                  ////

  public final static String MENUITEM_TYPE = "MENUITEM_TYPE";

  public final static String CHECKBOX_MENUITEM_TYPE = "CHECKBOX_MENUITEM_TYPE";

  public final static String NEW_JMENUITEM_KEY = "NEW_JMENUITEM";


  ///////////////////////////////////////////////////////////////////
  ////                         public methods                    ////


  /**
   * constructor
   * @param ptiiMenubar JMenuBar the existing ptii menubar containing the
   *        original menus
   * @param tableauFrameInstance TableauFrame the frame from which the ptii menu
   *        bar will be hidden, and to which the new menu bar will be added
   */
  public MenuMapper(final JMenuBar ptiiMenubar,
                    final TableauFrame tableauFrameInstance) {
    this._ptiiMenubar = ptiiMenubar;
    this._tableauFrameInstance = tableauFrameInstance;
  }


  public void run() {

    if (_ptiiMenubar == null || _tableauFrameInstance == null) {
      log.error(
        "MenuMapper cannot proceed, due to one or more NULL values:"
        + "\nptiiMenubar = " + _ptiiMenubar
        + "\ntableauFrameInstance = " + _tableauFrameInstance
        + "\ndefaulting to PTII menus");
    }

    //First, we need to make sure PTII has finished constructing its menus.
    //Those menus are created and assembled in a thread that is started in the
    //pack() method of ptolemy.gui.Top. As that thread finishes, it adds the
    //new ptii JMenuBar to the frame - so we test for that here, and don't
    //proceed until the frame has the new JMenuBar added
    //For safety, so we don't have an infinite loop, we also set a safety
    //counter:
    //maxWaitMS is the longest the app waits for the ptii
    //menus to be added, before it continues anyway.
    final int maxWaitMS = 500;
    //sleepTimeMS is the amount of time it sleeps per loop:
    final int sleepTimeMS = 5;
    //////
    final int maxLoops = maxWaitMS / sleepTimeMS;
    int safetyCounter = 0;

    while (safetyCounter++ < maxLoops
           && _tableauFrameInstance.getJMenuBar() != _ptiiMenubar) {
      if (isDebugging) {
        log.debug("Waiting for PTII menus to be created... " + safetyCounter);
      }
      try {
        Thread.sleep(sleepTimeMS);
      } catch (Exception e) {
        //ignore
      }
    }
    JMenuBar _keplerMenubar = null;

    if (_tableauFrameInstance.getJMenuBar() != null) {
      // gets here if a PTII menubar has been added to frame...

      // 1) Now PTII has finished constructing its menus, get all
      // menu items and put them in a Map for easier access later...
      Map ptiiMenuActionsMap = getPTIIMenuActionsMap();

      // 2) Now we have all the PTII menu items, get the
      // Kepler-specific menu mappings from the preferences file,
      // then go thru the Kepler menu mappings and
      // populate the new menubar with Kepler menus,
      // creating any new menu items that don't exist yet

      //this is a Map that will be used to keep track of
      //what we have added to the menus, and in what order
      _keplerMenubar = createKeplerMenuBar(ptiiMenuActionsMap);

      if (_keplerMenubar != null) {

        // First, look to see if any menus are empty. If
        // they are, remove the top-level menu from the menubar...
        // ** NOTE - do these by counting *down* to zero, otherwise the menus'
        // indices change dynamically as we remove menus, causing errors!
        for (int m = _keplerMenubar.getMenuCount() - 1; m >= 0; m--) {
          JMenu nextMenu = _keplerMenubar.getMenu(m);
          if (nextMenu.getMenuComponentCount() < 1) {
            if (isDebugging) {
              log.debug("deleting empty menu: " + nextMenu.getText());
            }
            _keplerMenubar.remove(nextMenu);
          }
        }
        //hide the ptii menubar
        _tableauFrameInstance.hideMenuBar();

        //add the new menu bar
        _tableauFrameInstance.setJMenuBar(_keplerMenubar);
      } else {
        log.error("Problem creating Kepler menus - defaulting to PTII menus");
      }

    } else {
      // gets here if a PTII menubar has *NOT* been added to frame...
      // Therefore, this frame doesn't have a menubar by default,
      // so we probably shouldn't add one, for now, at least

      //hide the ptii menubar (may not be necessary)
      _tableauFrameInstance.hideMenuBar();

      //add the new menu bar (may not be necessary)
      _tableauFrameInstance.setJMenuBar(null);
    }
  }




  public static Action getActionFor(String key, Map menuActionsMap,
                                    TableauFrame tableauFrameInstance) {
    Action action = null;

    if (key.indexOf(MENU_PATH_DELIMITER) > -1) {
      //it's a mapping...
      //NOTE that all keys in ptiiMenuActionsMap are
      //uppercase, to make the ptii value entries in the
      // menu mappings props file case-insensitive
      action = (Action) (menuActionsMap.get(key.toUpperCase()));
    } else {
      //it's a class or a separator

      if (key.toUpperCase().equals(MENU_SEPARATOR_KEY.toUpperCase())) {

        //it's a separator
        return null;

      } else {

        //it's a class - try to instantiate...
        Object actionObj = null;
        try {
          //create the class
          Class classDefinition = Class.forName(key.trim());

          //add the arg types
          Class[] args = new Class[] {TableauFrame.class};

          //create a constructor
          Constructor constructor = classDefinition.getConstructor(args);

          //set the args
          Object[] argImp = new Object[] {tableauFrameInstance};

          //create the object
          actionObj = constructor.newInstance(argImp);
        } catch (Exception e) {
          if (isDebugging) {
            //show a dialog and continue without this menu item
            log.error("Exception trying to create an Action for classname: <"
                      + key + ">:\n" + e.getCause() + " (" + e + ")");
          }
          actionObj = null;
        }
        if (actionObj == null) {
          if (isDebugging) {
            log.error(
            "Problem trying to create an Action for classname: <"
            + key + ">\nPossible reasons:\n"
            + "1) Should be a fully-qualified classname, including "
            + "the package - Check carefully for mistakes.\n"
            + "2) class must implement javax.swing.Action, and must "
            + "have a constructor of the form: \n"
            + "  MyConstructor(ptolemy.actor.gui.TableauFrame)\n"
            + "Returning NULL Action for classname: " + key);
          }
          return null;
        }
        action = (Action)actionObj;
      }
    }
    return action;
  }


    /**
     * Recurse through all the submenu heirarchy beneath the passed JMenu
     * parameter, and for each "leaf node" (ie a menu item that is not a container
     * for other menu items), add the Action and its menu path to the passed Map
     *
     * @param nextMenuItem the JMenu to recurse into
     * @param menuPathBuff a delimited String representation of the hierarchical
     *   "path" to this menu item. This will be used as the key in the
     *   actionsMap. For example, the "Graph Editor" menu item beneath the "New"
     *   item on the "File" menu would have a menuPath of File->New->Graph
     *   Editor. Delimeter is "->" (no quotes), and spaces are allowed within
     *   menu text strings, but not around the delimiters; i.e: New->Graph Editor
     *   is OK, but File ->New is not.
     * @param MENU_PATH_DELIMITER String
     * @param actionsMap the Map containing key => value pairs of the form:
     *   menuPath (as described above) => Action (the javax.swing.Action assigned
     *   to this menu item)
     */
    public static void storePTIIMenuItems(JMenuItem nextMenuItem,
                                          StringBuffer menuPathBuff,
                                          final String MENU_PATH_DELIMITER,
                                          Map actionsMap) {

      menuPathBuff.append(MENU_PATH_DELIMITER);
      menuPathBuff.append(nextMenuItem.getText().toUpperCase());

      log.debug(menuPathBuff.toString());

      if (nextMenuItem instanceof JMenu) {
        storePTIITopLevelMenus((JMenu)nextMenuItem, menuPathBuff.toString(),
                          MENU_PATH_DELIMITER, actionsMap);
      } else {
        Action nextAction = nextMenuItem.getAction();
        //if there is no Action, look for an ActionListener
        if (nextAction == null) {
          final ActionListener[] actionListeners
            = nextMenuItem.getActionListeners();
          log.debug("No Action for " + nextMenuItem.getText() + "; found "
                   + actionListeners.length + "ActionListeners");
          if (actionListeners.length > 0) {

            //ASSUMPTION: there is only one ActionListener
            nextAction = new AbstractAction() {
              public void actionPerformed(ActionEvent a) {
                actionListeners[0].actionPerformed(a);
              }
            };
            //add all these values - @see diva.gui.GUIUtilities
            nextAction.putValue(Action.NAME, nextMenuItem.getText());
            nextAction.putValue(GUIUtilities.LARGE_ICON, nextMenuItem.getIcon());
            nextAction.putValue(GUIUtilities.MNEMONIC_KEY,
                                new Integer(nextMenuItem.getMnemonic()));
            nextAction.putValue("tooltip", nextMenuItem.getToolTipText());
            nextAction.putValue(GUIUtilities.ACCELERATOR_KEY,
                                nextMenuItem.getAccelerator());
            nextAction.putValue("menuItem", nextMenuItem);
          } else {
            if (isDebugging) {
              log.warn("No Action or ActionListener found for "
                       + nextMenuItem.getText());
            }
          }
        }
        if (!actionsMap.containsValue(nextAction)) {
          actionsMap.put(menuPathBuff.toString().toUpperCase(), nextAction);
          if (isDebugging) {
            log.debug(menuPathBuff.toString().toUpperCase()
                      + " :: ACTION: " + nextAction);
          }
        }
      }
  }



  ///////////////////////////////////////////////////////////////////
  ////                         private methods                   ////


  private JMenuBar createKeplerMenuBar(Map ptiiMenuActionsMap) {

    final LinkedHashMap keplerMenuMap = new LinkedHashMap();
    final JMenuBar _keplerMenubar = new JMenuBar();

    log.debug("***************\nKEPLER MENUS:\n***************\n");

    Iterator it = null;
    try {

      if (getMenuMappingsResBundle() != null) {

        it = getMenuMappingsResBundle().getKeys();

      } else {
        log.warn("can't find menu mappings");
        return null;
      }

      if (it == null) {
        log.warn("Menu mappings do not contain any valid assignments");
        return null;
      }

      separatorJustAdded = false;

//      // for any given menu, if there is not a minimum of one real menu item
//      // (i.e. not counting separators) then don't show the menu
//      // - that's what this counter is for:
//      int menuItemCount = 0;
//      String prevTopLevel = null;

      while (it.hasNext()) {

        String nextKey = (String) (it.next());

        if (isDebugging) {
          log.debug("nextKey: " + nextKey);
        }
        if (nextKey == null || nextKey.trim().length() < 1) {
          continue;
        }

//        if (prevTopLevel != null
//            && !nextKey.startsWith(prevTopLevel.replaceAll(MNEMONIC_SYMBOL, ""))) {
//
//          log.debug("nextKey: " + nextKey
//                    +", but prevTopLevel: " + prevTopLevel);
//            //it's a new top-level menu -
//            // check if the previous menu needs deleting...
//            if (menuItemCount < 1) {
//
//          log.debug("menuItemCount < 1 - DELETING: "
//                    + _keplerMenubar.getMenu(_keplerMenubar.getMenuCount() - 1).getText());
//
//                _keplerMenubar.remove(_keplerMenubar.getMenuCount() - 1);
//            }
//
//            //then reset the counter...
//            menuItemCount = 0;
//        }

        String nextVal = getMenuMappingsResBundle().getString(nextKey);
        if (isDebugging) {
          log.debug("nextVal: " + nextVal);
        }

        if (nextVal == null || nextVal.trim().length() < 1) {
          log.warn("no menu mapping found for key: " + nextKey);
          continue;
        }

        Action action = null;

        if (nextKey.indexOf(MENU_SEPARATOR_KEY) < 0) {

          action
            = getActionFor(nextVal, ptiiMenuActionsMap, _tableauFrameInstance);

          if (action == null) {
            if (isDebugging) {
              log.warn("null action for value " + nextVal);
            }
            continue;
          }
          //action exists, and it's not a separator
//          menuItemCount++;
        }

        addMenuFor(nextKey, action, _keplerMenubar, keplerMenuMap);
//        prevTopLevel
//          = _keplerMenubar.getMenu(_keplerMenubar.getMenuCount() - 1).getText();
      }

    } catch (Exception ex) {
      if (isDebugging) {
        log.warn("Exception creating Kepler menus: "
                 + ex + "\nDefaulting to PTII menus");
        if (isDebugging) {
          ex.printStackTrace();
        }
        return _keplerMenubar;
      }
    }

    separatorJustAdded = false;

    log.debug("***************\nEND KEPLER MENUS:\n***************\n");
    return _keplerMenubar;
  }


  public Map getPTIIMenuActionsMap() {

    if (_ptiiMenuActionsMap == null) {
      log.debug("**************\nEXISTING PTII MENUS:\n**************\n");

      _ptiiMenuActionsMap = new HashMap();

      for (int m = 0; m < _ptiiMenubar.getMenuCount(); m++) {
        JMenu nextMenu = _ptiiMenubar.getMenu(m);

        storePTIITopLevelMenus(nextMenu, nextMenu.getText().toUpperCase(),
                               MENU_PATH_DELIMITER, _ptiiMenuActionsMap);
      }
      log.debug("\n**************\nEND PTII MENUS:\n*****************\n");
    }
    return _ptiiMenuActionsMap;
  }


  private static void storePTIITopLevelMenus(JMenu nextMenu, String menuPath,
                                             final String MENU_PATH_DELIMITER,
                                             Map ptiiMenuActionsMap) {

      int totMenuItems = nextMenu.getMenuComponentCount();

      for (int n = 0; n < totMenuItems; n++) {
        Component nextComponent = nextMenu.getMenuComponent(n);
        if (nextComponent instanceof JMenuItem) {
          storePTIIMenuItems((JMenuItem)nextComponent,
                             new StringBuffer(menuPath),
                             MENU_PATH_DELIMITER, ptiiMenuActionsMap);
        }
        // (if it's not an instanceof JMenuItem, it must
        //  be a separator, and can therefore be ignored)
      }
  }


  public static JMenuItem addMenuFor(String key, Action action,
                               JComponent topLvlContainer, Map keplerMenuMap) {

    if (topLvlContainer == null) {
      log.debug("NULL container received (eg JMenuBar) - returning NULL");
      return null;
    }
    if (key == null) {
      log.debug("NULL key received");
      return null;
    }
    key = key.trim();

    if (key.length() < 1) {
      log.debug("BLANK key received");
      return null;
    }
    if (action == null && key.indexOf(MENU_SEPARATOR_KEY) < 0) {
      if (isDebugging) {
        log.debug("NULL action received, but was not a separator: " + key);
      }
      return null;
    }

    if (keplerMenuMap.containsKey(key)) {
      if (isDebugging) {
        log.debug("Menu already added; skipping: " + key);
      }
      return null;
    }

    //split delimited parts and ensure menus all exist
    String[] menuLevel = key.split(MENU_PATH_DELIMITER);

    int totLevels = menuLevel.length;

    // create a menu for each "menuLevel" if it doesn't already exist
    final StringBuffer nextLevelBuff = new StringBuffer();
    String prevLevelStr = null;
    JMenuItem leafMenuItem = null;

    for (int levelIdx = 0; levelIdx < totLevels; levelIdx++) {

      //save previous value
      prevLevelStr = nextLevelBuff.toString();

      String nextLevelStr = menuLevel[levelIdx];
      //get the index of the first MNEMONIC_SYMBOL
      int mnemonicIdx = nextLevelStr.indexOf(MNEMONIC_SYMBOL);
      char mnemonicChar = 0;

      //if an MNEMONIC_SYMBOL exists, remove all underscores. Then, idx of
      //first underscore becomes idx of letter it used to precede - this
      //is the mnemonic letter
      if (mnemonicIdx > -1) {
        nextLevelStr = nextLevelStr.replaceAll(MNEMONIC_SYMBOL, "");
        mnemonicChar = nextLevelStr.charAt(mnemonicIdx);
      }
      if (levelIdx != 0) {
        nextLevelBuff.append(MENU_PATH_DELIMITER);
      }
      nextLevelBuff.append(nextLevelStr);

      //don't add multiple separators together...
      if (nextLevelStr.indexOf(MENU_SEPARATOR_KEY) > -1) {
          if (separatorJustAdded == false) {

            // Check if we're at the top level, since this makes sense only for
            // context menu - we can't add a separator to a JMenuBar
            if (levelIdx == 0) {
              if (topLvlContainer instanceof JContextMenu) {
                ( (JContextMenu)topLvlContainer).addSeparator();
              }
            } else {
              JMenu parent = (JMenu)keplerMenuMap.get(prevLevelStr);

              if (parent != null) {
                if (parent.getMenuComponentCount() < 1) {
                  log.debug("------ NOT adding separator to parent "
                            + parent.getText()
                            + ", since it does not contain any menu items");
                } else {
                  log.debug("------ adding separator to parent "
                            + parent.getText());
                  //add separator to parent
                  parent.addSeparator();
                  separatorJustAdded = true;
                }
              }
            }
          }
      } else if (!keplerMenuMap.containsKey(nextLevelBuff.toString())) {
        //If menu has not already been created, we need
        //to create it and then add it to the parent level...

        JMenuItem menuItem = null;

        //if we're at a "leaf node" - need to create a JMenuItem
        if (levelIdx == totLevels - 1) {

          // save old display name to use as actionCommand on menuitem,
          // since some parts of PTII still
          // use "if (actionCommand.equals("SaveAs")) {..." etc
          String oldDisplayName = (String)action.getValue(Action.NAME);

//                action.putValue(Action.NAME, nextLevelStr);

          if (mnemonicChar > 0) {
            action.putValue(GUIUtilities.MNEMONIC_KEY,
                            new Integer(mnemonicChar));
          }

          //Now we look to see if it's a checkbox
          //menu item, or just a regular one
          String menuItemType = (String) (action.getValue(MENUITEM_TYPE));

          if (menuItemType != null && menuItemType == CHECKBOX_MENUITEM_TYPE) {
            menuItem = new JCheckBoxMenuItem(action);
          } else {
            menuItem = new JMenuItem(action);
          }

          //--------------------------------------------------------------
          /** @todo - setting menu names - TEMPORARY FIX - FIXME */
          // Currently, if we use the "proper" way of setting menu names -
          // ie by using action.putValue(Action.NAME,  "somename");, then
          // the name appears on the port buttons on the toolbar, making
          // them huge. As a temporary stop-gap, I am just setting the new
          // display name using setText() instead of action.putValue(..,
          // but this needs to be fixed elsewhere - we want to be able to
          // use action.putValue(Action.NAME (ie uncomment the line above
          // that reads:
          //       action.putValue(Action.NAME, nextLevelStr);
          // and delete the line below that reads:
          //       menuItem.setText(nextLevelStr);
          // otherwise this may bite us in future...
          menuItem.setText(nextLevelStr);
          //--------------------------------------------------------------

          // set old display name as actionCommand on
          //menuitem, for ptii backward-compatibility
          menuItem.setActionCommand(oldDisplayName);

          //add JMenuItem to the Action, so it can be accessed by Action code
          action.putValue(NEW_JMENUITEM_KEY, menuItem);
          leafMenuItem = menuItem;
        } else {
          //if we're *not* at a "leaf node" - need to create a JMenu
          menuItem = new JMenu(nextLevelStr);
          if (mnemonicChar > 0) {
            menuItem.setMnemonic(mnemonicChar);
          }
        }
        //level 0 is a special case, since the container (JMenuBar or
        //JContextMenu) is not a JMenu or a JMenuItem, so we can't
        //use the same code to add child to parent...
        if (levelIdx == 0) {
            if (topLvlContainer instanceof JMenuBar) {
            //this handles JMenuBar menus
            ( (JMenuBar)topLvlContainer).add(menuItem);
          } else if (topLvlContainer instanceof JContextMenu) {
            //this handles popup context menus
            ( (JContextMenu)topLvlContainer).add(menuItem);
          }
          //add to Map
          keplerMenuMap.put(nextLevelBuff.toString(), menuItem);
          separatorJustAdded = false;
        } else {
          JMenu parent = (JMenu)keplerMenuMap.get(prevLevelStr);
          if (parent != null) {
            //add to parent
            parent.add(menuItem);
            //add to Map
            keplerMenuMap.put(nextLevelBuff.toString(), menuItem);
            separatorJustAdded = false;
          } else {
            if (isDebugging) {
              log.debug("Parent menu is NULL" + prevLevelStr);
            }
          }
        }
      }
    }
    return leafMenuItem;
  }


  private static OrderedResourceBundle getMenuMappingsResBundle() throws
    IOException {

    if (menuMappingsResBundle == null) {
      menuMappingsResBundle
        = OrderedResourceBundle.getBundle(MENU_MAPPINGS_BUNDLE);
    }
    return menuMappingsResBundle;
  }


  ///////////////////////////////////////////////////////////////////
////                         private variables                 ////


  private static final Log log
    = LogFactory.getLog("UI." + MenuMapper.class.getName());

  private static final boolean isDebugging = log.isDebugEnabled();

  private static OrderedResourceBundle menuMappingsResBundle;

  private JMenuBar _ptiiMenubar;

  private TableauFrame _tableauFrameInstance;

  private Map _ptiiMenuActionsMap;

  private static boolean separatorJustAdded = false;


  //NOTE - MUST NOT contain the char defined in MNEMONIC_SYMBOL,
  //or the String defined as MENU_PATH_DELIMITER
  public static final String MENU_PATH_DELIMITER = "->";

  //NOTE - MUST NOT contain the String defined in MNEMONIC_SYMBOL,
  //or the String defined as MENU_PATH_DELIMITER
  public static final String MENU_SEPARATOR_KEY = "MENU_SEPARATOR";

  //NOTE - MUST NOT match the String defined as MENU_PATH_DELIMITER,
  //or the String defined as MENU_SEPARATOR_KEY
  public static final String MNEMONIC_SYMBOL = "~";

  /**
   * Path to resource bundle containing mappings & settings for menus
   */
  private static final String MENU_MAPPINGS_BUNDLE
    = StaticResources.RESOURCEBUNDLE_DIR + "/uiMenuMappings";
}
