/**
 * Image compositor is an image Manipulation actor which can be used to 
 * overlay two images at the specified location.
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package util;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.StringToken;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;

import javax.imageio.*;

import java.io.*;
import java.util.*;
import java.awt.image.BufferedImage;
import java.net.MalformedURLException;
import ptolemy.data.type.BaseType;
import ptolemy.data.ObjectToken;
import ptolemy.data.expr.StringParameter;
import ptolemy.data.expr.Parameter;
import ptolemy.data.BooleanToken;
import ptolemy.data.AWTImageToken;
import ptolemy.vergil.kernel.attributes.ImageAttribute;
import ptolemy.actor.parameters.PortParameter;
import ptolemy.data.*;

import java.awt.*;
import java.awt.image.*;
import java.net.*;
import javax.imageio.*;
import javax.swing.*;
import java.io.File;
import java.lang.Integer;
import javax.imageio.ImageIO;

//For JAI usage
import javax.media.jai.JAI;
import javax.media.jai.PlanarImage;

/**
 * ImageCompositor can be used to overlay two images.
 *
 @author Nandita Mangal
 @version $Id: ImageCompositor.java,v 1.1 2005/11/02 00:21:52 
 @category.name Image Manipulation
 */
public class ImageCompositor extends TypedAtomicActor
{

  //////////////////////////////////////////////////
  ////        Parameters and Ports             ////
    
  /*
   * Upper Left corner X cordinate for overlaying
   */
  public PortParameter positionOverlay_XCoordinate;

  /*
   * Upper Left corner Y cordinate for overlaying
   */
  public PortParameter positionOverlay_YCoordinate;

  /*
   * Source Image to overlay
   */
  public TypedIOPort imgSrc;
  
  /*
   * Destination Image on which source is overlayed
   */
  public TypedIOPort imgDest;
  
  /*
   * Output composited image
   */
  public TypedIOPort outputImage;

  /*
   * Styles of Compositing
   * For more information on the affect of these styles:
   * http://java.sun.com/docs/books/tutorial/2d/display/compositing.html
   */
  public StringParameter compositingStyle;
  
   /*
   * The level of transparency of source image [0-1]
   */
  public StringParameter srcTransparency;
 

  /**
   * Imagecompositor constructor with the given container and name
   *
   *@param  container                     The container.
   *@param  name                          The name of this actor.
   *@exception  IllegalActionException    If the actor cannot be contained
   *   by the proposed container.
   *@exception  NameDuplicationException  If the container already has an
   *   actor with this name.
   */
  public ImageCompositor(CompositeEntity container, String name)
    throws
      NameDuplicationException, IllegalActionException
  {
     super(container, name);

         imgSrc = new TypedIOPort(this, "Source Image", true, false);     imgSrc.setTypeEquals(BaseType.OBJECT);
    imgSrc.setMultiport(false);

    imgDest = new TypedIOPort(this, "Dest Image", true, false);     imgDest.setTypeEquals(BaseType.OBJECT);
    imgDest.setMultiport(false);

    outputImage =new TypedIOPort(this,"Output Image",false,true);
    outputImage.setTypeEquals(BaseType.OBJECT);
    outputImage.setMultiport(false);
         compositingStyle = new StringParameter(this,"Compositing Syle:"); 
    compositingStyle.addChoice("SRC_OVER");
    compositingStyle.addChoice("DST_OVER");
    compositingStyle.addChoice("SRC_OUT");
    compositingStyle.addChoice("DST_OUT");
    compositingStyle.addChoice("SRC_IN");
    compositingStyle.addChoice("DST_IN");
    compositingStyle.addChoice("CLEAR");

    srcTransparency = new StringParameter(this,"Src Transparency Level (0-1):");
     
    positionOverlay_XCoordinate = new PortParameter(this,"Upper Level X Co-ordinate(Source Overlay Position)"); 
    positionOverlay_YCoordinate = new PortParameter(this,"Upper Level Y Co-ordinate(Source Overlay Position)");                                                       
  
  }

  
  /////////////////////////////////////////////////////////////////
  ////		         public methods                       ////


  /** Retrieve the Source and Destination image, after calculating the
   * co-ordinates of overlay draw the source image on destination.
   *
   *@exception  IllegalActionException  If there is no director.
   */
  public void fire()
    throws IllegalActionException
  {
    super.fire();
   
     /* PlanarImages in JAI are not inherited from the Java Image class and hence
    *  are stored in normal ObjectToken and transferred through actor ports. However
    * normal Java AWT images are stored in AWTImageTokens.
    */
    BufferedImage srcBuffered= null;
    BufferedImage destBuffered =null;
    
 
    Object inputToken1 = imgSrc.get(0);
    Object inputToken2 = imgDest.get(0);

    if(inputToken1 instanceof AWTImageToken)  //then we know its an AWT Image (*not PlanarImage)
    {
    
    	Image src = (Image)((AWTImageToken)inputToken1).getValue(); 
        srcBuffered  = toBufferedImage(src);    
           
    }
    else if(inputToken1 instanceof ObjectToken) //then its a PlanarImage probably.	
    {
	if (  ((ObjectToken)inputToken1).getValue() instanceof PlanarImage)
	 {
		PlanarImage pi=(PlanarImage)((ObjectToken)inputToken1).getValue(); 
		srcBuffered = pi.getAsBufferedImage();			  
         } 

    }    

    if(inputToken2 instanceof AWTImageToken)  //then we know its an AWT Image (*not PlanarImage)
    {
    
    	Image dest = (Image)((AWTImageToken)inputToken2).getValue(); 
          destBuffered = toBufferedImage(dest);   

    }
    else if(inputToken2 instanceof ObjectToken) //then its a PlanarImage probably.	
    {
	if (  ((ObjectToken)inputToken2).getValue() instanceof PlanarImage)
	 {
		PlanarImage pi=(PlanarImage)((ObjectToken)inputToken2).getValue(); 
		destBuffered = pi.getAsBufferedImage();			  
         } 

    }    

    positionOverlay_XCoordinate.update();
    positionOverlay_YCoordinate.update();
 
    int position_x=0;
    int position_y=0;

    Token t_x = positionOverlay_XCoordinate.getToken();
    Token t_y = positionOverlay_YCoordinate.getToken();
    if(t_x instanceof IntToken)
          position_x=((IntToken)t_x).intValue();
    if(t_y instanceof IntToken)
          position_y=((IntToken)t_y).intValue();
    

  
    String rule= compositingStyle.stringValue();
    int ruleValue=AlphaComposite.SRC_OVER;
 
    if(rule.equals("SRC_OVER"))
	ruleValue=AlphaComposite.SRC_OVER;
    else if(rule.equals("DST_OVER"))
	ruleValue=AlphaComposite.DST_OVER;
    else if(rule.equals("SRC_OUT"))
	ruleValue=AlphaComposite.SRC_OUT;
    else if(rule.equals("DST_OUT"))
	ruleValue=AlphaComposite.DST_OUT;	
    else if(rule.equals("SRC_IN"))
	ruleValue=AlphaComposite.SRC_IN;
    else if(rule.equals("DST_IN"))
	ruleValue=AlphaComposite.DST_IN;
    else if(rule.equals("CLEAR"))
	ruleValue=AlphaComposite.CLEAR;


    float alphaValueSrc = Float.parseFloat(srcTransparency.stringValue());
      
    Graphics2D graphics = destBuffered.createGraphics();
    graphics.setComposite(AlphaComposite.getInstance(ruleValue, alphaValueSrc));
    graphics.drawImage(srcBuffered,position_x,position_y, null); //387,215
    graphics.dispose();

    outputImage.broadcast(new AWTImageToken(destBuffered));

 
  }

  /**
   * Post fire the actor. Return false to indicate that the
   * process has finished. If it returns true, the process will
   * continue indefinitely.
   *
   *@return
   */
  public boolean postfire()
  {
    return false;
  }

  /**
   * Pre fire the actor.
   *  Calls the super class's prefire in case something is set there.
   *
   *@return
   *@exception  IllegalActionException
   */
  public boolean prefire()
    throws IllegalActionException
  {
    return super.prefire();
  }

   /**
     * Returns a buffered image with the contents of an image
     * @author: The Java Developers Almanac 1.4
     * @return: BufferedImage 
     */

    public static BufferedImage toBufferedImage(Image image) {
        if (image instanceof BufferedImage) {
            return (BufferedImage)image;
        }
	
	
        // This code ensures that all the pixels in the image are loaded
        image = new ImageIcon(image).getImage();
	
        // Determine if the image has transparent pixels; for this method's
        // implementation, see e661 Determining If an Image Has Transparent Pixels
        boolean hasAlpha = hasAlpha(image);
    
        // Create a buffered image with a format that's compatible with the screen
        BufferedImage bimage = null;
        GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
        try {
            // Determine the type of transparency of the new buffered image
            int transparency = Transparency.OPAQUE;

	    if (hasAlpha) {
                transparency = Transparency.BITMASK;
            }
    
            // Create the buffered image
            GraphicsDevice gs = ge.getDefaultScreenDevice();
            GraphicsConfiguration gc = gs.getDefaultConfiguration();
            bimage = gc.createCompatibleImage(
                image.getWidth(null), image.getHeight(null), transparency);
        } catch (HeadlessException e) {
            // The system does not have a screen
        }
    
        if (bimage == null) {
            // Create a buffered image using the default color model
            int type = BufferedImage.TYPE_INT_RGB;
            if (hasAlpha) {
                type = BufferedImage.TYPE_INT_ARGB;
            }
            bimage = new BufferedImage(image.getWidth(null), image.getHeight(null), type);
        }
    
        // Copy image to buffered image
        Graphics g = bimage.createGraphics();
    
        // Paint the image onto the buffered image
        g.drawImage(image, 0, 0, null);
        g.dispose();
    
        return bimage;
    }


    /**
     * This method returns true if the specified image has transparent pixels
     * @author: The Java Developers Almanac 1.4 	
     * @return: boolean if contains transparent pixels or not.
     */

     public static boolean hasAlpha(Image image) {
        // If buffered image, the color model is readily available
        if (image instanceof BufferedImage) {
            BufferedImage bimage = (BufferedImage)image;
            return bimage.getColorModel().hasAlpha();
        }
    
        // Use a pixel grabber to retrieve the image's color model;
        // grabbing a single pixel is usually sufficient
         PixelGrabber pg = new PixelGrabber(image, 0, 0, 1, 1, false);
        try {
            pg.grabPixels();
        } catch (InterruptedException e) {
	    //while(true) {System.err.println("checknow");}
        }
    
        // Get the image's color model
        ColorModel cm = pg.getColorModel();
	
        return cm.hasAlpha();
    }

}
