 /**
 *  '$RCSfile: EcogridQueryTransformer.java,v $'
 *    Purpose: Create and XML document from a QueryType
 *  Copyright: 2000 Regents of the University of California and the
 *             National Center for Ecological Analysis and Synthesis
 *    Authors: Jivka Bojilova
 *    Release: @release@
 *
 *   '$Author: ruland $'
 *     '$Date: 2005-12-15 21:18:45 $'
 * '$Revision: 1.7 $'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.ecoinformatics.ecogrid.queryservice.util;

import org.ecoinformatics.ecogrid.queryservice.query.*;

import org.ecoinformatics.ecogrid.EcogridUtils;

import java.io.*;

/**
 * Produces XML representing the Ecogrid Query
 */
public class EcogridQueryTransformer
{
    private static boolean mIdent = true;

    /**
     * Sets whether the XML will be indented
     * @param aIdent
     */
    public static void setIndent(boolean aIdent)
    {
        mIdent = aIdent;
        
    }
    
    /**
     * Generates indentation
     * @param aDepth the number of two spaces to append
     * @return the string of spaces
     */
    private static String getLeadingSpace(int aDepth)
    {
        if (!mIdent) return "";
        
        StringBuffer spaces = new StringBuffer();
        for (int i=0;i<aDepth;i++) 
        {
            spaces.append("  ");
        }
        return spaces.toString();
    }
    
    /**
     * Herlp method to process all the boolean containers and conidtions
     * @param aStrBuf
     * @param aAnds
     * @param aOrs
     * @param aConds
     * @param aDepth
     */
    private static void process(StringBuffer    aStrBuf, 
                                ANDType[]       aAnds, 
                                ORType[]        aOrs, 
                                ConditionType[] aConds, 
                                int             aDepth)
    {
        if (aAnds != null)
        {
            for (int i = 0; i < aAnds.length; i++)
            {
                processAND(aStrBuf, aAnds[i], aDepth + 1);
            }
        }
        
        if (aOrs != null)
        {
            for (int i=0;i<aOrs.length;i++)
            {
                processOR(aStrBuf, aOrs[i], aDepth+1);
            }
        }
        
        if (aConds != null)
        {
            for (int i = 0; i < aConds.length; i++)
            {
                processCond(aStrBuf, aConds[i], aDepth + 1);
            }
        }
    }
   
    /**
     * Process the AND container
     * @param aStrBuf
     * @param aAnd
     * @param aDepth
     */
    private static void processAND(StringBuffer aStrBuf, ANDType aAnd, int aDepth)
    {
        if (aAnd == null) return;
        aStrBuf.append(getLeadingSpace(aDepth));
        aStrBuf.append("<AND>\n");
        process(aStrBuf, aAnd.getAND(), aAnd.getOR(), aAnd.getCondition(), aDepth);
        aStrBuf.append(getLeadingSpace(aDepth));
        aStrBuf.append("</AND>\n");
    }
   
    /**
     * 
     * @param aStrBuf
     * @param aOR
     * @param aDepth
     */
    private static void processOR(StringBuffer aStrBuf, ORType aOR, int aDepth)
    {
        if (aOR == null) return;
        
        aStrBuf.append(getLeadingSpace(aDepth));
        aStrBuf.append("<OR>\n");
        process(aStrBuf, aOR.getAND(), aOR.getOR(), aOR.getCondition(), aDepth);
        aStrBuf.append(getLeadingSpace(aDepth));
        aStrBuf.append("</OR>\n");
    }
   
    /**
     * Process the OR container
     * @param aStrBuf
     * @param aCond
     * @param aDepth
     */
    private static void processCond(StringBuffer aStrBuf, ConditionType aCond, int aDepth)
    {
        if (aCond == null) return;
        
        aStrBuf.append(getLeadingSpace(aDepth));
        aStrBuf.append("<condition");
        
        OperatorType oper = aCond.getOperator();
        if (oper != null)
        {
            aStrBuf.append(" operator=\"" + oper.getValue() + "\"");
        }
        
        String concept = aCond.getConcept();
        if (concept != null)
        {
            aStrBuf.append(" concept=\"" + concept + "\"");
        }
        aStrBuf.append(">" + aCond + "</condition>\n");
    }
   
   /**
    * Create an XML string that represents the Query object
    * @param aQuery
    * @return
    */
    public static String toXMLString(QueryType aQuery)
    {
        StringBuffer strBuf = new StringBuffer("<query");
        if (aQuery.getQueryId() != null)
        {
            strBuf.append(" queryId=\"" + aQuery.getQueryId() + "\"");
        }
        if (aQuery.getSystem() != null)
        {
            strBuf.append(" system=\"" + aQuery.getSystem().toString() + "\"");
        }
        strBuf.append(">\n");
        
        QueryTypeNamespace[] namespaces = aQuery.getNamespace();
        
        for (int i=0; i < namespaces.length; i++) {
        	QueryTypeNamespace namespace = namespaces[i];
        	if (namespace != null)
	        { 
	            strBuf.append("  <namespace");
	            if (namespace.getPrefix() != null)
	            {
	                strBuf.append(" prefix=\"" + namespace.getPrefix() + "\"");
	            }
	            strBuf.append(">" + namespace.get_value().toString() + "</namespace>\n");
	        }
        }
        
        String[] returnfields = aQuery.getReturnField();
        if (returnfields != null)
        {
            for (int i = 0; i < returnfields.length; i++)
            {
                strBuf.append("  <returnField>" + returnfields[i] + "</returnField>\n");
            }
        }
        
        String[] titles = aQuery.getTitle();
        if (titles != null)
        {
            for (int i = 0; i < titles.length; i++)
            {
                strBuf.append("  <title>" + titles[i] + "</title>\n");
            }
        }
        processAND(strBuf, aQuery.getAND(), 1);
        processOR(strBuf, aQuery.getOR(), 1);
        processCond(strBuf, aQuery.getCondition(), 1);
        strBuf.append("</query>");
        return strBuf.toString();
    }
    
    /**
     * For Testing
     */
    public static void test(String aFileName)
    {
       QueryType          query     = null;
       String             domStr    = EcogridUtils.readXMLFile2Str(aFileName);
       StringReader       strReader = new StringReader(domStr);
       try
       {
           EcogridQueryParser parser    = new EcogridQueryParser(strReader);
           parser.parseXML();
           query = parser.getEcogridQuery();
           System.out.println("File contains:");
           System.out.println(domStr);
           System.out.println("Query XML:");
           System.out.println(EcogridQueryTransformer.toXMLString(query));
           System.out.println("\n");
       } catch(Exception e)
       {
           System.err.println(e);
           e.printStackTrace();
       }
    }
    
    public static void main(String[] args)
    {
        test("/home/globus/workspace/kepler/query.xml");
        test("/home/globus/seek/projects/ecogrid/tests/testfiles/query-digir.xml");
        test("/home/globus/seek/projects/ecogrid/tests/testfiles/query-test.xml");
    }
}
