/**
 *  '$RCSfile: EcogridJSPBeanIFace.java,v $'
 *  Copyright: 2004 Regents of the University of California and the
 *              National Center for Ecological Analysis and Synthesis
 *  Purpose: To test the MetaCatURL class by JUnit
 *    Authors: @Rod Spears@
 *    Release: @release@
 *
 *   '$Author: rspears $'
 *     '$Date: 2004-06-10 13:08:19 $'
 * '$Revision: 1.6 $'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package org.ecoinformatics.ecogrid.registry.impl;

import org.ecoinformatics.ecogrid.ResultsetType;
import org.ecoinformatics.ecogrid.registry.RegEntry;

/**
 * Ecogrid JSP Bean Interface - Helper class to aid in reusability of functionality
 * needed for JSP pages. This encapsulates all the SOAP calls from JSP
 *
 * @author  Rod Spears
 */
public interface EcogridJSPBeanIFace
{

  /**
   * Returns an XML list of all registry entries
   * @param aUrl URL of service to be run
   * @return returns an XML string of all the entries in the registry
   */
  public String getXMLListing(String aUrl);

  /**
   * Returns an HTML table listing of all registry entries
   * @param aUrl URL of service to be run
   * @return returns an HTML table string of all the entries in the registry
   */
  public String getHTMLListing(String aUrl);

  /**
   * Creates an entry in the registry, "name" must be unique
   * @param aUrl URL of service to be run
   * @param aEntry Entry object so its contents can be registered
   * @return status code from EcoRegInterface's error codes
   */
  public int registerEntry(String aUrl, RegEntry aEntry);

  /**
   * Updates an entry in the registry, it is looked up by the parameter "name", so "name"
   * can never be changed
   * @param aUrl URL of service to be run
   * @param aEntry Entry object so its contents can be registered
   * @return status code from EcoRegInterface's error codes
   */
  public int updateEntry(String aUrl, RegEntry aEntry);

  /**
   * Removes an entry in the registry, "name" must be unique
   * @param aUrl URL of service to be run
   * @param aName name of service
   * @return status code from EcoRegInterface's error codes
   */
  public int removeEntry(String aUrl, String aName);

  /**
   * Makes an Ecogrid Query and returns a resultset string in XML
   * @param aUrl URL of service to be run
   * @param aQuery Ecogrid query string
   * @return XML Ecogrid resultset
   */
  public ResultsetType query(String aUrl, String aQuery);

  /**
   * Makes an Ecogrid Query and returns a resultset 
   * @param aUrl URL of service to be run
   * @param aQueryStr Ecogrid query string
   * @return XML Ecogrid resultset as a string
   */
  public String queryWithResultsAsStr(String aUrl, String aQueryStr);

  /**
   * Fills in the "entry" object with the values from the Registry
   * @param aUrl URL of service to be run
   * @param aName name of service
   * @param aEntry RegEntry object to be filled in
   * @return true on success, false on failure
   */
  public boolean getEntry(String aUrl, String aName, RegEntry aEntry);

  /**
   * Converts XML Resultset to an HTML Table
   * @param aResultset ResultsetType object to be converted
   * @return true on success, false on failure
   */
  public String formatResultsetToHTMLTable(ResultsetType aResultset);

  /**
   * Returns the most recent error message as a string
   * @return string of last msg
   */
  public String getErrorMsg();

}
