/**
 *  '$RCSfile: EcogridJSPGlobusBean.java,v $'
 *  Copyright: 2004 Regents of the University of California and the
 *              National Center for Ecological Analysis and Synthesis
 *  Purpose: To test the MetaCatURL class by JUnit
 *    Authors: @Rod Spears@
 *    Release: @release@
 *
 *   '$Author: rspears $'
 *     '$Date: 2004-06-10 13:08:19 $'
 * '$Revision: 1.5 $'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package org.ecoinformatics.ecogrid.registry.impl;

import org.ecoinformatics.ecogrid.EcogridUtils;
import org.ecoinformatics.ecogrid.QueryType;
import org.ecoinformatics.ecogrid.registry.RegEntry;
import org.ecoinformatics.ecogrid.ResultsetType;
import org.ecoinformatics.ecogrid.EcogridResultsetTransformer;

import org.ecoinformatics.ecogrid.EcogridQueryParser;
import org.ecoinformatics.ecogrid.registry.EcoRegConstants;


// Globus
import org.ecoinformatics.ecogrid.registry.EcoRegInterfacePortType;
import org.ecoinformatics.ecogrid.registry.EcoRegInterfaceServiceLocator;

import org.ecoinformatics.ecogrid.stub.service.EcoGridQueryLevelOneServiceGridLocator;
import org.ecoinformatics.ecogrid.stub.EcoGridQueryInterfaceLevelOnePortType;

// For Generic XML Stuff
import java.io.StringReader;

import java.net.URL;

/**
 * Ecogrid JSP Bean - Helper class to aid in reusability of functionality
 * needed for JSP pages. This encapsulates all the SOAP calls from JSP
 *
 * @author  Rod Spears
 */
public class EcogridJSPGlobusBean extends EcogridJSPBaseBean implements EcogridJSPBeanIFace 
{
  private String                  _regUrl   = null;
  private EcoRegInterfacePortType _registry = null;

  private String                                _ecogridUrl = null;
  private EcoGridQueryInterfaceLevelOnePortType _ecogrid    = null;

  private String _errorMsg = "";

  /**
   * Constructor
   */
  public EcogridJSPGlobusBean()
  {
    EcogridUtils.debugMessage("Created EcogridJSPGlobusBean. ****");
  }

  /**
   * Returns the most recent error message as a string
   * @return string of last msg
   */
  public String getErrorMsg()
  {
    return _errorMsg;
  }

  /**
   * Display an error message using EcogridUtils and retains for later
   * @param aMsg error message to be written out and kept
   */
  public void displayErrorMsg(String aMsg)
  {
    _errorMsg = aMsg;
    EcogridUtils.debugMessage(aMsg);
  }

  /**
   * ConnectRegistry to Globus
   * @param aUrl URL of service to be run
   * @return returns true if connected, false if not
   */
  protected boolean ConnectRegistry(String aURL)
  {
    if (_regUrl == null || _registry == null || !_regUrl.equals(aURL)) {
      _regUrl = aURL;
      displayErrorMsg("Connecting to Registry with URL["+aURL+"]");
      try
      {
        // Get command-line arguments

        System.out.println("Connecting to Registry service["+aURL+"]");

        URL GSH = new java.net.URL(aURL);
        System.out.println("Registry URL["+GSH.toString()+"]");

        // Get a reference to the remote web service
        EcoRegInterfaceServiceLocator registryService = new EcoRegInterfaceServiceLocator();

        _registry = registryService.getEcoRegInterfaceService(GSH);
        return _registry != null;
      } 
      catch(Exception e)
      {
        displayErrorMsg("ERROR - Connecting to Registry Service " + e.toString());
        e.printStackTrace();
      }
    } else { // use the same connection
      return true;
    }
    return false;
  }

  /**
   * ConnectEcogrid to Globus
   * @param aUrl URL of service to be run
   * @return returns true if connected, false if not
   */
  protected boolean ConnectEcogrid(String aURL)
  {
    if (_ecogridUrl == null || _ecogrid == null || !_ecogridUrl.equals(aURL)) {
      _ecogridUrl = aURL;
      EcogridUtils.debugMessage("\nConnecting to Ecogrid with URL["+aURL+"]");
      try
      {
        // Get command-line arguments

        EcogridUtils.debugMessage("Connecting to Ecogrid service["+aURL+"]");

        URL GSH = new java.net.URL(aURL);
        System.out.println("Ecogrid URL["+GSH.toString()+"]");

        // Get a reference to the remote web service
        EcoGridQueryLevelOneServiceGridLocator locator = new EcoGridQueryLevelOneServiceGridLocator();
        _ecogrid = locator.getEcoGridQueryLevelOneServicePort(GSH);

        return _ecogrid != null;
      } 
      catch(Exception e)
      {
        displayErrorMsg("ERROR - Connecting to Ecogrid Service " + e.toString());
        e.printStackTrace();
      }
    } else { // use the same connection
      return true;
    }
    return false;
  }

  /**
   * Returns an XML list of all registry entries
   * @param aUrl URL of service to be run
   * @return returns an XML string of all the entries in the registry
   */
  public String getXMLListing(String aUrl)
  {
    if (aUrl == null) return null;

    if (!ConnectRegistry(aUrl)) return null;

    String retVal = null;
    try {
      System.out.println("Before Call to _registry.getXMLListing()");
      retVal = _registry.getXMLListing();
      System.out.println("After Call to _registry.getXMLListing ["+retVal+"]");
    } catch(Exception e) {
      displayErrorMsg("Error in Call to _registry.getXMLListing " + e.toString());
      e.printStackTrace();
    }
    return retVal;
  }


  /**
   * Creates an entry in the registry, "name" must be unique
   * @param aUrl URL of service to be run
   * @param aEntry Entry object so its contents can be registered
   * @return status code from EcoRegInterface's error codes
   */
  public int registerEntry(String aUrl, RegEntry aEntry)
  {
    if (aUrl == null ||aEntry == null ||
        aEntry.getName() == null || aEntry.getRegName() == null || 
        aEntry.getRegAddr() == null) return EcoRegConstants.kParamError;

    if (!ConnectRegistry(aUrl)) return EcoRegConstants.kServiceNotFound;

    EcogridUtils.debugMessage("\n---- EcogridJSPBean registerEntry ----");
    EcogridUtils.debugMessage("Name:    ["+aEntry.getName()+"]");
    EcogridUtils.debugMessage("RegName: ["+aEntry.getRegName()+"]");
    EcogridUtils.debugMessage("RegAddr: ["+aEntry.getRegAddr()+"]");
    EcogridUtils.debugMessage("---- EcogridJSPBean ----");

    int status = EcoRegConstants.kError;
    try {
      status = _registry.registerEntry(aEntry.getName(), aEntry.getRegName(), aEntry.getRegAddr());

    } catch(Exception e) {
      displayErrorMsg("Error in Call to _registry.registerEntry " + e.toString());
      e.printStackTrace();
    }
    return status;
  }


  /**
   * Updates an entry in the registry, it is looked up by the parameter "name", so "name"
   * can never be changed
   * @param aUrl URL of service to be run
   * @param aEntry Entry object so its contents can be registered
   * @return status code from EcoRegInterface's error codes
   */
  public int updateEntry(String aUrl, RegEntry aEntry)
  {
    if (aUrl == null || aEntry == null ||
        aEntry.getName() == null || aEntry.getRegName() == null || 
        aEntry.getRegAddr() == null) return EcoRegConstants.kParamError;

    if (!ConnectRegistry(aUrl)) return EcoRegConstants.kServiceNotFound;

    EcogridUtils.debugMessage("\n---- EcogridJSPBean updateEntry ----");
    EcogridUtils.debugMessage("Name:    ["+aEntry.getName()+"]");
    EcogridUtils.debugMessage("RegName: ["+aEntry.getRegName()+"]");
    EcogridUtils.debugMessage("RegAddr: ["+aEntry.getRegAddr()+"]");
    EcogridUtils.debugMessage("---- EcogridJSPBean ----");

    int status = EcoRegConstants.kError;
    try {
      status = _registry.updateEntry(aEntry.getName(), aEntry.getRegName(), aEntry.getRegAddr());
    } catch(Exception e) {
      displayErrorMsg("Error in Call to _registry.updateEntry " + e.toString());
      e.printStackTrace();
    }
    return status;
  }

  /**
   * Removes an entry in the registry, "name" must be unique
   * @param aUrl URL of service to be run
   * @param aName name of service
   * @return status code from EcoRegInterface's error codes
   */
  public int removeEntry(String aUrl, String aName)
  {
    if (aUrl == null || aName == null) return EcoRegConstants.kParamError;

    if (!ConnectRegistry(aUrl)) return EcoRegConstants.kServiceNotFound;

    EcogridUtils.debugMessage("\n---- EcogridJSPBean removeEntry ----");
    EcogridUtils.debugMessage("aName:    ["+aName+"]");
    EcogridUtils.debugMessage("---- EcogridJSPBean ----");

    int status = EcoRegConstants.kError;
    try {
      status = _registry.removeEntry(aName);
    } catch(Exception e) {
      displayErrorMsg("Error in Call to _registry.removeEntry " + e.toString());
      e.printStackTrace();
    }
    return status;
  }

  /**
   * Makes an Ecogrid Query and returns a resultset 
   * @param aUrl URL of service to be run
   * @param aQueryStr Ecogrid query string
   * @return XML Ecogrid resultset as a string
   */
  public String queryWithResultsAsStr(String aUrl, String aQueryStr)
  {
    ResultsetType resultSet = query(aUrl, aQueryStr);
    if (resultSet != null) {
      return EcogridResultsetTransformer.toXMLString(resultSet);
    }
    return null;
  }

  /**
   * Makes an Ecogrid Query and returns a resultset string in XML
   * @param aUrl URL of service to be run
   * @param aQueryStr Ecogrid query string
   * @return XML Ecogrid resultset
   */
  public ResultsetType query(String aUrl, String aQueryStr)
  {
    if (aUrl == null || aQueryStr == null) return null;

    if (!ConnectEcogrid(aUrl)) return null;

    EcogridUtils.debugMessage("query - [:   "+aQueryStr + "]");

    try {
      StringReader strReader = new StringReader(aQueryStr);
      EcogridQueryParser parser = new EcogridQueryParser(strReader);

      // get ecogrid query java object
      parser.parseXML();
      QueryType query = parser.getEcogridQuery();
      if (query != null) {
        try {
          ResultsetType resultSet = _ecogrid.query(query);

          if (resultSet != null) {
            return resultSet;

          } else {
            displayErrorMsg("Query Resultset was NULL!");
            return null;
          }
        }
        catch (Exception e) {
          displayErrorMsg("Error in Call to query " + e.toString());
          e.printStackTrace();
        }

      } else {
        displayErrorMsg("Query was NULL!");
        return null;
      }
    } 
    catch (Exception e) {
      displayErrorMsg("Error in parse of query " + e.toString());
      e.printStackTrace();
    }

    return null;
  }

  /**
   * Fills in the "entry" object with the values from the Registry
   * @param aUrl URL of service to be run
   * @param aName name of service
   * @param aEntry RegEntry object to be filled in
   * @return true on success, false on failure
   */
  public boolean getEntry(String aUrl, String aName, RegEntry aEntry)
  {
    if (aUrl == null || aName == null || aEntry == null) return false;

    if (!ConnectRegistry(aUrl)) return false;

    EcogridUtils.debugMessage("Name:   "+aName);
    EcogridUtils.debugMessage("aEntry: "+aEntry);

    RegEntry entry = null;
    try {
      entry = _registry.getEntry(aName);

      if (entry != null) {
        aEntry.setName(entry.getName());
        aEntry.setRegName(entry.getRegName());
        aEntry.setRegAddr(entry.getRegAddr());
        return true;
      }
    } catch (Exception e) {
      displayErrorMsg("Error in Call to _registry.getEntry " + e.toString());
      e.printStackTrace();
    }
    displayErrorMsg("Error in Call to _registry.getEntry (could find entry ["+aName+"])");
    return false;
  }

}
