/**
 *    Purpose: A Class that implements registry service.
 *  Copyright: 2004 Regents of the University of California and
 *             San Diego Supercomputer Center
 *    Authors: Bing Zhu 
 *             Sept 14, 2004
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.ecoinformatics.ecogrid.registry.impl;


import java.io.StringReader;
import java.io.InputStream;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.Properties;

import org.apache.log4j.Logger;

import org.ecoinformatics.ecogrid.queryservice.stub.QueryServicePortType;
import org.ecoinformatics.ecogrid.queryservice.QueryServiceClient;
import org.ecoinformatics.ecogrid.queryservice.query.ANDType;
import org.ecoinformatics.ecogrid.queryservice.query.ConditionType;
import org.ecoinformatics.ecogrid.queryservice.query.OperatorType;
import org.ecoinformatics.ecogrid.queryservice.query.ORType;
import org.ecoinformatics.ecogrid.queryservice.query.QueryType;
import org.ecoinformatics.ecogrid.queryservice.query.QueryTypeNamespace;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeResultsetMetadataNamespace;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeRecord;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetType;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeResultsetMetadata;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeRecordReturnField;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeResultsetMetadataSystem;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeResultsetMetadataRecordStructure;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeResultsetMetadataRecordStructureReturnField;
import org.ecoinformatics.ecogrid.queryservice.util.EcogridResultsetFactory;
import org.ecoinformatics.ecogrid.EcogridUtils;
import org.ecoinformatics.ecogrid.registry.stub.AddRequestElement;
import org.ecoinformatics.ecogrid.registry.stub.ListRequestElement;
import org.ecoinformatics.ecogrid.registry.stub.ListResponseElement;
import org.ecoinformatics.ecogrid.registry.stub.QueryRequestElement;
import org.ecoinformatics.ecogrid.registry.stub.QueryResponseElement;
import org.ecoinformatics.ecogrid.registry.stub.RegistryEntryArrayType;
import org.ecoinformatics.ecogrid.registry.stub.RegistryEntryType;
import org.ecoinformatics.ecogrid.registry.stub.RegistryEntryTypeDocumentType;
import org.ecoinformatics.ecogrid.registry.stub.RegistryServicePortType;
import org.ecoinformatics.ecogrid.registry.stub.RegistryServiceLocator;
import org.ecoinformatics.ecogrid.registry.stub.RemoveRequestElement;
import org.ecoinformatics.ecogrid.registry.stub.UpdateRequestElement;
import org.ecoinformatics.ecogrid.registry.util.RegistryEntryTypeTransformer;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import edu.ucsb.nceas.metacat.client.MetacatClient;
import edu.ucsb.nceas.metacat.client.MetacatFactory;


/**
 * The class implements registry service
 */
public class RegistryServiceImpl implements RegistryServicePortType
{
   // Default value can be overridden by the value of element 'metacatURL' in configuration file. 
   //String metacatURL = "http://pine.nceas.ucsb.edu:8090/ogsa/metacat";
   //String metacatQueryServiceGSH = "http://pine.nceas.ucsb.edu:8090/ogsa/services/org/ecoinformatics/ecogrid/QueryService";

   private static String metacatURL = "http://knb.ecoinformatics.org/knb/metacat";
   private static String metacatQueryServiceGSH = "http://ecogrid.ecoinformatics.org/ogsa/services/org/ecoinformatics/ecogrid/QueryService";
   private static final String METACATURL = "metacatURL";
   private static final String ECOGRIDURL = "ecogridQueryURL";
   private static Logger logMetacat = Logger.getLogger(RegistryServiceImpl.class);
   static final int NumPredefinedFields = 10;
   private static String[] PredefinedFieldIds = new String[NumPredefinedFields];
   private static String[] PredefinedFieldNames = {
	   								"RegistryEntry/serviceName",   
                                    "RegistryEntry/serviceGroup",     
                                    "RegistryEntry/serviceClassification",                  
                                    "RegistryEntry/wsdlURL",                      
                                    "RegistryEntry/serviceType",   
                                    "RegistryEntry/endPoint",                     
                                    "RegistryEntry/documentType/namespace",       
                                    "RegistryEntry/documentType/label",
                                    "RegistryEntry/description",
                                    "RegistryEntry/provider"
                                    };                          

   static
   {
      
      try
      {
          
          InputStream is = RegistryServiceImpl.class.getResourceAsStream("/metacat.properties");
          Properties props = new Properties();
          props.load(is);
          String tmp = props.getProperty(METACATURL);
          if (tmp != null && tmp.length() >0)
          {
              metacatURL = tmp;
          }
          
          String tmp1 = props.getProperty(ECOGRIDURL);
          if (tmp != null && tmp.length() >0)
          {
              metacatQueryServiceGSH = tmp1;
          }
      }
      catch (Exception e)
      {
          e.printStackTrace(System.err);
      }
   }

   

   public RegistryServiceImpl()
   {
      
   }

 

   public  String add(AddRequestElement parameter) throws RemoteException
   {
      //GetEcogridRegistryConfig();

      // build the xml document and store it in Metacat
      /*String xml_doc_string = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
      xml_doc_string += "<reg:RegistryEntry xmlns:reg=\"http://ecoinformatics.org/registryentry-1.0.0\">\n";
      xml_doc_string += "  <serviceName>" + regEntry.getServiceName() + "</serviceName>\n";
      xml_doc_string += "  <wsdlURL>" + regEntry.getWsdlURL() + "</wsdlURL>\n";
      xml_doc_string += "  <serviceType>" + regEntry.getServiceType() + "</serviceType>\n";
      xml_doc_string += "  <endPoint>" + regEntry.getEndPoint() + "</endPoint>\n";
      xml_doc_string += "  <serviceClassification>EcogridServiceRegistry</serviceClassification>\n";
      xml_doc_string += "  <description>" + regEntry.getServiceClassification() + "</description>\n";
      int i,j;
      RegistryEntryType_documentType[] docTypes = regEntry.getDocumentType();
      if(docTypes != null)
      {
         for(i=0;i<docTypes.length;i++)
         {
            xml_doc_string += "  <documentType>\n";
            xml_doc_string += "     <namespace>" + docTypes[i].getNamespace() + "</namespace>\n";
            if(docTypes[i].getLabel() != null)
              xml_doc_string += "     <label>" + docTypes[i].getLabel() + "</label>\n"; 
            else
              xml_doc_string += "     <label></label>\n";
            xml_doc_string += "  </documentType>\n";
         }
      }

      String[] providers = regEntry.getProvider();
      if(providers != null)
      {
         for(i=0;i<providers.length;i++)
         {
            xml_doc_string += "  <provider>" + providers[i] + "</provider>\n";
         }
      }
      xml_doc_string += "</reg:RegistryEntry>";

      EcogridUtils.debugMessage(xml_doc_string);*/
      String docid = null; //generateDocId();
      try
      {
         String sessionId = parameter.getSessionId();
         RegistryEntryType entry = parameter.getRegistryEntry();
         //check if there is a given docid in the object, or generate one if not
         docid = entry.getId();
         if (docid == null || docid.length() == 0) {
        	 docid = generateDocId();
        	 entry.setId(docid);
         }
         String xml_doc_string = RegistryEntryTypeTransformer.toXMLString(entry);
         
         MetacatClient mc = (MetacatClient)MetacatFactory.createMetacatConnection(metacatURL);
         mc.setSessionId(sessionId);
         String response = mc.insert(docid, new StringReader(xml_doc_string), null);

         // set the permission of the document to public readable
         response = mc.setAccess(docid, "public", "read", "allow", "denyFirst");
      }
      catch(Exception e)
      {
         logMetacat.error("addRegEntry() exception: " + e.getMessage());
         throw new RemoteException(e.getMessage());
      } 
      return docid;
   }
   
   /**
    * Update the registry entry
    * @param parameter
    * @return
    * @throws RemoteException
    */
   public String update(UpdateRequestElement parameter) throws RemoteException
   {
       String response = null;
       try
       {
          String docid = parameter.getDocid();
          String sessionId = parameter.getSessionId();
          RegistryEntryType entry = parameter.getRegistryEntry();
          //use the id from the parameter to make sure they match
          entry.setId(docid);
          String xml_doc_string = RegistryEntryTypeTransformer.toXMLString(entry);
          
          MetacatClient mc = (MetacatClient)MetacatFactory.createMetacatConnection(metacatURL);
          mc.setSessionId(sessionId);
          response = mc.update(docid, new StringReader(xml_doc_string), null);

          // set the permission of the document to public readable
          mc.setAccess(docid, "public", "read", "allow", "denyFirst");
       }
       catch(Exception e)
       {
          logMetacat.error("updateRegEntry() exception: " + e.getMessage());
          throw new RemoteException(e.getMessage());
       } 
       return response;
   }

   public String remove(RemoveRequestElement parameter) throws RemoteException
   {
      
      String response = null;
      try
      {
         String docid = parameter.getDocid();
         String sessionId = parameter.getSessionId();
         MetacatClient mc = (MetacatClient)MetacatFactory.createMetacatConnection(metacatURL);
         mc.setSessionId(sessionId);
         response = mc.delete(docid);
      }
      catch(Exception e)
      {
         logMetacat.error("removeRegEntry() exception: " + e.getMessage());
         throw new RemoteException(e.getMessage());
      }
      logMetacat.info(response);
      return response;
   }

   
   public ListResponseElement list(ListRequestElement rqstElement) throws RemoteException
   {
      

      String query_doc_string = getPartEcogridQueryDoc();
      query_doc_string += "    </AND>\n";
      query_doc_string += "</egq:query>";
      logMetacat.info(query_doc_string);
      // submit the 'query_doc_string' to ecogrid's query service
      RegistryEntryArrayType regEnteryArray = null;

      try
      {
         regEnteryArray = submitQueryDoc(query_doc_string);
      }
      catch(Exception e)
      {
         EcogridUtils.debugMessage(e.getMessage());
         e.printStackTrace();
         throw new RemoteException(e.getMessage());
      }



      ListResponseElement getAllServiceResponse = null;

      if(regEnteryArray != null)
      {
         getAllServiceResponse = new ListResponseElement(regEnteryArray);
        
      }
      else
      {
         logMetacat.warn("get null return after calling Metacat query service.");
      }

      return getAllServiceResponse;
   }

   public QueryResponseElement query(QueryRequestElement parameter) throws RemoteException
   {
      String sessionId = parameter.getSessionId();
      String queryField = parameter.getQueryField();
      String queryString = parameter.getQueryString();
      String QueryField = "";
      String QueryOp = "";
      String RivisedQueryStr = "";

      if(queryField.compareToIgnoreCase("serviceName") == 0)
      {
         QueryField = "RegistryEntry/serviceName";
      }
      else if(queryField.compareToIgnoreCase("serviceGroup") == 0)
      {
         QueryField = "RegistryEntry/serviceGroup";
      }
      else if(queryField.compareToIgnoreCase("wsdlURL") == 0)
      {
         QueryField = "RegistryEntry/wsdlURL";
      }
      else if(queryField.compareToIgnoreCase("serviceType") == 0)
      {
         QueryField = "RegistryEntry/serviceType";
      }
      else if(queryField.compareToIgnoreCase("endPoint") == 0)
      {
         QueryField = "RegistryEntry/endPoint";
      }
      else if(queryField.compareToIgnoreCase("serviceClassification") == 0)
      {
         QueryField = "RegistryEntry/serviceClassification";
      }
      else if (queryField.compareToIgnoreCase("namespace") == 0)
      {
          QueryField = "RegistryEntry/documentType/namespace";
      }
      else
      {
          throw new RemoteException("Unknow search field: " + queryField);
      }

      // check if there is a wild card such as '%' or '*'
      if((queryString.indexOf('%') >=0 ) || (queryString.indexOf('*') >= 0))
      {
          QueryOp = "LIKE";
          // replace all '*' with '%';
          RivisedQueryStr = queryString.replace('*','%');
      }
      else
      {
          QueryOp = "EQUALS";
          RivisedQueryStr = queryString;
      }


      String query_doc_string = getPartEcogridQueryDoc();
      query_doc_string += "       <condition operator=\"" + QueryOp + "\" concept=\""+ QueryField + "\">" + RivisedQueryStr + "</condition>\n";

      query_doc_string += "    </AND>\n";
      query_doc_string += "</egq:query>";

      logMetacat.info(query_doc_string);
      // submit the 'query_doc_string' to ecogrid's query service
      RegistryEntryArrayType  regEnteryArray = null;
      try
      {
         regEnteryArray = submitQueryDoc(query_doc_string);
      }
      catch(Exception e)
      {
         throw new RemoteException(e.getMessage());
      }
      QueryResponseElement getServiceResponse = null;

      if(regEnteryArray != null)
      {
         getServiceResponse = new QueryResponseElement(regEnteryArray);
        
      }
      else
      {
         logMetacat.warn("get null return after calling Metacat query service.");
      }

      return getServiceResponse;
   }
   
   
   private String getPartEcogridQueryDoc()
   {
      // build the query document for Metacat system
      String query_doc_string = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
      query_doc_string += "<egq:query queryId=\"query_reg.1.1\" system=\"http://pine.nceas.ucsb.edu:8090/ogsa/metacat\"\n";
      query_doc_string += "  xmlns:egq=\"http://ecoinformatics.org/query-1.0.1\"\n";
      query_doc_string += "  xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n";
      query_doc_string += "  xsi:schemaLocation=\"http://ecoinformatics.org/query-1.0.1 ../../src/xsd/query.xsd\">\n";
      query_doc_string += "    <namespace prefix=\"reg\">http://ecoinformatics.org/registryentry-1.0.0</namespace>\n";

      // return fields
      query_doc_string += "    <returnField>RegistryEntry/serviceName</returnField>\n";
      query_doc_string += "    <returnField>RegistryEntry/serviceGroup</returnField>\n";
      query_doc_string += "    <returnField>RegistryEntry/wsdlURL</returnField>\n";
      query_doc_string += "    <returnField>RegistryEntry/serviceType</returnField>\n";
      query_doc_string += "    <returnField>RegistryEntry/endPoint</returnField>\n";
      query_doc_string += "    <returnField>RegistryEntry/description</returnField>\n";
      query_doc_string += "    <returnField>RegistryEntry/serviceClassification</returnField>\n";
      query_doc_string += "    <returnField>RegistryEntry/documentType/namespace</returnField>\n";
      query_doc_string += "    <returnField>RegistryEntry/documentType/label</returnField>\n";
      query_doc_string += "    <returnField>RegistryEntry/provider</returnField>\n";
      

      query_doc_string += "    <title>query ecogrid registries</title>\n";

      query_doc_string += "    <AND>\n";

      // magic part
      query_doc_string += "       <condition operator=\"EQUALS\" concept=\"RegistryEntry/serviceClassification\">EcogridServiceRegistry</condition>\n";

      return query_doc_string;
   }

   private RegistryEntryArrayType submitQueryDoc(String query_doc) throws Exception
   {
      int i,j;
      String tmpstr;
      logMetacat.info("++++++++start of submitQueryDoc");
      QueryServiceClient client = new QueryServiceClient(metacatQueryServiceGSH);
      StringReader queryReader = new StringReader(query_doc);
      ResultsetType queryResults = client.query(queryReader);

      if(queryResults == null)
      {
        logMetacat.warn("Result set from ecogrid is null");
        return null; 
      }
      
      ResultsetTypeRecord[] queryRecords = queryResults.getRecord();

      if((queryRecords == null) || (queryRecords.length <= 0))
      {
          logMetacat.warn("Result records from ecogrid is null or is empty");
          return null;
      }

      logMetacat.info("++++++++start to handle resultset");
      // get prefinedFields IDs.
      // ResultsetType_resultsetMetadata_recordStructure[] rsMetaStruct = queryResults.getResultsetMetadata().getRecordStructure();
      //rsMetaStruct[0].getReturnField();

      // Here rsMetaStruct.length should be 1
      ResultsetTypeResultsetMetadata metadata = queryResults.getResultsetMetadata();
      //logMetacat.info("after get metadata");
      ResultsetTypeResultsetMetadataRecordStructure structure = metadata.getRecordStructure();
      //System.out.println("get sturcture array");
      //System.out.println("get the structure");
      ResultsetTypeResultsetMetadataRecordStructureReturnField[] metaReturnFields = 
               structure.getReturnField();
      //System.out.println("++++++++after getting resultset metadata");
      int idx;
      for(i=0;i<metaReturnFields.length;i++)
      {
          tmpstr = metaReturnFields[i].getName();

          // get index
          idx = GetIdxByPredefinedFieldName(tmpstr);
          if(idx >= 0)
          {
            PredefinedFieldIds[idx] = metaReturnFields[i].getId();

            logMetacat.info(""+tmpstr + " ==> idx=" + idx);
          }
      }

      RegistryEntryType[] regEntries = new RegistryEntryType[queryRecords.length];
      RegistryEntryType regEntry;
      ResultsetTypeRecord queryRecord;


      String[] nameSpaces = null;
      String[] labels = null;
      int namSpacesCnt, labelsCnt;
      String[] providers = null;
      int providerCnt;
      
      // EcogridUtils.debugMessage("after calling query service -- get record counts: " + queryRecords.length);

      // loop results and create entries
      for(i=0;i<queryRecords.length;i++)
      {
         regEntry = new RegistryEntryType();
         queryRecord = queryRecords[i];  
     
         ResultsetTypeRecordReturnField[] queryReturnFields = queryRecord.getReturnField();

         if((queryReturnFields == null) || (queryReturnFields.length <= 0))
         {
            regEntries[i] = null;
            break;
         }

         logMetacat.info("+++++++++queryReturnFields.length=" + queryReturnFields.length);
         nameSpaces = new String[queryReturnFields.length];
         labels = new String[queryReturnFields.length];
         providers = new String[queryReturnFields.length];

         namSpacesCnt = 0;
         labelsCnt = 0;
         providerCnt = 0;
         
         //get the docid
         logMetacat.info("++++++++ id " + queryRecord.getIdentifier());
         regEntry.setId(queryRecord.getIdentifier()); 

         for(j=0;j<queryReturnFields.length;j++)
         {
            idx = GetIdxByPredefinedFieldId(queryReturnFields[j].getId());
            // EcogridUtils.debugMessage("getName=" + queryReturnFields[j].getName() + ", getContent=" + queryReturnFields[j].getContent() + ", getId=" + queryReturnFields[j].getId() + ", idx=" + idx);
            switch(idx)
            {
               case 0:
                 logMetacat.info("++++++++ service name "+queryReturnFields[j].get_value());
                 regEntry.setServiceName(queryReturnFields[j].get_value());
                 break;
               case 1:
                 logMetacat.info("++++++++ serviceGroup "+queryReturnFields[j].get_value());
                 regEntry.setServiceGroup(queryReturnFields[j].get_value()); 
                 break;  
               case 2:
                 logMetacat.info("++++++++ service classification "+queryReturnFields[j].get_value());
                 regEntry.setServiceClassification(queryReturnFields[j].get_value());
                 break;
               case 3:
                 logMetacat.info("++++++++ wsdlurl "+queryReturnFields[j].get_value());
                 regEntry.setWsdlURL(queryReturnFields[j].get_value()); 
                 break;
               case 4:
                 logMetacat.info("++++++++ service type "+queryReturnFields[j].get_value());
                 regEntry.setServiceType(queryReturnFields[j].get_value()); 
                 break;                                
               case 5:
                 logMetacat.info("++++++++ endpoint "+queryReturnFields[j].get_value());
                 regEntry.setEndPoint(queryReturnFields[j].get_value());
                 break;
               case 6:
                 logMetacat.info("++++++++ namespace "+queryReturnFields[j].get_value().toString());
                 nameSpaces[namSpacesCnt++] = queryReturnFields[j].get_value().toString();
                 break;
               case 7:
                 logMetacat.info("++++++++ label "+queryReturnFields[j].get_value().toString());
                 labels[labelsCnt++] = queryReturnFields[j].get_value().toString();
                 break;
               case 8:
                   logMetacat.info("++++++++ description "+queryReturnFields[j].get_value());
                   regEntry.setDescription(queryReturnFields[j].get_value()); 
                   break;
               case 9:
                 logMetacat.info("++++++++ provider "+queryReturnFields[j].get_value().toString());
                 providers[providerCnt++] = queryReturnFields[j].get_value().toString();
                 break;  
            }
         }

         //EcogridUtils.debugMessage("namSpacesCnt=" + namSpacesCnt);
         //EcogridUtils.debugMessage("labelsCnt=" + labelsCnt);
         //EcogridUtils.debugMessage("providerCnt=" + providerCnt);
         if(namSpacesCnt > 0)
         {
            logMetacat.info("++++++++ setup doc type");
            RegistryEntryTypeDocumentType[] doctypes = new RegistryEntryTypeDocumentType[namSpacesCnt];
            for(j=0;j<namSpacesCnt;j++)
            {
               doctypes[j] = new RegistryEntryTypeDocumentType();
               doctypes[j].setNamespace(nameSpaces[j]);
               if(labels[j] != null)
                 doctypes[j].setLabel(labels[j]);
            }

            regEntry.setDocumentType(doctypes);
         }

         if(providerCnt > 0)
         {
            logMetacat.info("++++++++ namespace setup provider");
            String[] realSizedProviders = new String[providerCnt];
            for(j=0;j<providerCnt;j++)
              realSizedProviders[j] = providers[j];

            regEntry.setProvider(realSizedProviders);
         }

         regEntries[i] = regEntry;
      }

      // finally return the arrayElement
      RegistryEntryArrayType regArray = new RegistryEntryArrayType();
      logMetacat.info("++++++++ after create recogridRegarraytype");
      regArray.setRegistryEntry(regEntries);
      logMetacat.info("++++++++ end of submit query");
      return regArray;
   }


  
   
   private int GetIdxByPredefinedFieldName(String instr)
   {
      for(int i=0;i<NumPredefinedFields;i++)
      {
         if(instr.compareToIgnoreCase(PredefinedFieldNames[i]) == 0)
           return i;
      }
      return -1;
   }

   private int GetIdxByPredefinedFieldId(String instr)
   {
      for(int i=0;i<NumPredefinedFields;i++)
      {
         if(instr.compareToIgnoreCase(PredefinedFieldIds[i]) == 0)
           return i;
      }

      return -1;
   }
   
   /*
    * This method will automatic assign a docid for the given document
    */
   private String generateDocId()
   {
       String prefix = "ecogridService";
       int index1 = (new Double(Math.random() * 100)).intValue();
       int index2 = (new Double(System.currentTimeMillis()/10000)).intValue();
       int rev = 1;
       String docid = prefix+"."+index2+index1+"."+rev;
       logMetacat.info("Generated docid for registry document is "+docid);
       return docid;
   }
   // for testing purpose
   private void viewQuery(QueryType query)
   {
     if (query == null)
     {
       System.out.println("start to check the query");
       System.out.println("  query is null");
       System.out.println("end to check the query");
       return;
     }
     if (query.getAND() != null)
     {
       System.out.println("start to check the query");
       System.out.println("start first level of AND");
       ANDType and1= query.getAND();
       ORType [] orList = and1.getOR();
       if (orList != null)
       {
         for (int i =0; i< orList.length; i++)
         {
           ORType or = orList[i];
           System.out.println("  child " + i + "ORType start");
           ConditionType[] conditionList = or.getCondition();
           if (conditionList != null)
           {
             for (int j=0; j< conditionList.length; j++)
             {
                ConditionType condition = conditionList[j];
                System.out.println("    condtion kid " + j + " of ORType " + i +
                                    " start");
                 if (condition != null)
                 {
                    System.out.println("      expresspath is " +
                                             condition.getConcept().toString());
                    System.out.println("      operator is "+
                                            condition.getOperator().toString());
                    System.out.println("      value is "+ condition.get_value());
                  }//if
                System.out.println("    condtion kid " + j + " of ORType " + i +
                                   " end");
             }//for
           }//if
           System.out.println("  child " + i + " ORType end");
         }//for
       }//if
      System.out.println("end first level of AND");
      System.out.println("end to check the query");
     }//if
  }
}
