/* NexusFileReader actor for reading a Nexus file from the file system.

 Copyright (c) 1998-2003 The Regents of the University of California.
 All rights reserved.
 Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.

 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.

                                        PT_COPYRIGHT_VERSION_2
                                        COPYRIGHTENDKEY
*/

package org.cipres.kepler;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import ptolemy.actor.IOPort;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.Token;
import ptolemy.data.ObjectToken;
import ptolemy.data.StringToken;
import ptolemy.data.type.BaseType;
import ptolemy.data.type.Type;
import ptolemy.data.expr.Parameter;

import java.io.File;

import org.cipres.util.file.cipresFile;
import org.cipres.util.Config;

import javax.swing.JFileChooser;


//////////////////////////////////////////////////////////////////////////
//// NexusFileReader
/**
 * This is the implementation of the NexusFileReader actor using Ptolemy II.
 * This actor reads a Nexus file from the local file system and sends the file content as a string token.
 * 
 * @author Zhijie Guan  
 * guan@sdsc.edu
*/

public class NexusFileReader extends TypedAtomicActor {

	/** Construct NexusFileReader source with the given container and name.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the entity cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
	
    public NexusFileReader(CompositeEntity container, String name)
            throws NameDuplicationException, IllegalActionException  {
        super(container, name);
        
		inputTrigger = new TypedIOPort(this, "Trigger", true, false);
		inputTrigger.setTypeEquals(BaseType.GENERAL);
		
        outputFileContent = new TypedIOPort(this, "Nexus File Content", false, true);
        // Set the type constraint.
        outputFileContent.setTypeEquals(BaseType.GENERAL);


        fileNamePar = new Parameter(this, "Nexus File Name", new StringToken(""));
		
        _attachText("_iconDescription", "<svg>\n" +
                "<rect x=\"0\" y=\"0\" "
                + "width=\"60\" height=\"20\" "
                + "style=\"fill:white\"/>\n" +
                "</svg>\n");
		
    }

    ///////////////////////////////////////////////////////////////////
    ////                     ports and parameters                  ////

    /** The output port.  The type of this port will be set to String.
     */
	public TypedIOPort inputTrigger = null;
	public TypedIOPort outputFileContent = null;
    public Parameter fileNamePar;    
	
	///////////////////////////////////////////////////////////////////
	////                    functional variables                   ////
	
    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    /** Send the content of the Nexus file to the output.
     *  @exception IllegalActionException If it is thrown by the
     *   send() method sending out the token.
     */
    public void fire() throws IllegalActionException {
        super.fire();
		
		if (inputTrigger.hasToken(0)) {	
			inputTrigger.get(0);
			// get the nexus file name
			String fileNameStr = ((StringToken)fileNamePar.getToken()).stringValue(); 
			if (fileNameStr.length() == 0) {
				JFileChooser fileChooser = new JFileChooser();
				fileChooser.setDialogTitle("Choose Nexus File");
				fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
				fileChooser.setCurrentDirectory(new File(Config.getInstance().getDefaultNexusFileDir()));
				if (fileChooser.showOpenDialog(null) == JFileChooser.APPROVE_OPTION) {
					fileNameStr = fileChooser.getSelectedFile().getAbsolutePath();
				}
			}
			
			try {
				// This block is for sending file content
				cipresFile cf = new cipresFile(fileNameStr);
				StringToken fileContentToken = new StringToken(cf.getContent());
				outputFileContent.send(0,fileContentToken);

			} catch (Exception ex) {
				ex.printStackTrace();
			}
		}
    }
	
	/** Post fire the actor. Return false to indicated that the
	 *  process has finished. If it returns true, the process will
	 *  continue indefinitely.
	 */
	public boolean postfire() {
		return false;
	}	
}

