/**
 *    '$RCSfile: EcoGridServicesController.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2006/01/05 14:42:42 $'
 *   '$Revision: 1.7 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.seek.ecogrid;

import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xpath.XPathAPI;
import org.ecoinformatics.seek.ecogrid.exception.InvalidEcoGridServiceException;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This class will controll the current service user want to search:
 * user can add, remove and update the service list
 * @author Jing Tao
 *  
 */

public abstract class EcoGridServicesController 
{

	  protected final static Log log;
	  protected final static boolean isDebugging;
	  static {
		  	log = LogFactory.getLog( "org.ecoinformatics.seek.ecogrid.EcoGridServicesController" );
		  	isDebugging = log.isDebugEnabled();
		  }
  
  
  
  /**
  * Remove a given service from current service list. The key for removing
  * is endpoint. If endpoint is same, we think they are same service
  * @param service EcoGridService
  */
   public abstract void removeService(EcoGridService service);
 
 /**
  * Add a service to service list. If the service is duplicate or
  * the service is not a query service. It will throw an exception
  * @param service EcoGridService
  * @throws InvalidEcoGridServiceException
  */
  public abstract void addService(EcoGridService service) 
                                     throws InvalidEcoGridServiceException;
  
  /**
   * Update a sercie to sercie list. If the service couldn't be found, it
   * will throw a InvalidEcoGridServiceException
   * @param service EcoGridService
   * @throws InvalidEcoGridServiceException
   */
   public abstract void updateService(EcoGridService service) 
                                         throws InvalidEcoGridServiceException;
   
   /**
    * Method to get a service vector
    * @return Vector
    */                                 
    public abstract Vector getServicesList();
    public abstract Vector getSelectedServicesList();
    
    public abstract void setServiceList(Vector serviceList);
    
  /*
   * Given a service node in configure file, this method will transfer it to
   * EcoGridService java object 
   */
  protected EcoGridService generateServiceFromServiceNode(Node serviceNode)
  {
    EcoGridService service = null;
    if (serviceNode != null)
    {
      String serviceName = getKidNodeValue(serviceNode, 
                                  MetadataSpecificationInterface.SERVICENAME);
      String serviceType = getKidNodeValue(serviceNode, 
                                    MetadataSpecificationInterface.SERVICETYPE);
      String endPoint    = getKidNodeValue(serviceNode, 
                                       MetadataSpecificationInterface.ENDPOINT);
      DocumentType[] documentTypeList = getDocumentList(serviceNode);
      
      // generate a service java object
      try
      {
        service = new EcoGridService();
        service.setServiceName(serviceName);
        service.setServiceType(serviceType);
        service.setEndPoint(endPoint);
        service.setDocumentTypeList(documentTypeList);
      }//try
      catch (Exception e)
      {
        service = null;
        log.debug("couldn't get a service from configure file ",e);
      }//catch
    }//if
    return service;
  }//handleSericeNode
  
  /*
   * Give a parentnode and kid node name, it will return the kid node value
   */
  private String getKidNodeValue(Node parentNode, String kidName)
  {
    String nodeValue = null;
    try 
    {
      NodeList nodeList = XPathAPI.selectNodeList(parentNode, kidName);
      if (nodeList != null) 
      {
        Node node = nodeList.item(0);
        if (node != null && node.getFirstChild() != null) 
        {
          nodeValue = node.getFirstChild().getNodeValue();
        }//if
      }//if
    }//try
    catch (Exception e) 
    {
    	if (isDebugging) {
    		log.debug("Couldn't find " + kidName,e);
    	}
    } //catch
	if (isDebugging) {
		log.debug("The value of " + kidName + " is " + nodeValue);
	}
    return nodeValue;
  }//getKideNodeValue
  
  /*
   * Given a servcie node and return a documentType list array
   */
   private DocumentType[] getDocumentList(Node serviceNode)
   {
     DocumentType[] documentList = null;
     Vector documentVector       = new Vector();
     NodeList documentTypeNodeList   = null;
     try
     {
       //get documenttype node list
       documentTypeNodeList = XPathAPI.selectNodeList(serviceNode,
                                MetadataSpecificationInterface.DOCUMENTTYPE);
     }
     catch (Exception e)
     {
    	 log.debug("Couldn't find document list in config", e);
       return documentList;
     }//catch

     if (documentTypeNodeList != null) 
     {
       int size = documentTypeNodeList.getLength();
       for (int i = 0; i < size; i++) 
       {
         Node documentTypeNode = documentTypeNodeList.item(i);
         String namespace = getKidNodeValue(documentTypeNode,
                                            MetadataSpecificationInterface.
                                            NAMESPACE);
         String label = getKidNodeValue(documentTypeNode,
                                        MetadataSpecificationInterface.LABEL);
         DocumentType type = null;
         try 
         {
           type = new DocumentType(namespace, label);
         }
         catch (Exception e) 
         {
        	 log.debug("Couldn't generate a document type",e);
           continue;
         }
         // the vector will stored the document type
         documentVector.add(type);
       } //for
       //transfer a vector to array.
       int length = documentVector.size();
       documentList = new DocumentType[length];
       for (int i = 0; i < length; i++) 
       {
         documentList[i] = (DocumentType) documentVector.elementAt(i);
       } //for
 
     } //if
 
     return documentList;
   }//getDocumentList

  
 
  
  /*
   * Judage if a given service already existed in a service list. The key
   * is endpoint. If it is duplicate service, index in vector will be returned
   * Otherwise, -1 will be returned.
   */
  protected static int isDuplicateService(EcoGridService service, Vector serviceList)
                                      throws InvalidEcoGridServiceException
  {
    int duplicateIndex = -1;
    if (service == null || serviceList == null)
    {
      throw new InvalidEcoGridServiceException("The service or service list is"+
         " null and couldn't judge if the given service is duplicate in list");
    }
    String givenEndPoint = service.getEndPoint();
    if (givenEndPoint == null || givenEndPoint.trim().equals(""))
    {
      throw new InvalidEcoGridServiceException("The given service doesn't have "+
         "endpoint and couldn't judge if the given service is duplicate in list");
    }
    int size = serviceList.size();
    for (int i=0; i <size; i++)
    {
      EcoGridService currentService = (EcoGridService)serviceList.elementAt(i);
      String currentEndPoint = currentService.getEndPoint();
      if (currentEndPoint != null && currentEndPoint.equals(givenEndPoint))
      {
        duplicateIndex = i;
        break;
      }//if
    }//for
    return duplicateIndex;
  }//isDuplicateService

}//EcoGridServiceController