/**
 *    '$RCSfile: ActorBuilder.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2006/01/05 14:53:25 $'
 *   '$Revision: 1.14 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.seek.workflow;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringReader;
import java.lang.reflect.Constructor;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.kernel.ComponentEntity;
import ptolemy.kernel.CompositeEntity;

/**
 * class to build the actor source and compile it into a class
 */
public class ActorBuilder
{
  public static String PACKAGE = "${dynpackage}";
  public String DYNSRC = "@dynsrc@";

  TypedAtomicActor actor;
  StringBuffer src;
  String origName;
  String name;
  String description;
  String implClass;
  TypedIOPortObject[] inputs;
  TypedIOPortObject[] outputs;
  CompositeEntity container;

  protected final static Log log;
  static {
  	log = LogFactory.getLog( "org.ecoinformatics.seek.workflow.ActorBuilder" );
  }

  /**
   * constructor
   */
  public ActorBuilder(String name, String description,
    TypedIOPortObject[] inputs, TypedIOPortObject[] outputs,
    CompositeEntity container)
  {
    this.container = container;
    src = new StringBuffer();

    DYNSRC = System.getProperty("KEPLER") + DYNSRC;
    log.debug("DYNSRC: " + DYNSRC);
    log.debug("DYNPACKAGE: " + PACKAGE);

    this.origName = name;
    this.name = processName(name);
    this.description = description;
    this.implClass = implClass;
    this.inputs = inputs;
    this.outputs = outputs;

    buildActor();
    try
    {
      compileActor();
    }
    catch(Exception e)
    {
      e.printStackTrace();
      throw new RuntimeException("Error compiling actor: " + e.getMessage());
    }
  }

  /**
   * alters the name the user entered to make it a valid java class name
   */
  public static String processName(String name)
  {
    //right now, we'll just take the spaces out.  we could convert it to
    //camel caps or something too.  I think there are some illegal chars
    //that should also be taken out.
    Pattern p = Pattern.compile("\\s");
    Matcher m = p.matcher(name);
    return m.replaceAll("");
  }

  /**
   * create the source to be compiled
   */
  private void buildActor()
  {
    addLicense(src);
    src.append("package ").append(PACKAGE).append(";\n");
    addImports(src);

    src.append("/*<p>").append(description).append("</p>*/\n");
    src.append("\n");
    src.append("public class ").append(name).append(" extends TypedAtomicActor\n");
    src.append("{\n");
    src.append("  //input ports\n");

    //add the ports here
    for(int i=0; i< inputs.length; i++)
    {
      src.append("  public TypedIOPort ").append(inputs[i].getName());
      src.append(" = new ").append(inputs[i].getConstructorDeclaration());
      src.append(";\n");
    }

    src.append("  //output ports\n");

    for(int i=0; i< outputs.length; i++)
    {
      src.append("  public TypedIOPort ").append(outputs[i].getName());
      src.append(" = new ").append(outputs[i].getConstructorDeclaration());
      src.append(";\n");
    }

    src.append("\n");
    src.append("  public ").append(name);
    src.append("(CompositeEntity container, String name)\n");
    src.append("    throws NameDuplicationException, IllegalActionException\n");
    src.append("  {\n");
    src.append("    super(container, name);\n");
    src.append("    BlankActorPaneFactory factory = new BlankActorPaneFactory");
    src.append("(this, \"").append(name).append("\");\n");
    src.append("    factory.setContainer(container);\n");

    //add the types declarations for the ports here
    for(int i=0; i<inputs.length; i++)
    {
      src.append("    ");
      src.append(inputs[i].getName()).append(".setTypeEquals(");
      src.append(inputs[i].getTypeDeclaration()).append(");\n");
    }

    for(int i=0; i<outputs.length; i++)
    {
      src.append("    ");
      src.append(outputs[i].getName()).append(".setTypeEquals(");
      src.append(outputs[i].getTypeDeclaration()).append(");\n");
    }

    src.append("\n  }\n");
    src.append("\n");
    src.append("  public void initialize()\n");
    src.append("    throws IllegalActionException\n");
    src.append("  {\n");
    src.append("  }\n");
    src.append("\n");
    src.append("  public boolean prefire()\n");
    src.append("    throws IllegalActionException\n");
    src.append("  {\n");
    src.append("    return super.prefire();\n");
    src.append("  }\n");
    src.append("\n");
    src.append("  public void fire()\n");
    src.append("    throws IllegalActionException\n");
    src.append("  {\n");
    /*When this class can actually use the implementationClass, it will be
      fired from here*/
    src.append("    super.fire();\n");
    src.append("    JOptionPane.showMessageDialog(null,\n");
    src.append("      \"The actor ").append(name);
    src.append(" is not yet implemented.\",\n");
    src.append("      \"Error\", JOptionPane.ERROR_MESSAGE);\n");
    src.append("  }\n");

    addBlankActorPaneFactory(src, name);

    src.append("}\n");
  }

  /**
   * compile the source
   */
  private void compileActor() throws ClassNotFoundException, IOException,
    Exception
  {
    log.debug("Compiling actor....");
    String packageDirPath = DYNSRC + "/" +
      PACKAGE.replaceAll("\\.", "/") + "/";
    log.debug("packageDirPath: " + packageDirPath);
    File packageDir = new File(packageDirPath);
    packageDir.mkdirs();
    File sourceFile = new File(packageDirPath + "/" + name + ".java");

    //write out the source file to disk
    FileWriter fw = new FileWriter(sourceFile);
    StringReader sr = new StringReader(src.toString());
    char[] c = new char[1024];
    int numread = sr.read(c, 0, 1024);
    while(numread != -1)
    {
      fw.write(c, 0, numread);
      numread = sr.read(c, 0, 1024);
    }

    fw.flush();
    fw.close();

    //compile it
    int compileReturnCode = com.sun.tools.javac.Main.compile(
      new String[]{sourceFile.getAbsolutePath()});

    if(compileReturnCode != 0)
    {
      //there were compilation errors.  throw an exception here
      throw new Exception("There were errors compiling the actor " +
        "source.  This shouldn't happen.");
    }

    log.debug("Done compiling actor.");
    log.debug("Creating actor class object....");
    actor = (TypedAtomicActor)createObject(PACKAGE + "." + name, name, container);
    log.debug("Done creating actor class object.");
  }

  /**
   * returns the new actor class created from this class
   */
  public TypedAtomicActor getActor()
  {
    return actor;
  }

  /**
   * return the name of the created class
   */
  public String getActorClass()
  {
    return processName(PACKAGE + "." + name);
  }

  /**
   * returns the actor source that was compiled to create the actor class
   */
  public String getActorSource()
  {
    return src.toString();
  }

  /**
   * returns the &lt;entity&gt; MoML for the created class.  This can then be
   * appended into a MoML xml file fore reading in vergil
   */
  public String getActorMoml()
  {
    StringBuffer moml = new StringBuffer();
    moml.append("<entity name=\"").append(name).append("\"");
    moml.append(" class=\"").append(getActorClass()).append("\">");
    moml.append("<doc>").append(description).append("</doc></entity>");
    return moml.toString();
  }

  /**
   * creates an object of a type className with the constructor
   * (compositeEntity, String)
   * @param className the name of the class to create
   * @param actorName the name of the actor
   * @param container the model the actor is contained in
   */
  public static Object createObject(String className, String actorName,
    CompositeEntity container)
                throws RuntimeException
  {
    Object object = null;
    System.out.println("className: " + className);
    System.out.println("actorName: " + actorName);
    System.out.println("container: " + container);
    try
    {
      List l = container.entityList();
      System.out.println("l: " + l.toString());
      for(int i=0; i<l.size(); i++)
      {  //look for the actor in the container.  if it's already there, remove it
        ComponentEntity ce = (ComponentEntity)l.get(i);
        if(ce.getName().equals(actorName))
        {
          try
          {
            ce.setContainer(null);
            break;
          }
          catch(Exception e)
          {
            throw new RuntimeException("Error removing existing actor from " +
              "container: " + e.getMessage());
          }
        }
      }

      //create the class
      Class classDefinition = Class.forName(className);
      //add the arg types
      Class[] args = new Class[] {CompositeEntity.class, String.class};
      //create a constructor
      Constructor constructor = classDefinition.getConstructor(args);
      //set the args
      Object[] argImp = new Object[] {container, actorName};
      //create the object
      object = constructor.newInstance(argImp);
    }
    catch (InstantiationException e)
    {
      throw new RuntimeException("Error instantiating new actor class: " +
        e.getMessage());
    }
    catch (IllegalAccessException e)
    {
      throw new RuntimeException("Error accessing new actor class: " +
        e.getMessage());
    }
    catch (ClassNotFoundException e)
    {
      throw new RuntimeException("New actor class not found: " +
        e.getMessage());
    }
    catch(NoSuchMethodException e)
    {
      throw new RuntimeException("The constructor could not be found: " +
        e.getMessage());
    }
    catch(java.lang.reflect.InvocationTargetException e)
    {
      e.printStackTrace();
      throw new RuntimeException("The target actor could not be invoked: " +
        e.getMessage());
    }
    return object;
  }

  /**
   *
   */
  private static void addBlankActorPaneFactory(StringBuffer src, String name)
  {
     //private class that handles reopening the actor in the NewActorFrame
    src.append("\n  private class BlankActorPaneFactory extends EditorPaneFactory\n");
    src.append("  {\n");
    src.append("    private CompositeEntity cont;\n");
    src.append("    public BlankActorPaneFactory(NamedObj container, String name)\n");
    src.append("      throws NameDuplicationException, IllegalActionException\n");
    src.append("    {\n");
    src.append("      super(container, name);\n");
    src.append("    }\n");
    src.append("\n");
    src.append("    public java.awt.Component createEditorPane()\n");
    src.append("    {\n");
    src.append("      Effigy e = ptolemy.actor.gui.Configuration.findEffigy(cont);\n");
    src.append("      Container c = e.topEffigy().showTableaux().getFrame();\n");

    src.append("      Configuration conf = (Configuration)e.toplevel();\n");
    src.append("      CompositeEntity actorLibrary =");
    src.append("(CompositeEntity)conf.getEntity(\"actor library\");\n");
    src.append("      CompositeEntity actorLibNest =");
    src.append("(CompositeEntity)actorLibrary.getEntity(\"actor library\");\n");
    src.append("      CompositeEntity dynActor =");
    src.append("(CompositeEntity)actorLibNest.getEntity(\"dynamicActor\");\n");

    src.append("      NewActorFrame naFrame = new ");
    src.append("NewActorFrame(c, dynActor, \"").append(PACKAGE).append(".");
    src.append(name).append("\");\n");

    src.append("      naFrame.setActorLibrary(dynActor);\n");
    src.append("      return naFrame;\n");
    src.append("    }\n");
    src.append("\n");
    src.append("    public void setContainer(CompositeEntity cont)\n");
    src.append("    {\n");
    src.append("      this.cont = cont;\n");
    src.append("    }\n");
    src.append("  }\n");
  }


  /**
   * adds the license statement to the code
   */
  private static void addLicense(StringBuffer src)
  {
    src.append("/**\n");
    src.append(" *  Copyright: 2000 Regents of the University of California and the\n");
    src.append(" *             National Center for Ecological Analysis and Synthesis\n");
    src.append(" *\n");
    src.append(" * ********This class was auto-generated by the SEEK AMS engine.********  \n");
    src.append(" * This software is governed by the following license:\n");
    src.append(" *\n");
    src.append(" * This program is free software; you can redistribute it and/or modify\n");
    src.append(" * it under the terms of the GNU General Public License as published by\n");
    src.append(" * the Free Software Foundation; either version 2 of the License, or\n");
    src.append(" * (at your option) any later version.\n");
    src.append(" *\n");
    src.append(" * This program is distributed in the hope that it will be useful,\n");
    src.append(" * but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
    src.append(" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
    src.append(" * GNU General Public License for more details.\n");
    src.append(" *\n");
    src.append(" * You should have received a copy of the GNU General Public License\n");
    src.append(" * along with this program; if not, write to the Free Software\n");
    src.append(" * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA\n");
    src.append(" */\n");
  }

  /**
   * adds the import statements to the code
   */
  private static void addImports(StringBuffer src)
  {
    src.append("//tokens\n");
    src.append("import ptolemy.data.RecordToken;\n");
    src.append("import ptolemy.data.StringToken;\n");
    src.append("import ptolemy.data.IntToken;\n");
    src.append("import ptolemy.data.DoubleToken;\n");
    src.append("import ptolemy.data.LongToken;\n");
    src.append("import ptolemy.data.Token;\n");
    src.append("import ptolemy.actor.gui.EditorPaneFactory;\n");
    src.append("import ptolemy.data.type.RecordType;\n");
    src.append("import ptolemy.data.expr.Parameter;\n");
    src.append("import ptolemy.data.type.BaseType;\n");
    src.append("import ptolemy.data.type.Type;\n");
    src.append("import ptolemy.kernel.CompositeEntity;\n");
    src.append("import ptolemy.kernel.util.*;\n");
    src.append("import ptolemy.actor.gui.Effigy;\n");
    src.append("import ptolemy.actor.gui.Configuration;\n");
    src.append("\n");
    src.append("import ptolemy.actor.TypedAtomicActor;\n");
    src.append("import ptolemy.actor.TypedIOPort;\n");
    src.append("import org.ecoinformatics.seek.workflow.gui.NewActorFrame;\n");
    src.append("import javax.swing.JOptionPane;\n");
    src.append("import java.awt.Container;\n");
    src.append("\n");
  }
}
