/**
 *    '$RCSfile: DynamicActorMomlHandler.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2006/01/05 14:53:25 $'
 *   '$Revision: 1.9 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.seek.workflow;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.StringReader;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ecoinformatics.util.Util;

/**
 * This class creates or opens a new moml file that contains the entity
 * descriptions for all of the dynamically created actors.  If the moml
 * file does not already exist, it is created and the base moml markup
 * is inserted.  The only markup that should be appended dynamically
 * to this moml file is &lt;entity&gt; subtrees.
 */
public class DynamicActorMomlHandler
{
  private String PTINYACTORSFILESTR = "/configs/ptolemy/configs/kepler/basicActorLibrary.xml";
  private File PTINYACTORSFILE;
  private String MOMLFILEPATH = "/configs/ptolemy/configs/kepler/DynamicActors.xml";
  private String MOMLPACKAGE = "ptolemy/configs/kepler/DynamicActors.xml";
  File momlFile;

  protected final static Log log;
  static {
  	log = LogFactory.getLog( "org.ecoinformatics.seek.workflow.DynamicActorMomlHandler" );
  }

  /**
   * creates a new momlFile if it does not exist and opens it for editing
   * if it does exist
   * @param  momlFile  Description of Parameter
   */
  public DynamicActorMomlHandler(File momlFile)
  {
    String PTII = System.getProperty("KEPLER");
    PTINYACTORSFILE = new File(PTII + PTINYACTORSFILESTR);
    MOMLFILEPATH = PTII + MOMLFILEPATH;

    log.debug("PTINYACTORSFILE: " + PTINYACTORSFILE);
    log.debug("MOMLFILEPATH: " + MOMLFILEPATH);
    log.debug("MOMLPACKAGE: " + MOMLPACKAGE);
    this.momlFile = momlFile;
    try
    {
      if(!momlFile.exists())
      { //create the actual file if it doesn't already exist
        createNewMomlFile();
      }
      addDynFileToVergilConfig();
    }
    catch(Exception e)
    {
      throw new RuntimeException("Could not create new moml file: " +
          e.getMessage());
    }
  }

  /**
   * adds the input line into the actors file to add the new dynamic moml
   */
  public void addDynFileToVergilConfig() throws Exception
  {
    FileReader fr = new FileReader(PTINYACTORSFILE);
    StringBuffer sb = new StringBuffer();
    char[] c = new char[1024];
    int numread = fr.read(c, 0, 1024);
    while(numread != -1)
    {
      sb.append(c, 0, numread);
      numread = fr.read(c, 0, 1024);
    }
    fr.close();

    String fileString = sb.toString();
    if(fileString.indexOf(MOMLPACKAGE) == -1)
    { //make sure the path isn't already there
      String inputstring = "<input source=\"" + MOMLPACKAGE + "\"/>";
      String newPtinyFile = Util.substitute(fileString, "</group>",
        inputstring + "\n</group>");
      FileWriter fw = new FileWriter(PTINYACTORSFILE);
      StringReader sr = new StringReader(newPtinyFile);
      numread = sr.read(c, 0, 1024);
      while(numread != -1)
      {
        fw.write(c, 0, numread);
        numread = sr.read(c, 0, 1024);
      }

      fw.flush();
      fw.close();
    }
  }

  /**
   * adds an entity segment of moml code.  This string must conform to the
   * entity element in the moml dtd.
   * @param  momlEntity  the entity moml to add to the moml file.
   */
  public void addMoml(String momlEntity)
  {
    try
    {
      String newMomlText = addEntity(momlEntity);
      writeFile(newMomlText);
    }
    catch(Exception e)
    {
      throw new RuntimeException("Error adding moml entity fragment: " +
        e.getMessage());
    }
  }

  /**
   * write out the moml file
   */
  private void writeFile(String momlText) throws Exception
  {
    FileWriter fw = new FileWriter(momlFile);
    StringReader sr = new StringReader(momlText);
    char[] c = new char[1024];
    int numread = sr.read(c, 0, 1024);

    while(numread != -1)
    {
      fw.write(c, 0, numread);
      numread = sr.read(c, 0, 1024);
    }

    fw.flush();
    fw.close();
  }

  /**
   * adds an entity fragment to a moml document
   */
  private String addEntity(String entityFragment) throws Exception
  {
    String moml = toString();
    return Util.substitute(moml, "</group>",
      "  " + entityFragment + "\n\n      </group>");
  }

  /**
   * returns a string representation of the moml file as it currently exists
   * on disk.
   */
  public String toString()
  {
    StringBuffer sb = new StringBuffer();
    try
    {
      FileReader fr = new FileReader(momlFile);
      char[] c = new char[1024];
      int numread = fr.read(c, 0, 1024);

      while(numread != -1)
      {
        sb.append(c, 0, numread);
        numread = fr.read(c, 0, 1024);
      }
    }
    catch(Exception e)
    {
      throw new RuntimeException("Error opening moml file for read: " +
          e.getMessage());
    }

    return sb.toString();
  }

  /**
   * create a new base moml file to add entities to
   * @throws  Exception  if the new file cannot be created
   */
  private void createNewMomlFile()
    throws Exception
  {
    StringBuffer moml = new StringBuffer();

    moml.append("<?xml version=\"1.0\" standalone=\"no\"?>\n");
    moml.append("<!DOCTYPE plot PUBLIC \"-//UC Berkeley//DTD MoML 1//EN\"\n");
    moml.append("\"http://ptolemy.eecs.berkeley.edu/xml/dtd/MoML_1.dtd\">\n");
    moml.append("<entity name=\"dynamicActor\" ");
    moml.append("class=\"ptolemy.moml.EntityLibrary\">\n");
    moml.append("  <configure>\n");
    moml.append("    <?moml\n");
    moml.append("      <group>\n");
    moml.append("        <doc>Dynamically Created Actors</doc>\n");

    moml.append("      </group>\n    ?>\n  </configure>\n</entity>");

    writeFile(moml.toString());
  }
}

