/**
 *    '$RCSfile: NewActorFrame.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2006/01/05 14:53:26 $'
 *   '$Revision: 1.18 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.seek.workflow.gui;

import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.border.Border;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ecoinformatics.seek.workflow.ActorBuilder;
import org.ecoinformatics.seek.workflow.DynamicActorMomlHandler;
import org.ecoinformatics.seek.workflow.TypeObject;
import org.ecoinformatics.seek.workflow.TypedIOPortObject;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.gui.MessageHandler;
import ptolemy.kernel.ComponentEntity;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.ChangeRequest;
import ptolemy.moml.MoMLChangeRequest;

/**
 * A component for editing Step information.
 */
public class NewActorFrame extends JFrame
{
  // CONSTANTS
  private final static int PAD = 5;
  private final static int HORIZONTAL = 600;
  private final static int SPACE = 3;
  private final static int WIDE_SPACE = 10;
  private final static int COLUMNS = 25;
  private final static int ROWS = 4;
  private final static String MOMLFILESTR = "/configs/ptolemy/configs/kepler/DynamicActors.xml";

  private String DYNSRC = "@dynsrc@";
  private String DYNPACKAGE = "${dynpackage}";
  private File MOMLFILE;

  // PRIVATE MEMBERS
  private JTextField nameField;
  private JTextArea descriptionArea;
  private DefaultListModel inputsModel = null;
  private JList inputs;
  private DefaultListModel outputsModel = null;
  private JList outputs;
  private JTextArea scriptArea;
  private JButton cancelButton;
  private JButton saveButton;
  private JButton addInputButton;
  private JButton editInputButton;
  private JButton rmInputButton;
  private JButton addOutputButton;
  private JButton editOutputButton;
  private JButton rmOutputButton;
  private boolean cancelled;
  private int listSelection = -1;

  private CompositeEntity actorLibrary;
  private CompositeEntity container;

  private static Log log;
  private static boolean isDebugging;
  
  static {
	  log = LogFactory.getLog( "org.ecoinformatics.seek.workflow.gui.NewActorFrame" );
	  isDebugging = log.isDebugEnabled();
  }

  /**
   * Construct the editor dialog with the given Step.
   *
   * @param  parent  the parent frame for this dialog
   */
  public NewActorFrame(Container parent)
  {
    super();
    DYNSRC = System.getProperty("KEPLER") + DYNSRC;

    initializeFields();
    layoutFields();
    init(parent, "New Actor");
    setVisible(true);
  }

  /**
   * used to open an actor for editing
   */
  public NewActorFrame(Container parent, CompositeEntity container, String name)
  {
    String actorName = name.substring(name.lastIndexOf(".") + 1, name.length());

    initializeFields();
    layoutFields();
    init(parent, "Editing " + name);
    setVisible(true);
    nameField.setText(actorName);

    this.container = container;
    Object o = ActorBuilder.createObject(name, actorName, container);
    TypedAtomicActor actor = (TypedAtomicActor)o;
    List portList = actor.portList();
    for(int i=0; i<portList.size(); i++)
    {
      TypedIOPort port = (TypedIOPort)portList.get(i);

      TypeObject typeObj = new TypeObject(port.getType());

      TypedIOPortObject portObj = new TypedIOPortObject(port.getName(),
        typeObj, port.isInput(), port.isOutput());

      if(port.isInput())
      {
        inputsModel.addElement(portObj);
      }
      else if(port.isOutput())
      {
        outputsModel.addElement(portObj);
      }
    }
  }

  /**
   * setup all the visual components and pack the frame
   */
  private void init(Container parent, String name)
  {
    MOMLFILE = new File(System.getProperty("KEPLER") + MOMLFILESTR);

    setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    setTitle(name);
    cancelled = false;

    // register a listener for ActionEvents
    ActionHandler myActionHandler = new ActionHandler();
    MouseActionHandler mouseHandler = new MouseActionHandler();

    cancelButton.addActionListener(myActionHandler);
    saveButton.addActionListener(myActionHandler);
    addInputButton.addActionListener(myActionHandler);
    editInputButton.addActionListener(myActionHandler);
    rmInputButton.addActionListener(myActionHandler);
    addOutputButton.addActionListener(myActionHandler);
    editOutputButton.addActionListener(myActionHandler);
    rmOutputButton.addActionListener(myActionHandler);
    inputs.addMouseListener(mouseHandler);
    outputs.addMouseListener(mouseHandler);

    pack();

    //put the window in the middle of its parent
    if(parent != null)
    {
      int x = parent.getX();
      int y = parent.getY();
      int width = parent.getWidth();
      int height = parent.getHeight();
      setLocation(x + ((width / 2) - (this.getWidth() / 2)),
                  y + ((height / 2) - (this.getHeight() / 2)));
    }
  }

  /**
   * Get the cancelled flag.
   *
   * @return    boolean true if the dialog was dismissed using the Cancel button
   */
  public boolean isCancelled()
  {
    return cancelled;
  }

  /**
   * Initialize default values for all fields that will display in the editor.
   */
  private void initializeFields()
  {
    nameField = new JTextField(COLUMNS);
    descriptionArea = new JTextArea(ROWS, COLUMNS);
    inputsModel = new DefaultListModel();
    outputsModel = new DefaultListModel();
    scriptArea = new JTextArea(3 * ROWS, COLUMNS);
  }

  /**
   * Layout the form fields in a reasonable arrangement.
   */
  private void layoutFields()
  {

    Container content = getContentPane();

    //content.setBackground(Color.white);
    content.setLayout(new BoxLayout(getContentPane(), BoxLayout.X_AXIS));
    content.add(Box.createRigidArea(new Dimension(WIDE_SPACE, PAD)));

    JPanel editPane = new JPanel();

    editPane.setLayout(new BoxLayout(editPane, BoxLayout.Y_AXIS));
    //editPane.setBackground(Color.white);
    editPane.add(Box.createRigidArea(new Dimension(HORIZONTAL, PAD)));

    // Top section: Step Metadata
    editPane.add(Box.createRigidArea(new Dimension(PAD, WIDE_SPACE)));
    editPane.add(createPanel(nameField, "Name:"));

    descriptionArea.setLineWrap(true);
    editPane.add(Box.createRigidArea(new Dimension(PAD, WIDE_SPACE)));
    editPane.add(createPanel(descriptionArea, "Description"));

    // Need a horizontal line or image here
    editPane.add(Box.createRigidArea(new Dimension(PAD, WIDE_SPACE)));

    // Midle section: inputs/outputs
    JPanel ioPane = new JPanel();

    ioPane.setLayout(new BoxLayout(ioPane, BoxLayout.X_AXIS));
    //ioPane.setBackground(Color.white);

    JPanel inputsPane = new JPanel();

    inputsPane.setLayout(new BoxLayout(inputsPane, BoxLayout.Y_AXIS));
    //inputsPane.setBackground(Color.white);

    JLabel inlabel = new JLabel("Inputs");

    inputsPane.add(inlabel);
    inputsPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));
    inputs = new JList(inputsModel);
    inputs.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

    JScrollPane inputScroll = new JScrollPane(inputs);

    inputsPane.add(inputScroll);
    ioPane.add(inputsPane);
    ioPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));

    JPanel inButtonsPane = new JPanel();

    inButtonsPane.setLayout(new BoxLayout(inButtonsPane, BoxLayout.Y_AXIS));
    //inButtonsPane.setBackground(Color.white);
    addInputButton = new JButton("Add");
    inButtonsPane.add(addInputButton);
    inButtonsPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));
    editInputButton = new JButton("Edit");
    inButtonsPane.add(editInputButton);
    inButtonsPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));
    rmInputButton = new JButton("Remove");
    inButtonsPane.add(rmInputButton);
    inButtonsPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));
    ioPane.add(inButtonsPane);
    ioPane.add(Box.createRigidArea(new Dimension(WIDE_SPACE, PAD)));

    ioPane.add(Box.createHorizontalGlue());
    ioPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));

    JPanel outputsPane = new JPanel();

    outputsPane.setLayout(new BoxLayout(outputsPane, BoxLayout.Y_AXIS));
    //outputsPane.setBackground(Color.white);

    JLabel outlabel = new JLabel("Outputs");

    outputsPane.add(outlabel);
    outputsPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));
    outputs = new JList(outputsModel);

    JScrollPane outputScroll = new JScrollPane(outputs);

    outputsPane.add(outputScroll);
    ioPane.add(outputsPane);
    ioPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));

    JPanel outButtonsPane = new JPanel();

    outButtonsPane.setLayout(
        new BoxLayout(outButtonsPane, BoxLayout.Y_AXIS));
    //outButtonsPane.setBackground(Color.white);
    addOutputButton = new JButton("Add");
    outButtonsPane.add(addOutputButton);
    outButtonsPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));
    editOutputButton = new JButton("Edit");
    outButtonsPane.add(editOutputButton);
    outButtonsPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));
    rmOutputButton = new JButton("Remove");
    outButtonsPane.add(rmOutputButton);
    outButtonsPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));
    ioPane.add(outButtonsPane);
    ioPane.add(Box.createRigidArea(new Dimension(WIDE_SPACE, PAD)));

    editPane.add(ioPane);
    // Need a horizontal line or image here
    editPane.add(Box.createRigidArea(new Dimension(PAD, WIDE_SPACE)));

    cancelButton = new JButton("Cancel");
    saveButton = new JButton("Save");
    editPane.add(Box.createRigidArea(new Dimension(PAD, WIDE_SPACE)));

    JPanel buttonPanel = new JPanel();

    buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
    //buttonPanel.setBackground(Color.white);
    buttonPanel.add(Box.createHorizontalGlue());
    buttonPanel.add(cancelButton);
    buttonPanel.add(Box.createRigidArea(new Dimension(PAD, PAD)));
    buttonPanel.add(saveButton);
    editPane.add(buttonPanel);

    editPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));

    content.add(editPane);
    content.add(Box.createRigidArea(new Dimension(WIDE_SPACE, PAD)));
  }

  /**
   * sets the actor library that the new actors will be added to
   */
  public void setActorLibrary(CompositeEntity actorLib)
  {
    this.actorLibrary = actorLib;
  }

  public void setContainer(CompositeEntity container)
  {
    this.container = container;
  }

  /**
   * returns the actor library that this class is adding new actors to
   */
  public CompositeEntity getActorLibrary()
  {
    return actorLibrary;
  }

  /**
   * Create a JPanel with a horizontal layout, placing the component
   * and its field horizontally next to one another for text fields, and
   * as a titled border for text areas.
   *
   * @param  comp   the component to be placed on the panel
   * @param  label  the label for the component
   * @return        Description of the Returned Value
   */
  private JPanel createPanel(JComponent comp, String label)
  {
    JPanel pane = new JPanel();

    pane.setLayout(new BoxLayout(pane, BoxLayout.X_AXIS));
    //pane.setBackground(Color.white);

    if(comp instanceof JTextArea)
    {
      JScrollPane scroll = new JScrollPane(comp);

      //scroll.setBackground(Color.white);

      Border border = BorderFactory.createTitledBorder(
          BorderFactory.createLineBorder(Color.black), label);

      scroll.setBorder(border);
      pane.add(scroll);
    }
    else if(comp instanceof JTextField)
    {
      JLabel fieldLabel = new JLabel(label);

      pane.add(fieldLabel);
      pane.add(Box.createRigidArea(new Dimension(PAD, SPACE)));
      pane.add(comp);
    }

    return pane;
  }

  /**
   * Perform actions associated with the cancel button
   *
   * @param  event  Description of Parameter
   */
  private void cancelButtonHandler(ActionEvent event)
  {
    cancelled = true;
    dispose();
  }

  /**
   * Add an input to the list
   *
   * @param  event  The feature to be added to the InputButtonHandler attribute
   */
  private void addInputButtonHandler(ActionEvent event)
  {
    handleAddPort(inputsModel, true, false);
  }

  /**
   * Add an output to the list
   *
   * @param  event  The feature to be added to the OutputButtonHandler attribute
   */
  private void addOutputButtonHandler(ActionEvent event)
  {
    handleAddPort(outputsModel, false, true);
  }

  /**
   * add a new port to one of the lists.
   */
  private void handleAddPort(DefaultListModel portModel, boolean isInput,
    boolean isOutput)
  {
    PortEditorDialog win = new PortEditorDialog(this, isInput, isOutput);
    TypedIOPortObject portObj = win.getPort();
    portModel.addElement(portObj);
    win.dispose();
  }

  /**
   * Edit an input item
   *
   * @param  event  Description of Parameter
   */
  private void editInputButtonHandler(ActionEvent event)
  {
    handleEditPort(inputsModel, inputs);
  }

  /**
   * Edit an output item
   *
   * @param  event  Description of Parameter
   */
  private void editOutputButtonHandler(ActionEvent event)
  {
    handleEditPort(outputsModel, outputs);
  }

  /**
   * handles editing a port
   * @param portModel
   * @param list
   */
  private void handleEditPort(DefaultListModel portModel, JList list)
  {
    TypedIOPortObject port = (TypedIOPortObject)list.getSelectedValue();
    PortEditorDialog win = new PortEditorDialog(this, port);

    if(!win.isCancelled())
    {
      int position = list.getSelectedIndex();
      TypedIOPortObject portObj = win.getPort();
      portModel.set(position, portObj);
    }

    win.dispose();
  }

  /**
   * Remove an input item
   *
   * @param  event  Description of Parameter
   */
  private void rmInputButtonHandler(ActionEvent event)
  {
    int position = inputs.getSelectedIndex();

    inputsModel.remove(position);
  }

  /**
   * Remove an output item
   *
   * @param  event  Description of Parameter
   */
  private void rmOutputButtonHandler(ActionEvent event)
  {
    int position = outputs.getSelectedIndex();

    outputsModel.remove(position);
  }

  /**
   * Perform actions associated with the save button
   *
   * @param  event  Description of Parameter
   */
  private void saveButtonHandler(ActionEvent event)
  {
    //make sure the name isn't null
    if(nameField.getText() == null || nameField.getText().trim().equals(""))
    {
      JOptionPane.showMessageDialog(this,
        "Please fill out the name of the new actor.",
        "Error", JOptionPane.ERROR_MESSAGE);
      return;
    }

    //make sure none of the ports have a duplicate name
    if(checkPorts())
    {
      JOptionPane.showMessageDialog(this,
        "One or more ports have the same name.  Please change the name\n" +
        "of one of the ports and try again.",
        "Error", JOptionPane.ERROR_MESSAGE);
      return;
    }

    Object[] o = inputsModel.toArray();
    TypedIOPortObject[] inputports = new TypedIOPortObject[o.length];
    for(int i=0; i<o.length; i++)
    {
      inputports[i] = (TypedIOPortObject)o[i];
    }

    o = outputsModel.toArray();
    TypedIOPortObject[] outputports = new TypedIOPortObject[o.length];
    for(int i=0; i<o.length; i++)
    {
      outputports[i] = (TypedIOPortObject)o[i];
    }

    ActorBuilder builder = new ActorBuilder(nameField.getText(),
      descriptionArea.getText(), inputports, outputports, container);

    //get the new actor and add it to the library
    TypedAtomicActor newActor = builder.getActor();
    boolean success = addActorToLibrary(newActor);

    //write out a new moml file
    DynamicActorMomlHandler momlHandler = new DynamicActorMomlHandler(MOMLFILE);
    momlHandler.addMoml(builder.getActorMoml());

    if(success)
    {
      JOptionPane.showMessageDialog(this,
          "The actor has been added.",
          "Actor Added", JOptionPane.INFORMATION_MESSAGE);

      setVisible(false);
      dispose();
    }
  }

  /**
   * add the specified actor to the vergil library
   */
  private boolean addActorToLibrary(TypedAtomicActor actor)
  {
    if(actorLibrary == null)
    {
      return true;
    }

    try
    {
      List l = actorLibrary.entityList();
      for(int i=0; i<l.size(); i++)
      {  //look for the actor in the container.  if it's already there, remove it
        ComponentEntity ce = (ComponentEntity)l.get(i);
        if(ce.getName().equals(actor.getName()))
        {
          try
          {
            ce.setContainer(null);
            /*StringWriter buffer = new StringWriter();
            ce.exportMoML(buffer, 1);

            ChangeRequest request = new MoMLChangeRequest(ce, actorLibrary,
              buffer.toString());
            actorLibrary.requestChange(request);*/
            System.out.println("actor " + ce.getName() + " removed from lib");
            break;
          }
          catch(Exception e)
          {
            throw new RuntimeException("Error removing existing actor from " +
              "container: " + e.getMessage());
          }
        }
      }

      log.debug("Adding new actor to the library...");
      actor.setContainer(actorLibrary);
      StringWriter buffer = new StringWriter();
      actor.exportMoML(buffer, 1);

      ChangeRequest request = new MoMLChangeRequest(actor, actorLibrary,
        buffer.toString());
      actorLibrary.requestChange(request);
      log.debug("New actor added to the library.");
    }
    catch(ptolemy.kernel.util.IllegalActionException iae)
    {
      MessageHandler.error( "Error adding the new actor to the actor " +
        "library: " + iae.getMessage());
      return false;
    }
    catch(ptolemy.kernel.util.NameDuplicationException nde)
    {
      MessageHandler.error( "Save In Library failed: An object" +
                " already exists in the user library with name " +
                "\"" + actor.getName() + "\".");
      return false;
    }
    catch(IOException ioe)
    {
      MessageHandler.error("The new actor could not be added to the " +
        "library because an io exception occured: " +
        ioe.getMessage());
      return false;
    }
    return true;
  }

  /**
   * make sure none of the input ports have the same name as an output port
   */
  private boolean checkPorts()
  {
    //make sure none of the input ports have the same name as an output port
    for(int i=0; i<inputsModel.size(); i++)
    {
      TypedIOPortObject iport = (TypedIOPortObject)inputsModel.elementAt(i);
      for(int j=0; j<outputsModel.size(); j++)
      {
        TypedIOPortObject oport = (TypedIOPortObject)outputsModel.elementAt(j);
        if(oport != null && iport != null)
        {
          if(oport.getName().trim().equals(iport.getName().trim()))
          { //can't have a port with the same name
            return true;
          }
        }
      }
    }
    //check the inputs
    for(int i=0; i<inputsModel.size(); i++)
    {
      TypedIOPortObject iport = (TypedIOPortObject)inputsModel.elementAt(i);
      for(int j=i + 1; j<inputsModel.size(); j++)
      {
        TypedIOPortObject oport = (TypedIOPortObject)inputsModel.elementAt(j);
        if(oport.getName().trim().equals(iport.getName().trim()))
        { //can't have a port with the same name
          return true;
        }
      }
    }
    //check the outputs
    for(int i=0; i<outputsModel.size(); i++)
    {
      TypedIOPortObject iport = (TypedIOPortObject)outputsModel.elementAt(i);
      for(int j=i + 1; j<outputsModel.size(); j++)
      {
        TypedIOPortObject oport = (TypedIOPortObject)outputsModel.elementAt(j);
        if(oport.getName().trim().equals(iport.getName().trim()))
        { //can't have a port with the same name
          return true;
        }
      }
    }

    return false;
  }

  
  /**
   * Listener used to detect button presses
   */
  private class ActionHandler implements ActionListener
  {
    /**
     *  Description of the Method
     *
     * @param  event  Description of Parameter
     */
    public void actionPerformed(ActionEvent event)
    {
      Object object = event.getSource();

      if(object == saveButton)
      {
        saveButtonHandler(event);
      }
      else if(object == cancelButton)
      {
        cancelButtonHandler(event);
      }
      else if(object == addInputButton)
      {
        addInputButtonHandler(event);
      }
      else if(object == addOutputButton)
      {
        addOutputButtonHandler(event);
      }
      else if(object == editInputButton)
      {
        editInputButtonHandler(event);
      }
      else if(object == editOutputButton)
      {
        editOutputButtonHandler(event);
      }
      else if(object == rmInputButton)
      {
        rmInputButtonHandler(event);
      }
      else if(object == rmOutputButton)
      {
        rmOutputButtonHandler(event);
      }
    }
  }

  /**
   * mouse handler for double click on the list boxes
   */
  private class MouseActionHandler implements MouseListener
  {
    public void mouseClicked(MouseEvent e)
    {
      Object o = e.getSource();

      if(e.getClickCount() == 2)
      { //double click on the list box to edit the port
        if(o == inputs)
        {
          handleEditPort(inputsModel, inputs);
        }
        else if(o == outputs)
        {
          handleEditPort(outputsModel, outputs);
        }
      }
    }

    public void mouseEntered(MouseEvent e)
    {}

    public void mouseExited(MouseEvent e)
    {}

    public void mousePressed(MouseEvent e)
    {}

    public void mouseReleased(MouseEvent e)
    {}
  }
}

