/* Removes an element occurrences from an array.

Copyright (c) 1998-2004 The Regents of the University of California.
All rights reserved.
Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.

PT_COPYRIGHT_VERSION_2
COPYRIGHTENDKEY

*/

package org.geon;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.actor.parameters.PortParameter;
import ptolemy.data.ArrayToken;
import ptolemy.data.BooleanToken;
import ptolemy.data.Token;
import ptolemy.data.type.ArrayType;
import ptolemy.data.type.BaseType;
import ptolemy.data.type.Type;
import ptolemy.data.type.Typeable;
import ptolemy.graph.Inequality;
import ptolemy.graph.InequalityTerm;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

//////////////////////////////////////////////////////////////////////////
//// ArrayRemoveElement
/**
   Removes an element occurrences from an array. 
   This actor reads an array from the input port and 
   an element value from either a port or a parameter and 
   removes the element occurrences from the array. If the array contains
   no other values, leave a single occurrence.  
   @UserLevelDocumentation Removes an element occurrences an array.  
   This actor reads an array from the input port and and an element from either a port
   or a parameter and removes the element occurrences from the array. If the array contains
   no other values, leave a single occurrence.  

   @author Efrat Jaeger
   @version $Id: ArrayRemoveElement.java,v 1.3 2006/03/05 01:45:21 jaeger Exp $
   @since Ptolemy II 4.0.1
*/

public class ArrayRemoveElement extends TypedAtomicActor {

    /** Construct an actor with the given container and name.
     *  @param container The container.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the actor cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
    public ArrayRemoveElement(CompositeEntity container, String name)
            throws NameDuplicationException, IllegalActionException  {
        super(container, name);
        
        array = new TypedIOPort(this, "array", true, false);
        output = new TypedIOPort(this, "output", false, true);
        
        // set type constraints.
        array.setTypeEquals(new ArrayType(BaseType.UNKNOWN));
        ArrayType inputArrayType = (ArrayType)array.getType();
        Type elementType = inputArrayType.getElementType();
        
        output.setTypeAtLeast(array);

        // Set parameters.
        element = new PortParameter(this, "elemnt");
        element.setTypeEquals(elementType);
    }

    ///////////////////////////////////////////////////////////////////
    ////                         parameters                        ////

    /** Input array.
	 * @UserLevelDocumentation The input array. 
     */
    public TypedIOPort array;
    
    /** The resulting output array
	 * @UserLevelDocumentation The resulting output array. 
     */
    public TypedIOPort output;
    
    /** The element to be removed.
	 * @UserLevelDocumentation The element to be removed. 
     */
    public PortParameter element;

    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

     /** Consume an array and an element value and removes the element
      *  occurrences from the array.
     */
    public void fire() throws IllegalActionException {
        // NOTE: This has be outside the if because we need to ensure
        // that if an index token is provided that it is consumed even
        // if there is no input token.
        element.update();
        Token elementToken = element.getToken();
        if (array.hasToken(0)) {
            ArrayToken token = (ArrayToken)array.get(0);
            Vector outputElements = new Vector();
            for (int i=0; i<token.length(); i++) {
            	BooleanToken bt = elementToken.isEqualTo(token.getElement(i)); 
				if (!bt.booleanValue()) {
            		outputElements.add(token.getElement(i));
            	}
            }
            Token outputArray[] = null;
            if (outputElements.size() > 0) {
            	outputArray = new Token[outputElements.size()];
            	outputElements.toArray(outputArray);
            } else  {
            	outputArray = new Token[1];
            	outputArray[0] = elementToken;
            }
            output.broadcast(new ArrayToken(outputArray));
        }
    }

    /** Return the type constraints of this actor.
     *  In this class, the constraints are that the type of the array input port
     *  is an array type, and the type of the element port is equal to 
     *  the type of the elements of the input array.
     *  @return A list of instances of Inequality.
     *  @see ptolemy.actor.TypedAtomicActor#typeConstraintList
     */
    public List typeConstraintList() {
        Type inputType = array.getType();
        if (inputType == BaseType.UNKNOWN) {
            array.setTypeEquals(new ArrayType(BaseType.UNKNOWN));
        } else if ( !(inputType instanceof ArrayType)) {
            throw new IllegalStateException("ArrayElement.typeConstraintList: "
                    + "The input type, " + inputType.toString() + " is not an "
                    + "array type.");
        }

        List result = new LinkedList();

        // collect constraints from contained Typeables
        Iterator ports = portList().iterator();
        while (ports.hasNext()) {
            Typeable port = (Typeable)ports.next();
            result.addAll(port.typeConstraintList());
        }

        Iterator typeables = attributeList(Typeable.class).iterator();
        while (typeables.hasNext()) {
            Typeable typeable = (Typeable)typeables.next();
            result.addAll(typeable.typeConstraintList());
        }

        // Add type constraint for the input.
        ArrayType inputArrayType = (ArrayType)array.getType();
        InequalityTerm elementTerm = inputArrayType.getElementTypeTerm();
        Inequality inequality = new Inequality(elementTerm,
                element.getTypeTerm());

        result.add(inequality);
        return result;
    }
}
