/**
 * For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.authentication;

/**
 * ProxyRepository is used to store all of the proxies requested by the user.
 * @author Zhijie Guan
 * guan@sdsc.edu
 *
 */

class ProxyRepository {
	static int repositoryCapacity = 10;		// repository capacity
	ProxyEntity repository[]=new ProxyEntity[repositoryCapacity];
	int repositoryTop = 0;					// pointer for the current top of the repository

	/**
	 * This function gets the first available proxy and return it to the invoker.
	 * @return The requested proxy, null if no such proxy exists.
	 */
	synchronized ProxyEntity getDefaultProxy() {
		if (repositoryTop > 0) {
			return repository[0];
		} else {
			return null;
		}
	}

	/**
	 * This function returns the proxy with the specific index in the proxyRepository.
	 * @param index Proxy index in the proxyRepository
	 * @return	The specified proxy
	 */
	ProxyEntity getProxyAt(int index) {
		if (index < repositoryTop) {
			return repository[index];
		} else {
			return null;
		}
	}

	/**
	 * This function is used to wait for user login.
	 * It uses "wait" method to synchronize the user login thread and the AuthenticationManager thread
	 *
	 */
	synchronized void waitForUserLogin() {
		try {
			wait();
		} catch (InterruptedException e) {
			System.err.println("ProxyEntity wait error: " + e);
		}
	}
	
	/**
	 * This function is used by LoginGUI to insert the new proxy into the proxyRepository
	 * @param proxy The new proxy
	 */
	protected synchronized void insertProxy(ProxyEntity proxy) {
		if (repositoryTop < repositoryCapacity) {		// still have space for new proxy
			repository[repositoryTop] = proxy;
			repositoryTop++;
        	notifyAll();
		} else {										// no space available
			System.err.println("ProxyRepository overflows!!! Resetting the ProxyRepository!");
			repositoryTop=0;
		}
	}
	
	/** 
	 * This function is used to search proxy with the specified user name and domain.
	 * @param userName	The specified username
	 * @param domain The specified domain
	 * @return Index of the proxy, -1 for not found
	 */
	protected synchronized int searchProxyInRepository(String userName, Domain domain) {
		int i;
		for (i=repositoryTop-1; i>=0; i--) {
			if ( ( repository[i].getUserName() == userName ) && ( repository[i].getDomain().equalTo(domain) ) )
				break;
		}
		return i;
	}

	/** 
	 * This function is used to search proxy with the specified domain.
	 * @param domain The specified domain
	 * @return Index of the proxy, -1 for not found
	 */
	protected synchronized int searchProxyInRepository(Domain domain) {
		int i;
		for (i=repositoryTop-1; i>=0; i--) {
			if ( repository[i].getDomain().equalTo(domain) ) {
				break;
			}
		}
		return i;
	}

}
