/**
 *    '$RCSfile: OntLibrarySearcher.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2005/11/01 20:39:10 $'
 *   '$Revision: 1.3 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.gui;


import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Stack;
import java.util.Vector;

import javax.swing.JTree;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import org.ecoinformatics.seek.sms.AnnotationEngine;
import org.kepler.moml.NamedObjId;

import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.NamedObj;

/**
 * A very simple, first draft ontology-based library search engine
 */
public class OntLibrarySearcher extends LibrarySearcher {

    private AnnotationEngine _engine = null; // for searching
    private Stack _pathStack; //the stack to create the return paths with
    private Hashtable _hashTable; // key is lsid of leaf, object is vector of treepaths to leaf


    /**
     * Constructor
     */
    public OntLibrarySearcher(JTree library, LibrarySearchPane searchPane) {
	super(library, searchPane);
	_engine = AnnotationEngine.instance();
    }

    
    /**
     * Search for ontology-keyword in the library
     * @param value the keyword in the ontology to search for
     */
    public LibrarySearchResults search(String value) {
	// assign the new results of the search
	results = new LibrarySearchResults(); // this is a protected class member
	if(value.trim().equals(""))
	    return results;

	// build the hash table 
	NamedObj root = (NamedObj)library.getModel().getRoot();
	_pathStack = new Stack();
	_pathStack.push(root);
	_hashTable = new Hashtable();
	buildHashTable(root, library.getModel());

	// get the ontology search results
	Vector ont_results = _engine.search(value, true);
	// find the appropriate results in the library
	Iterator iter = ont_results.iterator();
	while(iter.hasNext()) {
	    NamedObj obj = (NamedObj)iter.next();
	    Iterator idIter = getIds(obj).iterator();
	    // for each id, add the tree path to the result
	    while(idIter.hasNext()) {
		String id = (String)idIter.next();
		Iterator pathIter = hashTableGet(id);
		while(pathIter.hasNext()) {
		    TreePath path = (TreePath)pathIter.next();
		    if(!results.contains(path))
			results.add(path);
		}
	    }
	}
 	return results;
    }


    //
    // helper function to build the hash table of paths leading to NamedObj's with ids
    private void buildHashTable(NamedObj parent, TreeModel model) {
	for(int i = 0; i < model.getChildCount(parent); i++) {
	    NamedObj child = (NamedObj)model.getChild(parent, i);
	    _pathStack.push(child);
	    if(model.isLeaf(child)) {
		Iterator iter = getIds(child).iterator();
		while(iter.hasNext()) {
		    TreePath path = new TreePath(_pathStack.toArray());
		    hashTablePut(iter.next().toString(), path);
		}
	    }
	    else
		buildHashTable(child, model);

	    _pathStack.pop();
	}

    }

    //
    // helper function: given named object returns it's ids
    // 
    private Vector getIds(NamedObj obj) {
	Vector ids = new Vector();
	List idAtts = obj.attributeList(NamedObjId.class);
	Iterator iter = idAtts.iterator();
	while(iter.hasNext()) {
	    NamedObjId id = (NamedObjId)iter.next();
	    ids.add(id.getExpression());
	}
	return ids;
    }


    //
    // helper function: add an (id, path) to _hashtable
    //
    private void hashTablePut(String id, TreePath path) {
	if(!_hashTable.containsKey(id)) {
	    Vector paths = new Vector();
	    paths.add(path);
	    _hashTable.put(id, paths);
	}
	else {
	    Vector paths = (Vector)_hashTable.get(id);
	    if(!paths.contains(path)) {
		paths.add(path);
	    }
	}
    }

    // 
    // helper function: given an id, returns an iterator of its paths
    //
    private Iterator hashTableGet(String id) {
	Vector ids = (Vector)_hashTable.get(id);
	if(ids == null) 
	    return new Vector().iterator();
	return ids.iterator();
    }



    /**
     * provides any initialization needed prior to searching.  It is
     * called when the class is constructed.  Note that any local
     * variables of extending classes should be initialized in init
     * since it is called be the constructor of the super class
     * (LibrarySearcher).
     */
    protected void init() {
    }

    /**
     * A factory that creates the searcher to search the library
     */
    public static class Factory extends LibrarySearcherFactory {
	/** 
	 * Create a factory with the given name and container.
	 * @param container The container.
	 * @param name The name of the entity.
	 * @exception NameDuplicationException If the name coincides
	 * with an attribute already in the container
	 */ 
	public Factory(NamedObj container, String name)
	    throws IllegalActionException, NameDuplicationException {
	    super(container, name);
	}

	/**
	 * Creates an Ontology Library Searcher and returns it.
	 * @param library The model containing the actor library.
	 * @return A new Library Searcher.
	 */
	public LibrarySearcher createLibrarySearcher(JTree library, LibrarySearchPane searchPane) {
	    return new OntLibrarySearcher(library, searchPane);
	}


    } // Factory



} // OntLibrarySearcher



