/**
 *    '$RCSfile: CreateKSWDialog.java,v $'
 *
 *     '$Author: berkley $'
 *       '$Date: 2006/03/14 21:34:30 $'
 *   '$Revision: 1.36 $'
 *
 *  For Details: http://kepler-project.org
 *
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.ksw.gui;

import java.awt.Dimension;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.*;
import java.util.*;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.UIManager;
import javax.swing.JTree;
import javax.swing.JScrollPane;

import org.ecoinformatics.seek.sms.AnnotationEngine;
import org.kepler.ksw.KSWAlreadyCachedException;
import org.kepler.ksw.KSWBuilder;
import org.kepler.moml.KSWLibraryBuilder;
import org.kepler.objectmanager.ActorMetadata;
import org.kepler.objectmanager.lsid.DuplicateLSIDException;
import org.kepler.objectmanager.lsid.KeplerLSID;
import org.kepler.objectmanager.lsid.LSIDTree;
import org.kepler.objectmanager.lsid.LSIDGenerator;
import org.kepler.objectmanager.cache.*;
import org.kepler.objectmanager.library.LibraryIndex;
import org.kepler.objectmanager.library.LibraryIndexItem;
import org.kepler.objectmanager.library.LibraryIndexComponentItem;
import org.kepler.objectmanager.library.LibraryIndexConceptItem;
import org.kepler.objectmanager.library.LibraryIndexOntologyItem;
import org.kepler.gui.AnnotatedPTree;
import org.kepler.sms.SemanticType;

import ptolemy.actor.TypedCompositeActor;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.Entity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NamedObj;
import ptolemy.kernel.util.StringAttribute;
import ptolemy.moml.EntityLibrary;

import com.ibm.lsid.LSIDException;
;

/**
 *@author    Chad Berkley
 *
 * This dialog lets a user create a KSW file from a model on the canvas.  It
 * lets the user select whether to create an archive ksw with all components
 * or a transport ksw with component references only.
 */
public class CreateKSWDialog extends JDialog
{
  private Frame owner;
  private Entity entity;
  private CompositeEntity actorLibrary;

  private JTextField saveToTextBox = new JTextField();
  private JTextField nameTextBox = new JTextField();
  private JRadioButton archiveRadioButton = new JRadioButton("Archive", false);
  //right now, the transport and archive buttons do not show up on the gui,
  //but the dialog still uses the values to send true or false to KSWBuilder
  private JRadioButton transportRadioButton = new JRadioButton("Transport", true);
  private ButtonGroup radioGroup = new ButtonGroup();
  private JButton okButton = new JButton("OK");
  private JButton cancelButton = new JButton("Cancel");
  private JButton browseButton = new JButton("Browse");
  private JComboBox categoryCombo;
  private AnnotatedPTree categoryTree;

  /**
   * anonymous class to handle button events
   */
  private ActionListener buttonListener =
    new ActionListener()
    {
      public void actionPerformed(ActionEvent e)
      {
        if(e.getActionCommand().equals("OK"))
        {
          try
          {
            File kswFile = new File(saveToTextBox.getText());
            String nameStr = nameTextBox.getText();
            if(nameStr != null)
            { //set the user's name
              entity.setName(nameStr);
            }
            else
            {
              try
              { //if the user didn't set the name, just use the old name
                //plus the revision number
                nameTextBox.setText(entity.getName() + "-" + 
                  LSIDTree.getInstance().findNextRevision(
                  new KeplerLSID(
                    ((StringAttribute)entity.getAttribute("entityId")
                    ).getExpression())).getRevision());
              }
              catch(LSIDException lside)
              {
                //do nothing...just don't set the text box default
              }
            }
            
            //this part of the code gets the concept name from the user
            //selection and adds it to the new kar metadata as a semanticType
            Iterator attIt = entity.attributeList().iterator();
            int semTypeCount = 0;

            LibraryIndexItem selectedItem = LibraryIndex.getInstance().findItem(
              ((EntityLibrary)categoryTree.getSelectionPath().getLastPathComponent()).getName());
            //String selectedConceptName = ((NamedObj)
            //      categoryTree.getSelectionPath().getLastPathComponent()).getName();
            String selectedConceptName = ((LibraryIndexConceptItem)selectedItem).getConceptId();
            String ontoLSID = findOntologyForItem(selectedItem).getLSID().toString();
            
            while(attIt.hasNext())
            { //update the semanticType with whatever the user chose in the JTree
              NamedObj obj = (NamedObj)attIt.next();
              //System.out.println("att name: " + obj.getName() + " classname: " + obj.getClassName());
              
              if(obj.getClassName().indexOf("org.kepler.sms.SemanticType") != -1)
              {
                semTypeCount++;
              }
            }
            
            //add the new semantic type
            SemanticType st = new SemanticType(entity, 
              "semanticType" + semTypeCount);
            String conceptId = ontoLSID + "#" + selectedConceptName;
            st.setConceptId(conceptId);
            KSWBuilder kswBuilder = new KSWBuilder(entity,
              archiveRadioButton.isSelected(), false, kswFile);
            processKSW(kswFile);
          }
          catch(Exception ex)
          {
            JOptionPane.showMessageDialog(getThis(),
             ex.getMessage(),
             "alert", JOptionPane.ERROR_MESSAGE);
            System.out.println(ex.getMessage());
            ex.printStackTrace();
          }
          getThis().setVisible(false);
        }
        else if(e.getActionCommand().equals("Cancel"))
        {
          getThis().setVisible(false);
        }
        else if(e.getActionCommand().equals("Browse"))
        {
          JFileChooser chooser = new JFileChooser();
          int returnVal = chooser.showOpenDialog(getThis());
          if(returnVal == JFileChooser.APPROVE_OPTION)
          {
            String savePath = chooser.getSelectedFile().getAbsolutePath();
            if(!savePath.endsWith(".kar"))
            {
              savePath += ".kar";
            }
            saveToTextBox.setText(savePath);
          }
        }
      }
    };
  
  /**
   *
   */
  public CreateKSWDialog(Frame owner, Entity entity, 
    CompositeEntity actorLibrary)
  {
    this(owner, entity, actorLibrary, null);
  }
  
  /**
   * This is the default constructor
   *
   *@param  owner
   *@param  namedObj
   */
  public CreateKSWDialog(Frame owner, Entity entity, 
    CompositeEntity actorLibrary, String conceptName)
  {
    super(owner);

    if(actorLibrary == null)
      System.out.println("ActorLibrary is null");
    
    this.actorLibrary = actorLibrary;
    this.entity = entity;
    if(entity instanceof TypedCompositeActor && !entity.isClassDefinition())
    {
      String msg = "You cannot create a kar file from an instantiated composite. " +
        "You must first make it into a class by right clicking on the composite, " +
        "and choosing \"Convert to Class\" from the menu.";
      JOptionPane.showMessageDialog(getThis(),
             msg,
             "alert", JOptionPane.ERROR_MESSAGE);
      return;
    }
    
    this.owner = owner;
    
    try
    {
      categoryTree = new AnnotatedPTree(
        LibraryIndex.getInstance().createLibrary(), null);
      categoryTree.setVisible(true);
    }
    catch(ptolemy.kernel.util.IllegalActionException iae)
    {
      throw new RuntimeException("Could not populate the category combo list: " + 
        iae.getMessage());
    }
    
    if(conceptName != null)
    {
      for(int i=0; i<categoryCombo.getItemCount(); i++)
      {
        if(categoryCombo.getItemAt(i).toString().equals(conceptName))
        {
          categoryCombo.setSelectedIndex(i);
          break;
        }
      }
    }
    
    try
    {
      
      nameTextBox.setText(entity.getName() + "-" + 
        LSIDTree.getInstance().findNextRevision(
        new KeplerLSID(
          ((StringAttribute)entity.getAttribute("entityId")
          ).getExpression())).getRevision());
    }
    catch(LSIDException lside)
    {
      //do nothing...just don't set the text box default
    }
    catch(Exception e)
    {
    }

    archiveRadioButton.setToolTipText("Archives all components into the KAR " +
        "file which may create a large KAR file size.");
    transportRadioButton.setToolTipText("Components are referenced in the KAR " +
        "file instead of copied, which creates a smaller, but less " +
        "complete KAR file.");
    //add the radiobuttons to a group so that they are not mutually selectable
    radioGroup.add(archiveRadioButton);
    radioGroup.add(transportRadioButton);

    okButton.setActionCommand("OK");
    okButton.addActionListener(buttonListener);
    cancelButton.setActionCommand("Cancel");
    cancelButton.addActionListener(buttonListener);
    browseButton.setActionCommand("Browse");
    browseButton.addActionListener(buttonListener);

    // set title and close behavior
    this.setTitle("Save Model as KAR File");
    this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);

    JPanel pane = new JPanel();

    pane.setLayout(new BoxLayout(pane, BoxLayout.Y_AXIS));
    pane.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));

    JPanel bodyPane = createBodyPane();

    // add components to pane
    pane.add(bodyPane);
    pane.add(Box.createRigidArea(new Dimension(0, 5)));

    // add the pane
    setContentPane(pane);
    // set size
    this.setSize(385, 500);
    //setResizable(false);
  }

  /**
   * Main method for testing the dialog.
   *
   *@param  args  the arguments to the program
   */
  public static void main(String[] args)
  {
    try
    {
      // a composite "wrapper"
      TypedCompositeActor swf = new TypedCompositeActor();

      // windows look and feel
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());

      CreateKSWDialog dialog = new CreateKSWDialog(null, swf, null);

      dialog.setVisible(true);
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }
  
  /**
   * find the top most ontology item that is the extended parent of item.  
   * this returns null if no ontology parent is found.
   * @param item the item for which to search for the ontology
   */
  private LibraryIndexOntologyItem findOntologyForItem(LibraryIndexItem item)
  {
    Vector parents = item.getParents();
    Iterator parentItt = item.getParents().iterator();
    while(parentItt.hasNext())
    {
      LibraryIndexItem parent = (LibraryIndexItem)parentItt.next();
      if(parent.getType().equals("ontology"))
      {
        return (LibraryIndexOntologyItem)parent;
      }
      else
      {
        return findOntologyForItem(parent);
      }
    }
    return null;
  }

  /**
   * process the new ksw file into the actor library
   * @param kswFile the file to process
   */
  private void processKSW(File kswFile)
  {
    try
    {
      KeplerLSID[] actorLSIDs;
      CacheManager cache = CacheManager.getInstance();
      KARCacheObject kco = new KARCacheObject(kswFile);
      cache.insertObject(kco);
      try
      {
        actorLSIDs = kco.getActors();
      }
      catch(LSIDException lside)
      {
        JOptionPane.showMessageDialog(getThis(),
               "Error getting actor lsid: " + lside.getMessage(), "alert", 
               JOptionPane.ERROR_MESSAGE);
        return;
      }
      
      if(actorLSIDs != null)
      {
        System.out.println("processing kar file: " + kswFile.getAbsolutePath());
        //AnnotationEngine AEngine = AnnotationEngine.instance();
        for(int j=0; j<actorLSIDs.length; j++)
        {
          ActorCacheObject aco = (ActorCacheObject)cache.getObject(actorLSIDs[j]);
          ActorMetadata am = aco.getMetadata();
          try
          { //build an entity for each actor in the ksw and add it to the actorLibrary
            NamedObj actorEntity = am.getActorAsNamedObj(actorLibrary);
            LSIDGenerator lsidGenerator = LSIDGenerator.getInstance();
            //KeplerLSID newlsid = lsidGenerator.getNewLSID(am.getLSID());
            
            LibraryIndexItem subtree = LibraryIndex.getInstance().findItem(
              ((EntityLibrary)categoryTree.getSelectionPath().getLastPathComponent()).getName());
            
            LibraryIndex.getInstance().add(
              new LibraryIndexComponentItem(am.getName(), am.getLSID()), subtree);
            LibraryIndex.getInstance().refresh();
            util.EmptyChangeRequest request = new util.EmptyChangeRequest(actorEntity, "request change");
            actorEntity.requestChange(request);

            System.out.println("actor " + actorEntity.getName() +
              " loaded into the library in category " +
              subtree.getName() + " with id " + 
              am.getId());
          }
          catch(Exception e)
          {
            e.printStackTrace();
            JOptionPane.showMessageDialog(getThis(),
               "Error creating ComponentEntity from actor " +
              "metadata: " + e.getMessage(), "alert", 
               JOptionPane.ERROR_MESSAGE);
            return;
          }
        }
      }
    }
    catch(CacheException ce)
    {
      ce.printStackTrace();
      JOptionPane.showMessageDialog(getThis(),
             "Error with the cache: " + ce.getMessage(), "alert", 
             JOptionPane.ERROR_MESSAGE);
      return;
    }
  }

  /**
   * create the main body pane for the dialog
   *
   */
  private JPanel createBodyPane()
  {
    JPanel bodyPanel = new JPanel();
    JPanel mainPanel = createMainPanel();
    JPanel buttonPanel = createButtonPanel();

    bodyPanel.setLayout(new BoxLayout(bodyPanel, BoxLayout.Y_AXIS));
    mainPanel.setBorder(BorderFactory.createEtchedBorder());
    //mainPanel.setPreferredSize(new Dimension(348, 50));
    bodyPanel.add(mainPanel);
    bodyPanel.add(Box.createRigidArea(new Dimension(10, 10)));
    bodyPanel.add(buttonPanel);

    return bodyPanel;
  }

  /**
   * create the panel with the main content in it
   *
   */
  private JPanel createMainPanel()
  {
    JPanel mainPanel = new JPanel();
    JPanel captionPanel = createCaptionPanel();
    JPanel inputPanel = createInputPanel();

    mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.X_AXIS));
    mainPanel.add(Box.createRigidArea(new Dimension(5, 5)));
    mainPanel.add(captionPanel);
    mainPanel.add(Box.createRigidArea(new Dimension(5, 5)));
    mainPanel.add(inputPanel);

    return mainPanel;
  }

  /**
   * create the panel with the captions for the inputs
   *
   */
  private JPanel createCaptionPanel()
  {
    JPanel captionPanel = new JPanel();
    JLabel saveToLabel = new JLabel("Save to:");
    JLabel actorNameLabel = new JLabel("Actor Name:");
    JLabel categoryLabel = new JLabel("Category:");
    JLabel kswTypeLabel = new JLabel("KAR Type:");

    captionPanel.setLayout(new BoxLayout(captionPanel, BoxLayout.Y_AXIS));
    //captionPanel.add(Box.createRigidArea(new Dimension(0, 1)));
    captionPanel.add(saveToLabel);
    captionPanel.add(Box.createRigidArea(new Dimension(0, 20)));
    captionPanel.add(actorNameLabel);
    captionPanel.add(Box.createRigidArea(new Dimension(0, 20)));
    captionPanel.add(categoryLabel);
    captionPanel.add(Box.createRigidArea(new Dimension(0, 310)));
    //captionPanel.add(kswTypeLabel);
    //captionPanel.add(Box.createRigidArea(new Dimension(0, 5)));

    return captionPanel;
  }

  /**
   * create the panel with the user inputs
   */
  private JPanel createInputPanel()
  {
    JPanel inputPanel = new JPanel();
    inputPanel.setLayout(new BoxLayout(inputPanel, BoxLayout.Y_AXIS));

    JPanel textBoxPanel = new JPanel();
    textBoxPanel.setLayout(new BoxLayout(textBoxPanel, BoxLayout.X_AXIS));
    textBoxPanel.add(saveToTextBox);
    textBoxPanel.add(Box.createRigidArea(new Dimension(10,0)));
    textBoxPanel.add(browseButton);
    textBoxPanel.add(Box.createRigidArea(new Dimension(10,0)));
    
    JPanel nameBoxPanel = new JPanel();
    nameBoxPanel.setLayout(new BoxLayout(nameBoxPanel, BoxLayout.X_AXIS));
    nameBoxPanel.add(nameTextBox);
    nameBoxPanel.add(Box.createRigidArea(new Dimension(10,0)));
    
    JPanel categoryPanel = new JPanel();
    categoryPanel.setLayout(new BoxLayout(categoryPanel, BoxLayout.X_AXIS));
    categoryPanel.add(new JScrollPane(categoryTree));
    categoryPanel.add(Box.createRigidArea(new Dimension(10,0)));
    categoryPanel.setMinimumSize(new Dimension(50, 50));

    inputPanel.add(Box.createRigidArea(new Dimension(0,5)));
    inputPanel.add(textBoxPanel);
    
    inputPanel.add(Box.createRigidArea(new Dimension(0,10)));
    inputPanel.add(nameBoxPanel);
    
    inputPanel.add(Box.createRigidArea(new Dimension(0,10)));
    inputPanel.add(categoryPanel);

    //inputPanel.add(Box.createRigidArea(new Dimension(0,20)));
    //JPanel radioButtonPanel = createRadioPanel();
    //inputPanel.add(radioButtonPanel);
    //inputPanel.add(Box.createRigidArea(new Dimension(0,5)));

    return inputPanel;
  }

  /**
   * creates the panel for the radio buttons
   */
  private JPanel createRadioPanel()
  {
    JPanel radioButtonLayoutPanel = new JPanel();
    JPanel radioButtonPanel = new JPanel();
    radioButtonLayoutPanel.setLayout(new BoxLayout(radioButtonLayoutPanel, BoxLayout.X_AXIS));
    radioButtonPanel.setLayout(new BoxLayout(radioButtonPanel, BoxLayout.Y_AXIS));
    radioButtonPanel.add(archiveRadioButton);
    radioButtonPanel.add(Box.createRigidArea(new Dimension(0, 2)));
    radioButtonPanel.add(transportRadioButton);
    radioButtonLayoutPanel.add(radioButtonPanel);
    radioButtonLayoutPanel.add(Box.createRigidArea(new Dimension(140, 0)));
    return radioButtonLayoutPanel;
  }

  /**
   * create the panel with the OK and Cancel button
   *
   *@return
   */
  private JPanel createButtonPanel()
  {
    JPanel buttonPanel = new JPanel();
    buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
    buttonPanel.add(Box.createRigidArea(new Dimension(200, 0)));
    buttonPanel.add(okButton);
    buttonPanel.add(Box.createRigidArea(new Dimension(5, 0)));
    buttonPanel.add(cancelButton);

    return buttonPanel;
  }

  /**
   * returns the 'this' object to the anonymous class that handles button events
   */
  private CreateKSWDialog getThis()
  {
    return this;
  }
}

