/**
 *  '$RCSfile: ActorMetadata.java,v $'
 *  '$Author: kommineni $'
 *  '$Date: 2006/03/16 21:01:23 $'
 *  '$Revision: 1.60 $'
 *
 *  For Details:
 *  http://www.kepler-project.org
 *
 *  Copyright (c) 2003-2004 The Regents of the University of California.
 *  All rights reserved.
 *  Permission is hereby granted, without written agreement and without
 *  license or royalty fees, to use, copy, modify, and distribute this
 *  software and its documentation for any purpose, provided that the above
 *  copyright notice and the following two paragraphs appear in all copies
 *  of this software.
 *  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 *  FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 *  ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 *  THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 *  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 *  PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 *  PT_COPYRIGHT_VERSION_2
 *  COPYRIGHTENDKEY
 */
package org.kepler.objectmanager;

import java.io.*;
import java.util.*;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.zip.ZipEntry;
import java.lang.reflect.Constructor;

import org.kepler.moml.NamedObjId;
import org.kepler.moml.PortAttribute;
import org.kepler.moml.PropertyAttribute;
import org.kepler.moml.CompositeClassEntity;
import org.kepler.sms.SemanticType;
import org.kepler.objectmanager.lsid.KeplerLSID;

import ptolemy.actor.Director;
import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedCompositeActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.expr.Variable;
import ptolemy.data.expr.StringParameter;
import ptolemy.data.expr.Parameter;
import ptolemy.data.unit.UnitAttribute;
import ptolemy.kernel.ComponentEntity;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.Entity;
import ptolemy.kernel.Port;
import ptolemy.kernel.ComponentRelation;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.NamedObj;
import ptolemy.kernel.util.StringAttribute;
import ptolemy.kernel.util.SingletonConfigurableAttribute;
import ptolemy.kernel.util.ConfigurableAttribute;
import ptolemy.moml.MoMLParser;
import ptolemy.kernel.util.Workspace;

/**
 * This class parses and contains metadata for an actor.  this creates an
 * entity of the form:
 * <pre>
    &lt;entity name="Constant"&gt;
    &lt;property name="documentation" class="org.kepler.moml.DocumentationAttribute"&gt;
    actor to provide constant input
    &lt;/property&gt;
    &lt;property name="entityId" class="org.kepler.moml.NamedObjId"
                    value="urn:lsid:lsid.ecoinformatics.org:actor:101:1"/&gt;
    &lt;property name="class" value="org.kepler.actor.TestActor"
              class="ptolemy.kernel.util.StringAttribute"&gt;
      &lt;property name="id" value="urn:lsid:lsid.ecoinformatics.org:actor:1001:1"
                class="ptolemy.kernel.util.StringAttribute"/&gt;
    &lt;/property&gt;
    &lt;property name="output" class="org.kepler.moml.PortProperty&gt;
      &lt;property name="direction" value="output" class="ptolemy.kernel.util.StringAttribute"/&gt;
      &lt;property name="dataType" value="unknown" class="ptolemy.kernel.util.StringAttribute"/&gt;
      &lt;property name="isMultiport" value="false" class="ptolemy.kernel.util.StringAttribute"/&gt;
    &lt;/property&gt;
    &lt;property name="trigger" class="org.kepler.moml.PortProperty&gt;
      &lt;property name="direction" value="input" class="ptolemy.kernel.util.StringAttribute"/&gt;
      &lt;property name="dataType" value="unknown" class="ptolemy.kernel.util.StringAttribute"/&gt;
      &lt;property name="isMultiport" value="true" class="ptolemy.kernel.util.StringAttribute"/&gt;
    &lt;/property&gt;
    &lt;property class="org.kepler.sms.SemanticType" name="semanticType" value="urn:lsid:lsid.ecoinformatics.org:onto:1:1#ConstantActor"/&gt;
    &lt;property class="org.kepler.moml.Dependency" name="dependency" value="urn:lsid:lsid.ecoinformatics.org:nativeLib:1:1"/&gt;
    &lt;/entity&gt;
 * </pre>
 *@author     Chad Berkley
 *@created    April 07, 2005
 */
public class ActorMetadata implements Serializable
{
  //the id of the class that implements this actor
  private String classId;
  //the java class name.  for composites, this is the name of the xml file
  //like org.kepler.actor.MyCompositeActor
  private String className;
  //the class name as represented in ptII.  for composites, this is always
  //ptolemy.actor.TypedCompositeActor, for java classes, this is the
  //name of the java class
  private String internalClassName;
  //the id of the actor
  private String actorId;
  //the name of the actor
  private String actorName;
  //docs about the actor
  private String actorDoc;
  //the instantiation of the actor
  private NamedObj actor;
  //dependency ids
  private Vector dependencyVector = new Vector();
  //semantic types
  private Vector semanticTypeVector = new Vector();
  //ports
  private Vector portVector = new Vector();
  //changed flag
  private boolean changed = false;
  //attribute vector for generic attributes
  private Vector attributeVector = new Vector();
  //vector for relations in a composite
  private Vector relationVector = new Vector();
  private String links = null;
  
  /**
   * default constructor.  this is for serialization only.  do not use this.
   */
  public ActorMetadata()
  {
    //nothing here.
  }
  
  /**
   * Constructor.  Takes in xml metadata.  This should be a
   * moml entity with the kepler additional metadata properties.  The entity
   * is parsed and an ActorMetadata object is created with appropriate fields.
   * @param moml the xml metadata
   */
  public ActorMetadata(InputStream moml) throws InvalidMetadataException
  {
    this(moml, false);
  }

  /**
   * Constructor.  Takes in xml metadata.  This should be a
   * moml entity with the kepler additional metadata properties.  The entity
   * is parsed and an ActorMetadata object is created with appropriate fields.
   * @param moml the xml metadata
   * @param printMoml will print the input moml to stdout if printMoml is true.
   * this is for debugging purposes only.
   */
  public ActorMetadata(InputStream moml, boolean printMoml) throws InvalidMetadataException
  {
    try
    {
      //need to get the following info:
      /*this.className = className;
      this.classId = classId;
      this.actorId = actorId;
      this.actorName = actorName;
      this.actorDoc = actorDoc;
      this.actor = actor;*/
      
      StringBuffer sb = new StringBuffer();
      byte b[] = new byte[1024];
      int numread = moml.read(b, 0, 1024);
      while(numread != -1)
      {
        sb.append(new String(b, 0, numread));
        numread = moml.read(b, 0, 1024);
      }
      
      if(printMoml)
      { //for debugging
        System.out.println(sb.toString());
      }
      
      MoMLParser parser = new MoMLParser(new Workspace());
      parser.reset();
      NamedObj obj = parser.parse(null, sb.toString());

      if(obj instanceof TypedCompositeActor)
      {
        links = ((TypedCompositeActor)obj).exportLinks(1, null);
      }
      actorName = obj.getName();
      StringAttribute classAttribute =
        (StringAttribute)obj.getAttribute("class");
      classId = ((StringAttribute)
        classAttribute.getAttribute("id")).getExpression();
      className = classAttribute.getExpression();
      internalClassName = obj.getClassName();
      actorId = ((NamedObjId)obj.getAttribute("entityId")).getExpression();
      this.actor = getActorClass(className, actorName, obj);
      
      //FIXME: this documentation doesn't work the way i want it to
      //you should be able to put CDATA in the property element but
      //you can't because the MoMLParser doesn't support it.
      actorDoc = "";

      //get the semantic type and dependency lsids and any general properties
      Iterator i = obj.attributeList().iterator();
      while(i.hasNext())
      {
        Attribute a = (Attribute)i.next();
        String attName = a.getName();
        if(!attName.equals("entityId") &&
           !attName.equals("documentation") &&
           !attName.equals("class") &&
           !a.getClassName().equals("org.kepler.moml.PropertyEntity") &&
           !a.getClassName().equals("org.kepler.moml.PropertyAttribute") &&
           !a.getClassName().equals("org.kepler.moml.CompositeClassEntity") &&
           !a.getClassName().equals("org.kepler.moml.PortAttribute"))
        {
          String attValue = null;
          if(a instanceof StringAttribute)
          {
            attValue = ((StringAttribute)a).getExpression();
          }
          else if(a instanceof UnitAttribute)
          {
            attValue = ((UnitAttribute)a).getExpression();
          }
          else if(a instanceof Variable)
          {
            attValue = ((Variable)a).getExpression();
          }

          if(a.getName().indexOf("semanticType") != -1)
          {
            addSemanticType(a.getName(), attValue);
          }
          else if(a.getName().indexOf("dependency") != -1)
          {
            ClassedProperty prop = new ClassedProperty(a.getName(),
              attValue, a.getClass().getName());
            dependencyVector.add(prop);
          }
          else if(a.getName().indexOf("_iconDescription") != -1)
          {
            if(actor != null && actor.getAttribute("_iconDescription") != null)
            {
              boolean objIsDefault = false;
              boolean actorIsDefault = false;
              String objIconStr = ((SingletonConfigurableAttribute)a).getExpression();
              String actorIconStr = ((SingletonConfigurableAttribute)
                actor.getAttribute("_iconDescription")).getExpression();
              if(objIconStr.indexOf("<polygon points=\"-20,-10 20,0 -20,10\" style=\"fill:blue\"/>") != -1)
              {
                objIsDefault = true;
              }
              
              if(actorIconStr.indexOf("<polygon points=\"-20,-10 20,0 -20,10\" style=\"fill:blue\"/>") != -1)
              {
                actorIsDefault = true;
              }
              
              if((objIsDefault && actorIsDefault) ||
                 (objIsDefault && actorIsDefault) ||
                 (!objIsDefault && !actorIsDefault))
              {
                addAttribute(a);
              }
              else if(objIsDefault && !actorIsDefault)
              {
                //do nothing.  leave the attribute in the actor
              }
            }
            else
            {
              addAttribute(a);
            }
          }
          else
          {
            addAttribute(a); //add generic attributes
          }
        }
      }

      //get the ports
      parseNamedObj(obj);
      addAllRelations(obj);
    }
    catch(IOException ioe)
    {
      throw new InvalidMetadataException("Error reading data from lsid " + actorId +
        ": " + ioe.getMessage());
    }
    catch(Exception e)
    {
      e.printStackTrace();
      throw new InvalidMetadataException("Error in parsing actor metadata: " +
        e.getMessage());
    }
  }
  
  /**
   * builds a new ActorMetadata object from an existing NamedObj
   * @param am the ActorMetadata to build this object from.
   */
  public ActorMetadata(NamedObj obj)
  {
    this.actor = obj;
    this.className = obj.getClassName();
    this.actorId = ((NamedObjId)obj.getAttribute("entityId")).getExpression();
    this.actorName = obj.getName();
    
    try
    {
      if(obj instanceof TypedCompositeActor)
      {
        links = ((TypedCompositeActor)obj).exportLinks(1, null);
      }
    }
    catch(Exception e)
    {
      System.out.println("Error looking at links: " + e.getMessage());
    }
    
    //set the internalclassname
    if(obj instanceof TypedAtomicActor)
    {
      this.internalClassName = "ptolemy.kernel.ComponentEntity";
    }
    else if(obj instanceof Attribute)
    {
      this.internalClassName = "org.kepler.moml.PropertyEntity";
    }
    else if(obj instanceof TypedCompositeActor)
    {
      this.internalClassName = "org.kepler.moml.CompositeClassEntity";
    }
    
    addAllAttributes(actor);
    addAllRelations(actor);
    parseActor(actor);
  }

  /**
   * return the actor this object was built from.
   */
  public NamedObj getActor()
  {
    return actor;
  }

  /**
   * return the name of the actor
   */
  public String getName()
  {
    return actorName;
  }

  /**
   * set the name
   */
  public void setName(String name)
  {
    actorName = name;
  }

  /**
   * return the lsid of the actor class object
   */
  public String getId()
  {
    return actorId;
  }
  
  /**
   * returns the id of this object as a KeplerLSID
   */
  public KeplerLSID getLSID()
    throws com.ibm.lsid.LSIDException
  {
    return new KeplerLSID(actorId);
  }

  /**
   * set the id
   */
  public void setId(String id)
  {
    actorId = id;
  }

  /**
   * return the classname of the actor object.  this will be the java type
   * classname no matter what kind of actor we are describing.  for instance
   * if it is a java class, it will be a.b.c.ClassName where the file is
   * stored in a/b/c/ClassName.class.  If it is a MoML class, it will be
   * a.b.c.MomlClassName where the file is a/b/c/MomlClassName.xml
   */
  public String getClassName()
  {
    return className;
  }

  /**
   * return the internal class name of the actor object.  for composites this
   * will be ptolemy.actor.TypedCompositeActor.  For atomics, it will be
   * the full java class name.
   */
  public String getInternalClassName()
  {
    return internalClassName;
  }

  /**
   * return the id of the class
   */
  public String getClassId()
  {
    return classId;
  }

  /**
   * return the actor documentation
   */
  public String getActorDoc()
  {
    return actorDoc;
  }

  /**
   * add the id of a dependency to the metadata
   * @param id the id of the dependency to add
   */
  public void addDependency(String id)
  {
    ClassedProperty cp = new ClassedProperty("dependency", id,
      "org.kepler.moml.Dependency");
    dependencyVector.addElement(cp);
  }

  /**
   * add the id of a semantic type to the metadata
   * @param id the id of the semantic type to add
   */
  public void addSemanticType(String name, String id)
  {
    ClassedProperty cp = new ClassedProperty(name, id,
      "org.kepler.sms.SemanticType");
    semanticTypeVector.addElement(cp);
  }

  /**
   * return a vector of the ids of the semantic type reference
   */
  public Vector getSemanticTypes()
  {
    Vector v = new Vector();
    for(int i=0; i<semanticTypeVector.size(); i++)
    {
      v.addElement(((ClassedProperty)semanticTypeVector.elementAt(i)).value);
    }
    return v;
  }

  /**
   * return a vector of the ids of the semantic type reference
   */
  public Vector getDependencies()
  {
    Vector v = new Vector();
    for(int i=0; i<dependencyVector.size(); i++)
    {
      v.addElement(((ClassedProperty)dependencyVector.elementAt(i)).value);
    }
    return v;
  }

  /**
   * get the changed flag.  this is useful for keeping track of the state
   * of the actor metadata.  this flag does not affect this class in any
   * internal way except for setting the flag.
   */
  public boolean getChanged()
  {
    return changed;
  }

  /**
   * set the changed flag.  this is useful for keeping track of the state
   * of the actor metadata.  this flag does not affect this class in any
   * internal way except for setting the flag.
   * @param b
   */
  public void setChanged(boolean b)
  {
    changed = b;
  }

  /**
   * add a generic attribute to this actorMetadata object.
   * @param a the attribute to add
   */
  public void addAttribute(Attribute a)
  {
    if(a.getClassName().equals("ptolemy.actor.parameters.PortParameter"))
      {
      return;
      }
    attributeVector.add(a);
  }
  
  /**
   * add a relation
   */
  public void addRelation(ComponentRelation r)
  {
    relationVector.add(r);
  }

  /**
   * return this actor as a ComponentEntity
   * @param container the new ComponentEntity's container
   */
  public NamedObj getActorAsNamedObj(CompositeEntity container)
    throws Exception
  {
    NamedObj obj = null;

    if(actor instanceof TypedCompositeActor)
    { //if we are dealing with a composite entity, do stuff a bit differently
      //we need to instantiate the composite or else it will show up in the 
      //library as a class instead of an entity.  this causes kepler to think
      //that the user wants to drag a class to the canvas and it requires the 
      //user to instantiate the actor before using it.  by calling 
      //instantiate here, we bypass that.
      if(internalClassName.equals("org.kepler.moml.CompositeClassEntity"))
      {
        return (NamedObj)actor.clone(container.workspace());  //this kinda works
        //return (NamedObj)((TypedCompositeActor)actor).instantiate(container, actor.getName());
        //obj = (NamedObj)((TypedCompositeActor)actor).instantiate(container, actor.getName());
        //obj.setClassName(className);
        //return obj;
      }
      else
      {
        obj = (NamedObj)((TypedCompositeActor)actor).instantiate(container, actor.getName());
        obj.setClassName(className);
      }
    }
    else if(actor instanceof Director)
    { //this is a director or other Attribute derived class
      obj = new Director(container, actorName);
      obj.setClassName(className);
    }
    else if(actor instanceof Attribute)
    {
      obj = new Attribute(container, actorName);
      obj.setClassName(className);
    }
    else
    { //this is an atomic actor
      if(container != null)
      {
        obj = (NamedObj)actor.clone(container.workspace());
        ((TypedAtomicActor)obj).setContainer(container);
      }
      else
      {
        obj = (NamedObj)actor.clone(null);
      }
    }

    NamedObjId objId;
    StringAttribute classObj;
    StringAttribute classIdObj;

    try
    {
      objId = new NamedObjId(obj, "entityId");
    }
    catch(ptolemy.kernel.util.InternalErrorException iee)
    {
      objId = (NamedObjId)obj.getAttribute("entityId");
    }
    catch(ptolemy.kernel.util.NameDuplicationException nde)
    {
      objId = (NamedObjId)obj.getAttribute("entityId");
    }

    try
    {
      classObj = new StringAttribute(obj, "class");
      classIdObj = new StringAttribute(classObj, "id");
    }
    catch(ptolemy.kernel.util.InternalErrorException iee)
    {
      classObj = (StringAttribute)obj.getAttribute("class");
      classIdObj = (StringAttribute)classObj.getAttribute("id");
    }
    catch(ptolemy.kernel.util.NameDuplicationException nde)
    {
      classObj = (StringAttribute)obj.getAttribute("class");
      classIdObj = (StringAttribute)classObj.getAttribute("id");
    }

    objId.setExpression(actorId);
    classObj.setExpression(className);
    classIdObj.setExpression(classId);
    for(int i=0; i<semanticTypeVector.size(); i++)
    {
      ClassedProperty cp = (ClassedProperty)semanticTypeVector.elementAt(i);
      SemanticType semType = new SemanticType(obj, cp.name);
      semType.setExpression(cp.value);
    }
    /*
      FIXME:
      TODO: add dependencies and other info to the NamedObj
    */
    
    //add the general attributes to the object
    for(int i=0; i<attributeVector.size(); i++)
    {
      Attribute a = (Attribute)attributeVector.elementAt(i);
      Attribute aClone = (Attribute)a.clone(obj.workspace());
      try
      {
        aClone.setContainer(obj);
      }
      catch(NameDuplicationException nde)
      {
        //System.out.println("obj already has attribute " + a.getName());
        //ignore this, it shouldn't matter.
        // Specialized versions of some actors (e.g. RExpression actor
        // require that parameters be reset to new values
        // without the following code, these will not be reset
        // Dan Higgins - 1/20/2006
        String attValue;
        String attName;
        attName = aClone.getName();
        if(aClone instanceof StringAttribute)
          {
            attValue = ((StringAttribute)aClone).getExpression();
            StringAttribute sa = (StringAttribute)obj.getAttribute(attName);
            sa.setExpression(attValue);
          }
        else if(aClone instanceof StringParameter)
          {
            attValue = ((StringParameter)aClone).getExpression();
            StringParameter sp = (StringParameter)obj.getAttribute(attName);
            sp.setExpression(attValue);
          }
        else if(aClone instanceof Parameter)
          {
            attValue = ((Parameter)aClone).getExpression();
            Parameter pp = (Parameter)obj.getAttribute(attName);
            pp.setExpression(attValue);
          }

      }
    }

    //copy any extra moml ports over
    if(!(actor instanceof Director))
    {
      boolean found = false;

      for(int i=0; i<portVector.size(); i++)
      { //if there are any ports in the port vector that are not in the
        //port iterator, add them
        Iterator portIterator = ((Entity)actor).portList().iterator();
        PortMetadata pm = (PortMetadata)portVector.elementAt(i);
        while(portIterator.hasNext())
        {
          TypedIOPort p = (TypedIOPort)portIterator.next();
          String pmname = pm.name;
          if(pm.name.indexOf("kepler:") != -1)
          { //strip the kepler: off of the name.  kepler: is there to 
            //prevent namedup exceptions in the moml parser
            pmname = pm.name.substring(7, pmname.length());
          }
          
          if(p.getName().equals(pmname))
          {
            found = true;
            break;
          }
        }

        if(!found)
        {
          /*if(pm.name.indexOf("port_") != -1)
          {
            pm.name = pm.name.substring(5, pm.name.length());
          }*/
          
          TypedIOPort port = null;
          
          if(obj instanceof TypedAtomicActor)
          {
            Iterator portList = ((TypedAtomicActor)obj).portList().iterator();
            boolean flag = true;
            while(portList.hasNext())
            {
              TypedIOPort oldport = (TypedIOPort)portList.next();
              if(oldport.getName().equals(pm.name))
                flag = false;
            }
            if(flag)
              port = new TypedIOPort((TypedAtomicActor)obj, pm.name);
          }
          else
          {
            Iterator portList = ((TypedCompositeActor)obj).portList().iterator();
            boolean flag = true;
            while(portList.hasNext())
            {
              TypedIOPort oldport = (TypedIOPort)portList.next();
              if(oldport.getName().equals(pm.name))
                flag = false;
            }
            if(flag)
              port = new TypedIOPort((TypedCompositeActor)obj, pm.name);
          }
          
          if(port == null)
          {
            continue;
          }
          
          if(pm.direction.equals("input"))
          {
            port.setInput(true);
          }
          else if(pm.direction.equals("output"))
          {
            port.setOutput(true);
          }
          else if(pm.direction.equals("inputoutput"))
          {
            port.setInput(true);
            port.setOutput(true);
          }

          if(pm.multiport)
          {
            port.setMultiport(true);
          }
        }
        else
        {
          found = false;
        }
      }
    }

    return obj;
  }

  /**
   * return the moml xml representation of the actor.
   */
  public String toString()
  {
    StringBuffer sb = new StringBuffer();

    sb.append("<?xml version=\"1.0\"?>\n");
    if(actor instanceof PropertyAttribute)
    {
      sb.append("<property name=\"" + actorName + "\" class=\"org.kepler.moml.PropertyAttribute\">\n");
    }
    else if(actor instanceof Attribute)
    {
      sb.append("<property name=\"" + actorName + "\" class=\"org.kepler.moml.PropertyEntity\">\n");
    }
    else if(internalClassName.equals("org.kepler.moml.CompositeClassEntity"))
    {
      sb.append("<entity name=\"" + actorName + "\" class=\"org.kepler.moml.CompositeClassEntity\">\n");
    }
    else
    {
      sb.append("<entity name=\"" + actorName + "\" class=\"ptolemy.kernel.ComponentEntity\">\n");
    }
    sb.append("<property name=\"entityId\"  value=\"" + actorId + "\" class=\"org.kepler.moml.NamedObjId\"/>\n");
    sb.append("<property name=\"documentation\" class=\"org.kepler.moml.DocumentationAttribute\">\n");
    sb.append(actorDoc);
    sb.append("\n</property>\n");

    if(internalClassName.equals("ptolemy.actor.Director") ||
      internalClassName.equals("ptolemy.actor.TypedCompositeActor") ||
      internalClassName.equals("ptolemy.kernel.ComponentEntity") ||
      internalClassName.equals("org.kepler.moml.PropertyEntity") ||
      internalClassName.equals("org.kepler.moml.CompositeClassEntity"))
    {
      sb.append("<property name=\"class\" value=\"" + className +
        "\" class=\"ptolemy.kernel.util.StringAttribute\">\n");
    }
    else
    {
      sb.append("<property name=\"class\" value=\"" + internalClassName +
        "\" class=\"ptolemy.kernel.util.StringAttribute\">\n");
    }

    if(internalClassName.equals("ptolemy.actor.TypedCompositeActor"))
    { //add another property to tell what the moml class is
      sb.append("  <property name=\"momlClass\" value=\"" + className +
        "\" class=\"ptolemy.kernel.util.StringAttribute\"/>\n");
    }
    sb.append("  <property name=\"id\" value=\"" + classId +
      "\" class=\"ptolemy.kernel.util.StringAttribute\"/>\n");
    sb.append("</property>\n");

    //print the ports
    if(portVector.size() != 0)
    {
      for(int i = 0; i < portVector.size(); i++)
      {
        sb.append(((PortMetadata)portVector.get(i)).toString());
      }
    }

    //print the dependencies
    if(dependencyVector.size() != 0)
    {
      for(int i = 0; i < dependencyVector.size(); i++)
      {
        ClassedProperty cp = (ClassedProperty)dependencyVector.get(i);
        cp.addNameIterator(i);
        sb.append(cp.toString() + "\n");
      }
    }

    //print the semantic types
    if(semanticTypeVector.size() != 0)
    {
      for(int i = 0; i < semanticTypeVector.size(); i++)
      {
        ClassedProperty cp = (ClassedProperty)semanticTypeVector.get(i);
        cp.addNameIterator(i);
        sb.append(cp.toString() + "\n");
      }
    }

    //add general attributes
    if(attributeVector.size() != 0)
    {
      for(int i=0; i<attributeVector.size(); i++)
      {
        Attribute a = (Attribute)attributeVector.elementAt(i);
        sb.append(a.exportMoML() + "\n");
      }
    }
    
    //add relations
    if(relationVector.size() != 0)
    {
      for(int i=0; i<relationVector.size(); i++)
      {
        ComponentRelation r = (ComponentRelation)relationVector.elementAt(i);
        sb.append(r.exportMoML() + "\n");
      }
      
    }
    
    if(internalClassName.equals("org.kepler.moml.CompositeClassEntity"))
    {
      Iterator entityItt = ((CompositeEntity)actor).entityList().iterator();
      while(entityItt.hasNext())
      {
        Entity ent = (Entity)entityItt.next();
        sb.append(ent.exportMoML());
      }
    }
    
    if(links != null)
    {
      sb.append(links);
    }

    if(actor instanceof Attribute || actor instanceof PropertyAttribute)
    {
      sb.append("</property>");
    }
    else
    {
      sb.append("</entity>\n");
    }

    return sb.toString();
  }

  /**
   * parse an actor to create the actorMetadata object
   *
   *@param  actor
   */
  private void parseActor(NamedObj actor)
  {
    if(actor instanceof ComponentEntity && !(actor instanceof TypedCompositeActor))
    {
      List list = ((ComponentEntity)actor).portList();
      Iterator portIterator = list.iterator();
      while(portIterator.hasNext())
      {
        TypedIOPort p = (TypedIOPort)portIterator.next();
        PortMetadata pm = new PortMetadata();

        pm.name = p.getName();
        pm.type = p.getType().toString();
        pm.multiport = p.isMultiport();
        if(p.isInput() && p.isOutput())
        {
          pm.direction = "inputoutput";
        }
        else if(p.isInput())
        {
          pm.direction = "input";
        }
        else if(p.isOutput())
        {
          pm.direction = "output";
        }
        //get any other port attributes and copy them
        List portAttList = p.attributeList();
        Iterator portAttsIt = portAttList.iterator();
        while(portAttsIt.hasNext())
        {
          NamedObj att = (NamedObj)portAttsIt.next();
          pm.addAttribute(att);
        }

        portVector.addElement(pm);
      }
    }
  }

  /**
   * parse a named obj and get it's port info
   */
  private void parseNamedObj(NamedObj obj)
  {
    if(obj instanceof ComponentEntity)
    {
      List list = null;
      try
      {
        list = ((ComponentEntity)obj).attributeList(
          Class.forName("org.kepler.moml.PortAttribute"));
      }
      catch(ClassNotFoundException cnfe) {}

      Iterator portIterator = list.iterator();

      while(portIterator.hasNext())
      {
        PortAttribute pa = (PortAttribute)portIterator.next();
        PortMetadata pm = new PortMetadata();

        pm.name = pa.getName();
        pm.type = ((StringAttribute)pa.getAttribute("dataType")).getExpression();
        pm.multiport = new Boolean(
          ((StringAttribute)pa.getAttribute("isMultiport")).getExpression()
            ).booleanValue();
        pm.direction = ((StringAttribute)pa.getAttribute("direction")).getExpression();

        List attList = pa.attributeList();
        Iterator listIt = attList.iterator();
        while(listIt.hasNext())
        {
          NamedObj att = (NamedObj)listIt.next();
          if(!att.getName().equals("dataType") &&
             !att.getName().equals("isMultiport") &&
             !att.getName().equals("direction"))
          {
            pm.addAttribute(att);
          }
        }

        portVector.addElement(pm);
      }
    }
  }

  /**
   * get the actor class and instantiate it to a NamedObj and return it
   */
  public NamedObj getActorClass(String className,
    String actorName, NamedObj actorMetadataMoml)
  {
    //try to get the actor class and instantiate it
    try
    {
      TypedAtomicActor actor;
      Class actorClass = Class.forName(className);
      Object[] args = {new CompositeEntity(), actorName};
      actor = (TypedAtomicActor)createInstance(actorClass, args);
      return actor;
    }
    catch(Exception e)
    {
      try
      { //look for a moml class and try to instantiate it
        if(internalClassName.equals("org.kepler.moml.CompositeClassEntity"))
        {
          //This is a total hack, but I cannot get the CompositeActors to 
          //clone correctly, so this try statement basically clones them 
          //manually.  It works so i'm leaving it for now.
          try
          {
            StringBuffer sb = new StringBuffer();
            sb.append("<class name=\"" + actorName + "\" extends=\"" + className + "\">");
            
            //get attributes and clone them over to the new obj
            Iterator attItt = ((CompositeClassEntity)actorMetadataMoml).attributeList().iterator();
            while(attItt.hasNext())
            {
              Attribute a = (Attribute)attItt.next();
              sb.append(a.exportMoML());
            }
            
            //get entities and clone them over to the new obj
            Iterator entItt = ((CompositeClassEntity)actorMetadataMoml).entityList().iterator();
            while(entItt.hasNext())
            {
              ComponentEntity ent = (ComponentEntity)entItt.next();
              sb.append(ent.exportMoML());
            }
            
            //get ports and clone them over to the new obj
            Iterator portItt = ((CompositeClassEntity)actorMetadataMoml).portList().iterator();
            while(portItt.hasNext())
            {
              Port p = (Port)portItt.next();
              sb.append(p.exportMoML());
            }
            
            //get relations and clone them over to the new obj
            Iterator relationItt = ((TypedCompositeActor)actorMetadataMoml).relationList().iterator();
            while(relationItt.hasNext())
            {
              ComponentRelation r = (ComponentRelation)relationItt.next();
              sb.append(r.exportMoML());
            }
            
            sb.append(links);
            sb.append("</class>");
            Workspace w = actorMetadataMoml.workspace();
            MoMLParser parser = new MoMLParser(w);
            parser.reset();
            NamedObj obj = parser.parse(sb.toString());
            return obj;
          }
          catch(Exception exc)
          {
            System.out.println("error creating compositeClassEntity: " + exc.getMessage());
            exc.printStackTrace();
            throw exc;
          }
        }
        else
        {
          ComponentEntity actor = parseMoMLFile(className);
          return (TypedCompositeActor)actor;
        }
      }
      catch(Exception ee)
      {
        try
        { //this handles directors which are not entities, but attributes
          Director director;
          Class directorClass = Class.forName(className);
          Object[] args = {new CompositeEntity(), actorName};
          director = (Director)createInstance(directorClass, args);
          return director;
        }
        catch(Exception eee)
        {
          try
          {
            Attribute att;
            Class attClass = Class.forName(className);
            Object[] args = {new CompositeEntity(), actorName};
            att = (Attribute)createInstance(attClass, args);
            return att;
          }
          catch(Exception eeee)
          {
            System.out.println("The class name you entered was not found in the " +
                "classpath.  Note that the class you are trying to enter must be " +
                "in the classpath from which you launched this program: " +
                 eeee.getMessage());
            return null;
          }
        }
      }
    }
  }
  
  /**
   * add all of the attributes with the exception of the kepler attributes
   * (which should already be in the metadata) to the AM object.
   * @param obj the NamedObj to get the attributes from.
   */
  public void addAllAttributes(NamedObj obj)
  {
    if(obj != null && obj.attributeList() != null)
    {
      Iterator i = obj.attributeList().iterator();
      while(i.hasNext())
      {
        Attribute a = (Attribute)i.next();
        String name = a.getName();
        if(!name.equals("entityId") &&
           !name.equals("documentation") &&
           !name.equals("class") &&
           name.indexOf("dependency") == -1)
         {
           addAttribute(a);
         }
      }
    }
  }
  
  /**
   * add all of the relations
   * @param obj the NamedObj to get the attributes from.
   */
  public void addAllRelations(NamedObj obj)
  {
    if(obj != null &&
       obj instanceof TypedCompositeActor &&
       ((TypedCompositeActor)obj).relationList() != null)
    {
      Iterator i = ((TypedCompositeActor)obj).relationList().iterator();
      while(i.hasNext())
      {
        ComponentRelation r = (ComponentRelation)i.next();
        addRelation(r);
      }
    }
  }

  /**
   * search the classpath for a specific class and return the file.  In
   * the tradition of ptolemy, this will also search for moml files with
   * a class definition. this is required for composite actors.
   * @param className the name of the class to search for
   * @param workDir the directory where temp files can be created
   */
  protected File searchClasspath(String className)
    throws FileNotFoundException
  {
    //separate the class name from the package path
    String actualClassName = className.substring(
      className.lastIndexOf(".") + 1, className.length());
    String packagePath = className.substring(0, className.lastIndexOf("."));
    //split up the package name into it's parts
    String[] packagePathStruct = packagePath.split("\\.");

    //get the classpath so we can search for the class files
    String classpath = System.getProperty("java.class.path");
    String sep = System.getProperty("path.separator");
    StringTokenizer st = new StringTokenizer(classpath, sep);
    while(st.hasMoreTokens())
    {
      String path = st.nextToken();
      File pathDir = new File(path);
      if(pathDir.isDirectory())
      { //search the directory for the file
         if(matchDirectoryPath(pathDir, packagePathStruct, 0))
         {
           //now we found a candidate...see if the class is in there
           File classDir = new File(pathDir.getAbsolutePath() + File.separator +
             packagePath.replace('.', File.separator.toCharArray()[0]));
           File[] classDirFiles = classDir.listFiles();
           for(int i=0; i<classDirFiles.length; i++)
           {
             String dirFileName = classDirFiles[i].getName();
             if(dirFileName.indexOf(".") != -1)
             {
               String extension = dirFileName.substring(
                 dirFileName.lastIndexOf("."), dirFileName.length());
               String prefix = dirFileName.substring(0, dirFileName.lastIndexOf("."));
               if(actualClassName.equals(prefix) &&
                   (extension.equals(".class") || extension.equals(".xml")))
               { //search for xml or class files
                 return classDirFiles[i];
               }
             }
           }
         }
      }
      else if(pathDir.isFile())
      { //search a jar file for the file. if it's not a jar file, ignore it
        try
        {
          String entryName = className.replace('.', '/') + ".class";
          JarFile jarFile = new JarFile(pathDir);
          //this looks for a class file
          JarEntry entry = jarFile.getJarEntry(entryName);
          if(entry != null)
          {
            //get the class file from the jar and return it
            return pathDir;
          }
          else
          { //look for the xml file instead
            entryName = className.replace('.', '/') + ".xml";
            entry = jarFile.getJarEntry(entryName);
            if(entry != null)
            {
              return pathDir;
            }
          }
        }
        catch(Exception e)
        { //keep going if this isn't a jar file
          continue;
        }
      }
    }
    throw new FileNotFoundException("Cannot find the specified class " +
       "file in the classpath.");
  }

  /**
   * locates a directory based on an array of paths to look for.  for instance,
   * if you are searching for org/kepler/ksw searchForMe would be
   * {org, kepler, ksw}.  dir is where the search starts.  Index should start out
   * as 0.
   */
  private boolean matchDirectoryPath(File dir, String[] searchForMe, int index)
  {
    File[] dirContents = dir.listFiles();
    for(int i=0; i<dirContents.length; i++)
    {
      if(dirContents[i].getName().equals(searchForMe[index]))
      {
        if(index < searchForMe.length - 1)
        {
          if(index < searchForMe.length)
          {
            return matchDirectoryPath(dirContents[i], searchForMe, ++index);
          }
          else
          {
            return false;
          }
        }
        else
        {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * try to locate and parse a moml file as a class
   */
  protected ComponentEntity parseMoMLFile(String className)
    throws Exception
  {

    //first we need to find the file and read it
    File classFile = searchClasspath(className);
    StringWriter sw = new StringWriter();
    InputStream xmlStream;
    if(classFile.getName().endsWith(".jar"))
    {
      JarFile jarFile = new JarFile(classFile);
      ZipEntry entry = jarFile.getEntry(className.replace('.', '/') + ".xml");
      xmlStream = jarFile.getInputStream(entry);
    }
    else
    {
      xmlStream = new FileInputStream(classFile);
    }

    byte[] b = new byte[1024];
    int numread = xmlStream.read(b, 0, 1024);
    while(numread != -1)
    {
      String s = new String(b, 0, numread);
      sw.write(s);
      numread = xmlStream.read(b, 0, 1024);
    }
    sw.flush();
    //get the moml document
    String xmlDoc = sw.toString();
    sw.close();

    //use the moml parser to parse the doc
    MoMLParser parser = new MoMLParser();
    parser.reset();
    NamedObj obj = parser.parse(xmlDoc);
    return (ComponentEntity)obj;
  }

  /**
   * createInstance.  creates an instance of a class object.  taken from
   * ptolemy's MomlParser class.  modified for this application.
   *@param  newClass
   *@param  arguments
   *@return TypedAtomicActor
   *@exception  Exception
   */
  protected NamedObj createInstance(Class newClass, Object[] arguments)
    throws Exception
  {
    Constructor[] constructors = newClass.getConstructors();
    for(int i = 0; i < constructors.length; i++)
    {
      Constructor constructor = constructors[i];
      Class[] parameterTypes = constructor.getParameterTypes();

      for(int j=0; j<parameterTypes.length; j++)
      {
        Class c = parameterTypes[j];
      }

      if(parameterTypes.length != arguments.length)
      {
        continue;
      }

      boolean match = true;

      for(int j = 0; j < parameterTypes.length; j++)
      {
        if(!(parameterTypes[j].isInstance(arguments[j])))
        {
          match = false;
          break;
        }
      }

      if(match)
      {
        NamedObj newEntity = (NamedObj)constructor.newInstance(arguments);
        return newEntity;
      }
    }

    // If we get here, then there is no matching constructor.
    // Generate a StringBuffer containing what we were looking for.
    StringBuffer argumentBuffer = new StringBuffer();

    for(int i = 0; i < arguments.length; i++)
    {
      argumentBuffer.append(arguments[i].getClass() + " = \""
           + arguments[i].toString() + "\"");

      if(i < (arguments.length - 1))
      {
        argumentBuffer.append(", ");
      }
    }

    throw new Exception("Cannot find a suitable constructor ("
         + arguments.length + " args) (" + argumentBuffer + ") for '"
         + newClass.getName() + "'");
  }

  /**
   * metadata object for a classed property
   */
  private class ClassedProperty implements Serializable
  {
    public String name = "";
    public String value = "";
    public String className = "";
    public Integer iterator = null;

    /**
     * constructor
     */
    public ClassedProperty(String name, String value, String className)
    {
      this.name = name;
      this.value = value;
      this.className = className;
    }

    /**
     * adds an integer to the end of the name of the property if there
     * is more than one of these properties.  for instance:
     * semanticType0, semanticType1, etc.
     * @param int i the number to add to the name.
     */
    public void addNameIterator(int i)
    {
      iterator = new Integer(i);
    }

    /**
     * create the xml rep of the ClassedProperty
     */
    public String toString()
    {
      if(iterator == null)
      {
        return "<property name=\"" + name + "\" value=\"" + value +
          "\" class=\"" + className + "\"/>";
      }
      else
      {
        int itt = iterator.intValue();
        return "<property name=\"" + name + itt + "\" value=\"" + value +
          "\" class=\"" + className + "\"/>";
      }
    }
  }

  /**
   * metadata object that represents a port
   */
  private class PortMetadata implements Serializable
  {
    public String name = "";
    public String direction = "";
    public String type = "";
    public boolean multiport = false;
    private Vector attributes = new Vector();

    public void addAttribute(NamedObj att)
    {
      attributes.addElement(att);
    }

    /**
     * return the moml xml rep of the port
     *
     *@return
     */
    public String toString()
    {
      StringBuffer sb = new StringBuffer();

      sb.append("<property name=\"kepler:" + name + "\" class=\"org.kepler.moml.PortAttribute\">\n");
      sb.append("  <property name=\"direction\" value=\"" + direction +
        "\" class=\"ptolemy.kernel.util.StringAttribute\"/>\n");
      sb.append("  <property name=\"dataType\" value=\"" + type +
        "\" class=\"ptolemy.kernel.util.StringAttribute\"/>\n");
      sb.append("  <property name=\"isMultiport\" value=\"" + multiport +
        "\" class=\"ptolemy.kernel.util.StringAttribute\"/>\n");
      for(int i=0; i<attributes.size(); i++)
      {
        sb.append("  " + ((NamedObj)attributes.elementAt(i)).exportMoML());
      }
      sb.append("</property>\n");
      /*sb.append("<port name=\"" + name + "\" class=\"ptolemy.actor.IOPort\">\n");
      sb.append("  <property name=\"direction\" value=\"" + direction +
        "\" class=\"ptolemy.kernel.util.StringAttribute\"/>\n");
      sb.append("  <property name=\"dataType\" value=\"" + type +
        "\" class=\"ptolemy.kernel.util.StringAttribute\"/>\n");
      sb.append("  <property name=\"isMultiport\" value=\"" + multiport +
        "\" class=\"ptolemy.kernel.util.StringAttribute\"/>\n");
      sb.append("</port>\n");*/
      return sb.toString();
    }
  }
}