/*
 *  Copyright (c) 2003-2004 The Regents of the University of California.
 *  All rights reserved.
 *  Permission is hereby granted, without written agreement and without
 *  license or royalty fees, to use, copy, modify, and distribute this
 *  software and its documentation for any purpose, provided that the above
 *  copyright notice and the following two paragraphs appear in all copies
 *  of this software.
 *  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 *  FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 *  ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 *  THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 *  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 *  PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 *  PT_COPYRIGHT_VERSION_2
 *  COPYRIGHTENDKEY
 */
package org.kepler.objectmanager;

import org.kepler.objectmanager.lsid.LSIDResolver;

import java.util.Vector;
import java.io.InputStream;
import java.io.FileOutputStream;
import java.io.File;

/**
 * This class resolves IDs to various forms.  It figures out what kind
 * of id is being resolved and loads the correct resolver.
 *
 *@author     Chad Berkley
 *@created    April 07, 2005
 */
public class IDResolver
{
  /**singleton instance**/
  private static IDResolver resolverSingleton;
  /**list of possible resolvers**/
  private static Vector resolverList;

  public IDResolver()
  {
    resolverList = new Vector();
    resolverList.addElement(LSIDResolver.instanceOf());
  }

  /**
   * return a singleton instance of the resolver
   */
  public IDResolver instanceOf()
  {
    if(resolverSingleton == null)
    {
      resolverSingleton = new IDResolver();
    }
    return resolverSingleton;
  }

  /**
   * resolve the id to a class and load the class if possible.  return the
   * name of the class that the id resolved to
   * @param id
   * @return the name of the class that was resolved
   */
  public static String resolveIDToClass(String id)
    throws IDNotFoundException, InvalidMetadataException
  {
    for(int i=0; i<resolverList.size(); i++)
    {
      IDResolverInterface resolver =
        (IDResolverInterface)resolverList.elementAt(i);
      if(resolver.isResolvable(id))
      {
        return loadClass(resolver, id);
      }
    }
    throw new IDNotFoundException("ID " + id + "was not found by the resolver");
  }

  /**
   * take the id of the class and the resolver that resolved it and load the
   * class that is produced from the source
   */
  private static String loadClass(IDResolverInterface resolver, String id)
    throws IDNotFoundException, InvalidMetadataException
  {
    //we need to put the class into the classpath if it isn't there already
    //before returning the name to the momlparser to actually load the class
    ActorMetadata metadata = resolver.resolveIDToActor(id);
    String className = metadata.getClassName();
    String classId = metadata.getId();
    InputStream classStream = resolver.resolveID(classId);
    //we've got the stream of the bytes for the class.  we need to give
    //it to the cache and load it.

    try
    { //see if the class is loadable
      System.out.println("checking for class " + className + " in classpath...");
      Class.forName(className);
      System.out.println("class " + className + " is in the classpath.");
    }
    catch(ClassNotFoundException cnfe)
    { //if not, write the class into the classpath
      System.out.println("class " + className + " is NOT in the classpath.");
      String fileName = className.substring(className.lastIndexOf(".") + 1,
        className.length());
      fileName += ".class";
      String dir = parseClassNameToDir(className);
      String basedir = System.getProperty("KEPLER");
      dir = basedir + "/build/classes/" + dir;
      try
      {
        File classFile = new File(dir + "/" + fileName);
        File classDir = new File(dir);
        classDir.mkdirs();
        classFile.createNewFile();
        FileOutputStream fw = new FileOutputStream(classFile);

        byte b[] = new byte[1024];
        int numread = classStream.read(b, 0, 1024);
        while(numread != -1)
        { //write out the class into the classpath
          fw.write(b, 0, numread);
          numread = classStream.read(b, 0, 1024);
        }
      }
      catch(Exception e)
      {
        throw new IDNotFoundException("Error writing class to classpath: " +
          e.getMessage());
      }
    }

    return className;
  }

  /**
   * resolve a class name to a directory name
   */
  private static String parseClassNameToDir(String className)
  {
    className = className.substring(0, className.lastIndexOf("."));
    return className.replaceAll("\\.", "/");
  }
}
