/**
 *   '$RCSfile: LSIDGenerator.java,v $'
 *
 *   '$Author: berkley $'
 *   '$Date: 2006/03/16 23:15:23 $'
 *   '$Revision: 1.11 $'
 *
 *  For Details: http://kepler-project.org
 *  Copyright (c) 2003-2004 The Regents of the University of California.
 *  All rights reserved.
 *  Permission is hereby granted, without written agreement and without
 *  license or royalty fees, to use, copy, modify, and distribute this
 *  software and its documentation for any purpose, provided that the above
 *  copyright notice and the following two paragraphs appear in all copies
 *  of this software.
 *  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 *  FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 *  ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 *  THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 *  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 *  PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 *  PT_COPYRIGHT_VERSION_2
 *  COPYRIGHTENDKEY
 */
package org.kepler.objectmanager.lsid;

import java.io.*;
import java.util.*;
import java.util.zip.CRC32;

import com.ibm.lsid.LSIDException;

/**
 * This class maintains a list of LSIDs in the system and can generate new
 * LSIDs that are unique to the local system.  These ids are not garauteed
 * to be unique to any external system.
 *
 *@created    June 20, 2005
 */

public class LSIDGenerator
{
  //singleton instance
  private static LSIDGenerator generator = null;
  //store all ids here
  private LSIDTree idStore;

  /**
   * builds a new instance of the LSIDGenerator
   */
  public LSIDGenerator()
  {
    idStore = LSIDTree.getInstance();
  }

  /**
   * returns a singleton instance of this class
   */
  public static synchronized LSIDGenerator getInstance()
  {
    if(generator == null)
    {
      generator = new LSIDGenerator();
    }
    return generator;
  }
  
  /**
   * returns a new KeplerLSID based ont he lsid passed in.  If the passed
   * lsid is valid, this will just return an lsid with the next revision
   * number.
   * @param lsid the lsid to create a new lsid from
   */
  public KeplerLSID getNewLSID(KeplerLSID lsid)
    throws LSIDException
  {
    KeplerLSID newLSID = idStore.getNextRevision(lsid);
    return newLSID;
  }

  /**
   * returns a new lsid with the 'kepler-project.org' authority.  These id's will be
   * of the form: urn:lsid:kepler-project.org:<namespace>:<object>:<revision> where
   * object will be a unique (serial) number within the namespace and revision
   * will always be 1
   *
   * @param namespace
   */
  public KeplerLSID getNewLSID(String namespace)
    throws LSIDException
  {
    return getNewLSID("kepler-project.org", namespace);
  }

  /**
   * Create a 'localhost' lsid using the given string as a seed.  The string is converted
   * deterministically into another string which can be used as part of a valid filename.
   * Currently, the magicstring's CRC32 is computed and used as the namespace of the generated
   * LSID.  This method always returns the same LSID for a given input and different inputs
   * should return different LSIDs.
   *
   * @param magicstring
   * @return valid LSID
   * @throws LSIDException
   */
  public KeplerLSID convertToLSID(String magicstring) throws LSIDException
  {
	  CRC32 c = new CRC32();
	  c.update( magicstring.getBytes() );
	  String namespace =  Long.toHexString( c.getValue() );
	  return new KeplerLSID("localhost",namespace,0,0);
  }
  
  /**
   * return a new lsid with the given authority and namespace.  these lsids
   * are not garanteed to be unique within the authority.  note that this
   * could create invalid lsids so be careful.  this method will form a
   * new lsid of the form: urn:lsid:<authority>:<namespace>:<object>:<revision>
   * where authority and namespace are given and object will be the
   * next (serial) integer known to the system to be unique.  revision will
   * always be 1.
   *
   * @param authority
   * @param namespace
   */
  public KeplerLSID getNewLSID(String authority, String namespace)
    throws LSIDException
  {
    KeplerLSID lsid = new KeplerLSID(authority, namespace, 1, 1);
    lsid = idStore.getNextObject(lsid);
    return lsid;
  }

  /**
   * this method registers an lsid with the system so that a duplicate
   * will not be created
   * @param lsid the id to register
   * @throws LSIDException if the lsid is already in use
   */
  public void registerLSID(KeplerLSID lsid)
    throws LSIDException
  {
    try
    {
      idStore.addLSID(lsid);
    }
    catch(LSIDTreeException lsidte)
    {
      throw new LSIDException("The LSID " + lsid.toString() + " is already " +
        "in use.");
    }
    catch(IOException ioe)
    {
      System.out.println("Error adding lsid: " + ioe.getMessage());
      ioe.printStackTrace();
    }
  }
  
  /**
   * clear the cache of lsids used with this generator.
   */
  public void clearCache()
  {
    idStore.clearTree();
  }

}
