/**
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.scia;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;

import org.jdom.input.SAXBuilder;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;

import java.util.Iterator;
import java.util.List;


/**
 * The parser for EML file. The EMLParser is to create an XML schema file for
 * representing the structure of the data (e.g., a text file) described by
 * an  EML file.  It is assumed that 1) The input EML document is contains a
 * single dataset/dataTable/attributeList path, and 2) The output XML Schema
 * file follows the assumption in
 * @see XMLSchemaParser, e.g., schema tree root
 * element is named as schemaName + Type.
 *
 *@author Guilian Wang
 */
public class EMLParser {
    public EMLParser() {
    }

    /**
     * Extract the structural information of the data described by the input
     * EML file and save it in an XML Schema file, but not conforming to the
     * standard, so apache XML schema API cannot be used to parse it.
     *
     * @para emlFile The name of the EML file that describes the data.
     * @return The name of the extracted XML Schema file.
     */
    public String getSchema(String emlFile) {
        SAXBuilder builder = new SAXBuilder();

        try {
            InputStream is = new FileInputStream(emlFile);

            //InputStream is = this.getClass().getResourceAsStream(emlFile);
            Document doc = builder.build(is);
            Element rootElem = doc.getRootElement();
            Element dataSetElem = null;
            Element dataTableElem = null;
            Element attrListElem = null;
            String name;
            String attrName;
            String attrDesc;
            String attrType;

            if (rootElem != null) {
                name = rootElem.getName();

                if (SCIA.debug_on) {
                    System.err.println("name = " + name);
                }

                if (name.equals("eml")) {
                    dataSetElem = rootElem.getChild("dataset");
                } else {
                    if (SCIA.debug_on) {
                        System.err.println("Root is not eml");
                    }
                }

                if (dataSetElem != null) {
                    dataTableElem = dataSetElem.getChild("dataTable");
                } else {
                    if (SCIA.debug_on) {
                        System.err.println(
                            "eml does not have dataset sub-element");
                    }
                }

                if (dataTableElem != null) {
                    attrListElem = dataTableElem.getChild("attributeList");
                } else {
                    if (SCIA.debug_on) {
                        System.err.println("dataset does not have dataTable " +
                            "sub-element");
                    }
                }

                if (attrListElem != null) {
                    // create an XML schema file
                    int i = emlFile.lastIndexOf(".");
                    int j = emlFile.lastIndexOf("/");
                    String filePath = emlFile.substring(0, j + 1) + "schema-" +
                        emlFile.substring(j + 1, i) + ".xml";
                    String fileName = "schema-" + emlFile.substring(j + 1, i) +
                        ".xml";

                    PrintWriter pw = new PrintWriter(new FileOutputStream(
                                filePath), true);
                    pw.println("<?xml version=\"1.0\" ?>");
                    pw.println("<schema " +
                        "xmlns=\"http://www.w3.org/2001/XMLSchema\" " +
                        "name=\"Table\">");
                    pw.println("<element name=\"Table\" type=\"TableType\" />");

                    pw.println("<complexType name=\"TableType\">");
                    pw.println("<sequence>");
                    pw.println("<element name=\"Row\" type=\"RowType\" />");
                    pw.println("</sequence>");
                    pw.println("</complexType>");
                    pw.println("<elementType name=\"RowType\">");

                    List attrs = attrListElem.getChildren();
                    Iterator it = attrs.iterator();

                    if (SCIA.debug_on) {
                        System.err.println("===== attribute name & type ======");
                    }

                    while (it.hasNext()) {
                        Element el = (Element) it.next();

                        if (el.getName().equals("attribute")) {
                            attrName = el.getChildTextTrim("attributeName");
                            attrDesc = el.getChildTextTrim(
                                    "attributeDefinition");
                            attrType = el.getChildTextTrim("storageType");

                            if (SCIA.debug_on) {
                                System.err.println("     " + attrName +
                                    "     " + attrType + "     " + attrDesc);
                            }

                            pw.println("<element name=\"" + attrName + "\"" +
                                " type=\"" + attrType + "\"/>");
                        }
                    }

                    pw.println("</elementType>");
                    pw.println("</schema>");
                    pw.flush();

                    return fileName;
                } else {
                    if (SCIA.debug_on) {
                        System.err.println(
                            "dataTable does not have attributeList" +
                            " sub-element");
                    }

                    return null;
                }
            } else {
                if (SCIA.debug_on) {
                    System.err.println("Root Elem is null");
                }

                return null;
            }
        } catch (JDOMException e) { // indicates a well-formedness error 

            if (SCIA.debug_on) {
                System.err.println("JDOMException");
            }

            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;
    }

    /**
     * Extract the structural information of the data described by the input
     * EML file and save it in an XML Schema file which follows xsd standard,
     * so apache XML schema API can be used to parse it.
     *
     * @para emlFile The name of the EML file that describes the data.
     * @return The name of the extracted XML Schema file.
     */
    public String getXSD(String emlFile) {
        SAXBuilder builder = new SAXBuilder();

        try {
            InputStream is = new FileInputStream(emlFile);

            //InputStream is = this.getClass().getResourceAsStream(emlFile);
            Document doc = builder.build(is);
            Element rootElem = doc.getRootElement();
            Element dataSetElem = null;
            Element dataTableElem = null;
            Element attrListElem = null;
            String name;
            String attrName;
            String attrDesc;
            String attrType;

            if (rootElem != null) {
                name = rootElem.getName();

                if (SCIA.debug_on) {
                    System.err.println("name = " + name);
                }

                if (name.equals("eml")) {
                    dataSetElem = rootElem.getChild("dataset");
                } else {
                    if (SCIA.debug_on) {
                        System.err.println("Root is not eml");
                    }
                }

                if (dataSetElem != null) {
                    dataTableElem = dataSetElem.getChild("dataTable");
                } else {
                    if (SCIA.debug_on) {
                        System.err.println(
                            "eml does not have dataset sub-element");
                    }
                }

                if (dataTableElem != null) {
                    attrListElem = dataTableElem.getChild("attributeList");
                } else {
                    if (SCIA.debug_on) {
                        System.err.println("dataset does not have dataTable " +
                            "sub-element");
                    }
                }

                if (attrListElem != null) {
                    // create an XML schema file
                    int i = emlFile.lastIndexOf(".");
                    int j = emlFile.lastIndexOf("\\");

                    if (j == -1) {
                        j = emlFile.lastIndexOf("/");
                    }

                    if (SCIA.debug_on) {
                        System.err.println("emlFile: " + emlFile + " index i " +
                            i + " index j " + j);
                    }

                    String filePath = emlFile.substring(0, j + 1) + "schema-" +
                        emlFile.substring(j + 1, i) + ".xsd";
                    String fileName = "schema-" + emlFile.substring(j + 1, i) +
                        ".xsd";

                    if (SCIA.debug_on) {
                        System.err.println("Filename: " + fileName);
                    }

                    PrintWriter pw = new PrintWriter(new FileOutputStream(
                                filePath), true);
                    pw.println("<?xml version=\"1.0\" ?>");
                    pw.println("<xs:schema " +
                        "xmlns:xs=\"http://www.w3.org/2001/XMLSchema\">");
                    pw.println("  <xs:element name=\"Table\">");
                    pw.println("    <xs:complexType>");
                    pw.println("      <xs:sequence>");
                    pw.println("        <xs:element name=\"Row\">");
                    pw.println("          <xs:complexType>");
                    pw.println("            <xs:all>");

                    List attrs = attrListElem.getChildren();
                    Iterator it = attrs.iterator();

                    if (SCIA.debug_on) {
                        System.err.println("===== attribute name & type ======");
                    }

                    while (it.hasNext()) {
                        Element el = (Element) it.next();

                        if (el.getName().equals("attribute")) {
                            attrName = el.getChildTextTrim("attributeName");
                            attrDesc = el.getChildTextTrim(
                                    "attributeDefinition");
                            attrType = el.getChildTextTrim("storageType");

                            if (SCIA.debug_on) {
                                System.err.println("     " + attrName +
                                    "     " + attrType + "     " + attrDesc);
                            }

                            pw.println("              <xs:element name=\"" +
                                attrName + "\"" + " type=\"" + "xs:" +
                                attrType + "\">");
                            pw.println("                <xs:annotation>");
                            pw.println("                  <xs:documentation>" +
                                attrDesc + "</xs:documentation>");
                            pw.println("                </xs:annotation>");
                            pw.println("              </xs:element>");
                        }
                    }

                    pw.println("            </xs:all>");
                    pw.println("          </xs:complexType>");
                    pw.println("        </xs:element>");
                    pw.println("      </xs:sequence>");
                    pw.println("    </xs:complexType>");
                    pw.println("  </xs:element>");
                    pw.println("</xs:schema>");
                    pw.flush();

                    return fileName;
                } else {
                    if (SCIA.debug_on) {
                        System.err.println(
                            "dataTable does not have attributeList" +
                            " sub-element");
                    }

                    return null;
                }
            } else {
                if (SCIA.debug_on) {
                    System.err.println("Root Elem is null");
                }

                return null;
            }
        } catch (JDOMException e) { // indicates a well-formedness error 

            if (SCIA.debug_on) {
                System.err.println("JDOMException");
            }

            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;
    }

    public static void main(String[] args) {
        EMLParser p = new EMLParser();
        String SCIA_WORK_DIR = System.getProperty("env.SCIA_WORK_DIR");

        // extract an non-standard xsd from the input eml file
        String schema1 = p.getSchema(SCIA_WORK_DIR +
                "/schemas/PLT-GCED-0409.1.1.xml");
        SchemaTree tr1 = new SchemaTree(SCIA_WORK_DIR + "/schemas/" + schema1,
                "XMLSchema");

        //extract an standard xsd from the input eml file
        String schema2 = p.getXSD(SCIA_WORK_DIR +
                "/schemas/PLT-GCED-0409.1.1.xml");

        SchemaTree tr2 = (new XMLSchemaImporter(SCIA_WORK_DIR + "/schemas/" +
                schema2)).tree;

        PrintWriter cout = new PrintWriter(System.out, true);
        tr2.write(cout, false);

        /*
        SchemaTree tr1 = new SchemaTree(SCIA_WORK_DIR +
                                        "/schemas/GCE-0409.xml",
                                        "XMLSchema", null);
            tr1.write(cout, false);
        */
        cout.flush();
    }
}
